﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/util/util_BitPack.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

#include <nn/sasbus/sasbus.h>

#include <nnt/nntest.h>

#include "../TargetDevices/lsm6ds3/detail/lsm6ds3.h"

namespace
{

/**
* @brief       本体6軸センサードライバ向けフルスケールレンジ(FSR)・出力データレート(ODR)・フィルタ設定です
*
* @details     本体6軸センサードライバ向けの設定値です。
*              ドライバ側の設定変更を行うと、取得される生値の感度が変わります。
*/
const nnt::sasbus::lsm6ds3::AccelerometerFsr DriverAccFsr = nnt::sasbus::lsm6ds3::AccelerometerFsr_2G;
const nnt::sasbus::lsm6ds3::AccelerometerOdr DriverAccOdr = nnt::sasbus::lsm6ds3::AccelerometerOdr_1660Hz;
const nnt::sasbus::lsm6ds3::GyroscopeFsr     DriverGyroFsr = nnt::sasbus::lsm6ds3::GyroscopeFsr_2000dps;
const nnt::sasbus::lsm6ds3::GyroscopeOdr     DriverGyroOdr = nnt::sasbus::lsm6ds3::GyroscopeOdr_1660Hz;
const bool                           IsFilterEnabled = true;

/**
* @brief       定期取得用のバッファです。
*
* @details     定期取得用のバッファです。nn::os::MemoryPageSize の整数倍である必要があります。
*/
NN_ALIGNAS(nn::os::MemoryPageSize) char PeriodicReceiveModeBuffer[nn::os::MemoryPageSize];

const int64_t SamplingIntervalMilliSeconds = 1;                 //!< サンプリング周期の値 [ms]

const uint8_t OutXLG = 0x22;            // [Ref1] 9.28 OUTX_L_G  (22h)
const uint8_t SasbusReceiveMask = 0x80; // [Ref1] 6.2.1 spi Read

//!< Turn on specified bits
NN_FORCEINLINE void SetBitOn8(uint8_t* reg, uint8_t flags) NN_NOEXCEPT
{
    auto temp = *reg;
    temp |= flags;
    *reg = temp;
}

// 単純な Receive と Send のテストのため、 Reset コマンドを投げてみる。
void ResetLsm6ds3(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    uint8_t       ctrl3CValue    = 0x00;
    const uint8_t Ctrl3COffset   = 0x12; // [Ref1] 9.14 CTRL3_C   (12h)
    const uint8_t SasbusReceiveMask = 0x80; // [Ref1] 6.2.1 Sasbus Read

    typedef nn::util::BitPack8::Field< 0, 1, bool > Ctrl3CSwReset;

    // 現在の値の取得
    auto receiveResult = nn::sasbus::Receive(&ctrl3CValue, pSession, 1, Ctrl3COffset | SasbusReceiveMask);
    EXPECT_TRUE(receiveResult.IsSuccess());

    SetBitOn8(&ctrl3CValue, Ctrl3CSwReset::Mask);
    auto sendeResult = nn::sasbus::Send(pSession, &ctrl3CValue, 1, Ctrl3COffset);
    EXPECT_TRUE(sendeResult.IsSuccess());
}

void Lsm6ds3Initialize(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    // Lsm6ds3 ドライバライブラリの初期化
    nnt::sasbus::lsm6ds3::Initialize(pSession);

    // センサーの電源を入れる
    nnt::sasbus::lsm6ds3::StartAccelerometer(pSession);
    nnt::sasbus::lsm6ds3::StartGyroscope(pSession);

    // 各種設定
    nnt::sasbus::lsm6ds3::SetAccelerometerFsr(pSession, DriverAccFsr);
    nnt::sasbus::lsm6ds3::SetAccelerometerOdr(pSession, DriverAccOdr);
    nnt::sasbus::lsm6ds3::SetGyroscopeFsr(pSession, DriverGyroFsr);
    nnt::sasbus::lsm6ds3::SetGyroscopeOdr(pSession, DriverGyroOdr);
    nnt::sasbus::lsm6ds3::SetNoiseReductionFilter(pSession, IsFilterEnabled);
}

}


TEST( TargetSpecificNX, Lsm6ds3_Basic )
{
    nn::sasbus::Session session;

    nn::sasbus::Initialize();

    nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

    // sasbus の Send / Receive の確認のため、チップにリセットコマンドを投げる。
    ResetLsm6ds3(&session);

    nn::sasbus::CloseSession(&session);

    nn::sasbus::Finalize();
}

TEST( TargetSpecificNX, MultipleLsm6ds3_Basic )
{
    nn::sasbus::Session session;

    nn::sasbus::Initialize();

    for(int i = 0; i < 100; i++)
    {
        nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

        // sasbus の Send / Receive の確認のため、チップにリセットコマンドを投げる。
        ResetLsm6ds3(&session);

        nn::sasbus::CloseSession(&session);
    }

    nn::sasbus::Finalize();
}

// PeriodicReceive の最低限のテスト
TEST(TargetSpecificNX, Lsm6ds3_PeriodicReceiveBasic)
{
    nn::sasbus::Session                              session;
    nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType   receiveData;

    nn::sasbus::Initialize();

    nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

    Lsm6ds3Initialize(&session);

    // Start
    nn::sasbus::StartPeriodicReceiveMode(&session,
                                         nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize,
                                         static_cast<uint8_t>(OutXLG) | SasbusReceiveMask,
                                         nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds),
                                         PeriodicReceiveModeBuffer,
                                         sizeof(PeriodicReceiveModeBuffer));

    nn::sasbus::GetPeriodicReceiveModeData<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>(&receiveData, &session, 1);

    // ここでは値の正当性は確認しないのでログの表示だけ。あくまで機能が動くことの確認
    NN_LOG("[sasbus Test] received data = ");
    for (int i = 0; i < nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize; i++)
    {
        NN_LOG("%x", receiveData.data[i]);
    }
    NN_LOG("\n");

    // Stop
    nn::sasbus::StopPeriodicReceiveMode(&session);

    nn::sasbus::CloseSession(&session);

    nn::sasbus::Finalize();

}

// 渡したバッファに保存されている最大の数分を一気に取得するテスト
TEST(TargetSpecificNX, Lsm6ds3_PeriodicReceiveMaxState)
{
    nn::sasbus::Session                              session;
    nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType   receiveData[nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax];

    nn::sasbus::Initialize();

    nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

    Lsm6ds3Initialize(&session);

    // Start
    nn::sasbus::StartPeriodicReceiveMode(&session,
        nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize,
        static_cast<uint8_t>(OutXLG) | SasbusReceiveMask,
        nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds),
        PeriodicReceiveModeBuffer,
        sizeof(PeriodicReceiveModeBuffer));

    // 値が安定するまで少し待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Lsm6ds3PeriodicReceiveModeDataCountMax 個分、一気に取得
    nn::sasbus::GetPeriodicReceiveModeData<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>(receiveData, &session, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax);

    // ここでは値の正当性は確認しないのでログの表示だけ。あくまで機能が動くことの確認
    for (int i = 0; i < nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax; i++)
    {
        NN_LOG("[sasbus Test] received data = ");
        for (int j = 0; j < nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize; j++)
        {
            NN_LOG("%x", receiveData[i].data[j]);
        }
        NN_LOG("\n");
    }

    // Stop
    nn::sasbus::StopPeriodicReceiveMode(&session);

    nn::sasbus::CloseSession(&session);

    nn::sasbus::Finalize();

}

// StartPeriodicReceive と Stop を繰り返すテスト
TEST(TargetSpecificNX, Lsm6ds3_PeriodicReceiveRepeat)
{
    nn::sasbus::Session                              session;
    nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType   receiveData;

    nn::sasbus::Initialize();

    nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

    Lsm6ds3Initialize(&session);
    for (int i = 0; i < 100; i++)
    {
        // Start
        nn::sasbus::StartPeriodicReceiveMode(&session,
            nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize,
            static_cast<uint8_t>(OutXLG) | SasbusReceiveMask,
            nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds),
            PeriodicReceiveModeBuffer,
            sizeof(PeriodicReceiveModeBuffer));

        // 値が取れ始めるまで待つ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds));

        nn::sasbus::GetPeriodicReceiveModeData<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>(&receiveData, &session, 1);

        // ここでは値の正当性は確認しないのでログの表示だけ。あくまで機能が動くことの確認
        NN_LOG("[sasbus Test] received data = ");
        for (int j = 0; j < nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize; j++)
        {
            NN_LOG("%x", receiveData.data[j]);
        }
        NN_LOG("\n");

        // Stop
        nn::sasbus::StopPeriodicReceiveMode(&session);
    }

    nn::sasbus::CloseSession(&session);

    nn::sasbus::Finalize();

}

// StartPeriodicReceive 中に普通の receive ができることのテスト
TEST(TargetSpecificNX, Lsm6ds3_PeriodicReceiveWithNormalReceive)
{
    nn::sasbus::Session                              session;
    nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType   receiveData;
    char                                             receiveBuffer[nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize];

    nn::sasbus::Initialize();

    nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

    Lsm6ds3Initialize(&session);

    // Start
    nn::sasbus::StartPeriodicReceiveMode(&session,
        nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize,
        static_cast<uint8_t>(OutXLG) | SasbusReceiveMask,
        nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds),
        PeriodicReceiveModeBuffer,
        sizeof(PeriodicReceiveModeBuffer));

    // 値が安定するまで少し待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    for (int i = 0; i < 3; i++)
    {
        // Periodic の Receive
        nn::sasbus::GetPeriodicReceiveModeData<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>(&receiveData, &session, 1);

        NN_LOG("[sasbus Test]        received data = ");
        for (int j = 0; j < nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize; j++)
        {
            NN_LOG("%x", receiveData.data[j]);
        }
        NN_LOG(", Sampling Number = %lld\n", receiveData.samplingNumber);

        // 通常の Receive
        auto receiveResult = nn::sasbus::Receive(receiveBuffer, &session, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize, static_cast<uint8_t>(OutXLG) | SasbusReceiveMask);
        EXPECT_TRUE(receiveResult.IsSuccess());

        // ここでは値の正当性は確認しないのでログの表示だけ。あくまで機能が動くことの確認
        NN_LOG("[sasbus Test] normal received data = ");
        for (int j = 0; j < nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize; j++)
        {
            NN_LOG("%x", receiveBuffer[j]);
        }
        NN_LOG("\n");
    }

    // Stop
    nn::sasbus::StopPeriodicReceiveMode(&session);

    nn::sasbus::CloseSession(&session);

    nn::sasbus::Finalize();

}

// 通常の Receive を wait 無しで高速に行ってみるテスト
TEST(TargetSpecificNX, Lsm6ds3_Receive)
{
    nn::sasbus::Session                              session;
    char                                             receiveBuffer[nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize];

    nn::sasbus::Initialize();

    nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

    Lsm6ds3Initialize(&session);

    // 値が安定するまで少し待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    for (int i = 0; i < 100; i++)
    {
        // 通常の Receive
        auto receiveResult = nn::sasbus::Receive(receiveBuffer, &session, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize, static_cast<uint8_t>(OutXLG) | SasbusReceiveMask);
        EXPECT_TRUE(receiveResult.IsSuccess());

        // ここでは値の正当性は確認しないのでログの表示だけ。あくまで機能が動くことの確認
        NN_LOG("[sasbus Test] normal received data = ");
        for (int j = 0; j < nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize; j++)
        {
            NN_LOG("%x", receiveBuffer[j]);
        }
        NN_LOG("\n");
    }

    nn::sasbus::CloseSession(&session);

    nn::sasbus::Finalize();

}
