﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <tuple>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>

#include <nn/os.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>

#include "fileutils.h"

using namespace std;

namespace helperutils
{

int64_t ReadLine(nn::fs::FileHandle fh, size_t offset, char *buf, size_t maxSize)
{
    int64_t                     ret;
    nn::Result                  result;
    size_t                      bytesRead;
    bool                        gotEof = false;
    bool                        gotEol = false;

    for (ret = 0; (ret < maxSize) && !gotEol; /*  advance in loop  */) {
        result = nn::fs::ReadFile(&bytesRead, fh, offset + ret, buf + ret, 1);
        if (!result.IsSuccess())
        {
            if (nn::fs::ResultOutOfRange::Includes(result))
            {
                gotEof = true;
            }
            break;
        }

        if (bytesRead == 0)
        {
            //  Exact EOF
            gotEof = true;
            break;
        }

        ret += (int64_t)bytesRead;
        if (buf[ret - 1] == '\n')
        {
            gotEol = true;
        }
    }

    if ((ret == 0) && !gotEof)
    {
        ret = -1;
    }

    return ret;
}


int64_t CopyFile(char *dstName, char *srcName)
{
    int64_t                     ret;
    nn::fs::FileHandle          dst;
    nn::fs::FileHandle          src;
    nn::Result                  result;
    size_t                      bytesRead = 0;
    int64_t                     curOffset = 0;
    uint8_t                     readBuf[READ_BLOCK_SIZE];

    //  Create the destination file, we're going to copy in all contents
    memset(&dst, 0, sizeof(dst));
    memset(&src, 0, sizeof(src));
    result = nn::fs::CreateFile(dstName, 0);
    if (result.IsFailure())
    {
        ret = -1;
        goto _errExit;
    }

    result = nn::fs::OpenFile(&dst,
                              dstName,
                              (nn::fs::OpenMode::OpenMode_Write |
                               nn::fs::OpenMode::OpenMode_AllowAppend));
    if (result.IsFailure())
    {
        ret = -1;
        goto _errExit;
    }

    //  Open the source file for read only
    result = nn::fs::OpenFile(&src,
                              srcName,
                              nn::fs::OpenMode::OpenMode_Read);
    if (result.IsFailure())
    {
        ret = -1;
        goto _errExit;
    }

    //  Loop, reading data from source and writing to output
    do
    {
        bytesRead = 0;
        result = nn::fs::ReadFile(&bytesRead, src, curOffset, readBuf, READ_BLOCK_SIZE);
        if (result.IsFailure())
        {
            break;
        }

        result = nn::fs::WriteFile(dst,
                                   curOffset,
                                   readBuf,
                                   bytesRead,
                                   nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        if (result.IsFailure())
        {
            break;
        }

        curOffset += bytesRead;
    } while (bytesRead > 0);

    if (result.IsFailure() && nn::fs::ResultOutOfRange::Includes(result))
    {
        ret = -1;
        goto _errExit;
    }

    ret = curOffset;

    //  Fall through to return the amount read and written

_errExit:
    if (dst.handle != NULL)
    {
        nn::fs::CloseFile(dst);
    }

    if (src.handle != NULL)
    {
        nn::fs::CloseFile(src);
    }

    return ret;
}


}
