﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/os.h>
#include <nnt.h>

#include "executils.h"
#include "fileutils.h"
#include "textutils.h"



using namespace std;


//  External declaration of test mains used by existing NSS test code
//  so we can call it from the test framework.
extern "C" int certutilMain(int argc, char **argv);
extern "C" int modutilMain(int argc, char **argv);

#define SIGLOHOST_OPTION                "-sighost"

static const char                       *HOST_MOUNT_PATH = "C:\\WINDOWS\\TEMP";
static const int                        TEST_NOISE_SIZE = (1024 * 100);


//  Tests to run.  ALL SHOULD BE DEFINED WHEN COMMITTED TO THE REPO
#define ENABLE_TEST_CERT_ALL_CA
//#define INSTALL_CKBI    /*  Not needed on Siglo - will automatically pick up CKBI  */

#define CERTUTIL_NAME           "certutil"

#define MOUNT_POINT             "host"
#define TEST_VERSION            "1"
#define TEST_TMP_DIR            "host:/test_tmp"
#define TEST_PW_FILE            TEST_TMP_DIR "/tests.pw"
#define TEST_NOISE_FILE         TEST_TMP_DIR "/tests.noise"
#define TEST_PW_CONTENTS        "nss"
#define TEST_CA_DIR             TEST_TMP_DIR "/CA"
#define TEST_DOM_CA             "TestCA." TEST_VERSION

namespace
{
const char                  *hostRootPath;


class CertutilCATest : public ::testing::TestWithParam<const char **>
{
    virtual void SetUp()
    {
        const char              **caArgs = GetParam();
        const char              *cadir = caArgs[1];
        nn::Result              result;
        nn::fs::DirectoryHandle d;
        nn::fs::FileHandle      pwFile;
        nn::fs::FileHandle      noiseFile;

        //  Make the temp directory, if it doesn't exist already
        result = nn::fs::CreateDirectory(TEST_TMP_DIR);
        if (result.IsFailure())
        {
            EXPECT_TRUE(nn::fs::ResultPathAlreadyExists::Includes(result));
        }

        //  Create the test password file
        result = nn::fs::CreateFile(TEST_PW_FILE, 0);
        if (result.IsFailure())
        {
             EXPECT_TRUE(nn::fs::ResultPathAlreadyExists::Includes(result));
        }

        result = nn::fs::OpenFile(&pwFile,
                                  TEST_PW_FILE,
                                  nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        EXPECT_TRUE(result.IsSuccess());

        char pwContents[] = TEST_PW_CONTENTS;
        nn::fs::WriteOption opt = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush);
        result = nn::fs::WriteFile(pwFile,
                                   0,
                                   pwContents,
                                   sizeof(pwContents),
                                   opt);
        EXPECT_TRUE(result.IsSuccess());

        nn::fs::FlushFile(pwFile);
        nn::fs::CloseFile(pwFile);

        //  Create a random "noise" file.
        //  NOTE: THIS SHOULD BE FOR TEST ONLY, NOT FOR PRODUCTION
        char *noiseData = (char *)malloc(TEST_NOISE_SIZE);
        EXPECT_TRUE(noiseData != NULL);

        result = nn::fs::CreateFile(TEST_NOISE_FILE, 0);
        if (result.IsFailure())
        {
             EXPECT_TRUE(nn::fs::ResultPathAlreadyExists::Includes(result));
        }

        result = nn::fs::OpenFile(&noiseFile,
                                  TEST_NOISE_FILE,
                                  nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        EXPECT_TRUE(result.IsSuccess());

        result = nn::fs::WriteFile(noiseFile,
                                   0,
                                   noiseData,
                                   TEST_NOISE_SIZE,
                                   opt);
        EXPECT_TRUE(result.IsSuccess());

        nn::fs::FlushFile(noiseFile);
        nn::fs::CloseFile(noiseFile);

        //  Create the CA directory, if it doesn't already exist
        result = nn::fs::OpenDirectory(&d, cadir, nn::fs::OpenDirectoryMode_Directory);
        if (result.IsFailure())
        {
            //  Assume this means the directory doesn't exist, so we need to
            //  create it.
            result = nn::fs::CreateDirectory(cadir);
            EXPECT_TRUE(result.IsSuccess());
        }
        else
        {
            nn::fs::CloseDirectory(d);
        }
    }

    virtual void TearDown()
    {
/*****@@@@@  LJS: HERE - NEED TO COPY BACK THE root.cert TO BE NICKNAME.root.cert  @@@@@*****/
    }
};

}


//  Setup standard directories and paths used for temp CAs, certs, etc.
//  See the file nss/tests/cert/cert.sh and the other scripts it includes
//  for more details.  The paths setup here mimick what is used in those
//  scripts.

#ifdef ENABLE_TEST_CERT_ALL_CA

#define SELF_SIGN_FLAG          "-x"

static const char               *CA_TEST_1[] =
{
    //  From the test script:
    //  ALL_CU_SUBJECT="CN=NSS Test CA, O=BOGUS NSS, L=Mountain View, ST=California, C=US"
    //  cert_CA $CADIR TestCA -x "CTu,CTu,CTu" ${D_CA} "1"
    "CN=NSS Test CA, O=BOGUS NSS, L=Mountain View, ST=California, C=US",
    TEST_CA_DIR,
    "TestCA",
    "-x",
    "CTu,CTu,CTu",
    TEST_DOM_CA,
    "1"
};

const char                      **ALL_CA_TESTS[] =
{
    CA_TEST_1,
};


#ifdef INSTALL_CKBI
static const char               *MODUTIL_INSTALL_CA_ARGS[] =
{
    "modutil",
    "-add",
    "Root Certs",
    "-libfile",
    "libnn_ckbi3.a",
    "-force",
    "-dbdir",         /*  MUST BE THE LAST ARGUMENT  */
    NULL
};
#endif


TEST_P(CertutilCATest, SetupCA)
{
    DECLARE_TEST_ARGS();

    //  Pull the parameters and build the command line options to pass
    const char                  **caArgs = GetParam();
    const char                  *subject = caArgs[0];
    const char                  *caDir = caArgs[1];
    const char                  *nickname = caArgs[2];
    const char                  *signer = caArgs[3];
    const char                  *trustArg = caArgs[4];
    const char                  *domain = caArgs[5];
    const char                  *certserial = caArgs[6];
    int                         ret;

    NN_UNUSED(domain);

    //  Check to see if the signer is equal to SELF_SIGN_FLAG
    if (strcmp(SELF_SIGN_FLAG, signer) == 0)
    {
        //  We're creating a self signed cert store, so create a db
        TEST_ARGS_INIT();
        ADD_TEST_ARG(CERTUTIL_NAME);

        ADD_TEST_ARG("-N");
        ADD_TEST_ARG("-d");
        ADD_TEST_ARG(caDir);
        ADD_TEST_ARG("-f");
        ADD_TEST_ARG(TEST_PW_FILE);

        EXEC_DEBUG_WAIT();
        ret = CALL_TEST_MAIN(certutilMain);
        EXPECT_EQ(0, ret);

#ifdef INSTALL_CKBI
        //  Install the built-in root certs
        TEST_ARGS_INIT();
        for (int i = 0; MODUTIL_INSTALL_CA_ARGS[i] != NULL; i++) {
            ADD_TEST_ARG(MODUTIL_INSTALL_CA_ARGS[i]);
        }

        ADD_TEST_ARG(caDir);

        EXEC_DEBUG_WAIT();
        ret = CALL_TEST_MAIN(modutilMain);
        EXPECT_EQ(0, ret);
#endif
    }

    TEST_ARGS_INIT();
    ADD_TEST_ARG(CERTUTIL_NAME);

    if (subject != NULL)
    {
        ADD_TEST_ARG("-s");
        ADD_TEST_ARG(subject);
    }

    ADD_TEST_ARG("-S");
    ADD_TEST_ARG("-n");
    ADD_TEST_ARG(nickname);
    ADD_TEST_ARG("-t");
    ADD_TEST_ARG(trustArg);
    ADD_TEST_ARG("-v");
    ADD_TEST_ARG("600");
    ADD_TEST_ARG(signer);
    ADD_TEST_ARG("-d");
    ADD_TEST_ARG(caDir);
    ADD_TEST_ARG("--keyUsage");
    ADD_TEST_ARG("certSigning,crlSigning");

    ADD_TEST_ARG("--basicConstraint");
    ADD_TEST_ARG("caCert,-1");

    ADD_TEST_ARG("--nsCertType");
    ADD_TEST_ARG("sslCA,smimeCA,objectSigningCA");

    ADD_TEST_ARG("-f");
    ADD_TEST_ARG(TEST_PW_FILE);
    ADD_TEST_ARG("-z");
    ADD_TEST_ARG(TEST_NOISE_FILE);
    ADD_TEST_ARG("-m");
    ADD_TEST_ARG(certserial);

    EXEC_DEBUG_WAIT();
    ret = CALL_TEST_MAIN(certutilMain);
    EXPECT_EQ(0, ret);
}

INSTANTIATE_TEST_CASE_P(SetupAllCAsTest,
                        CertutilCATest,
                        ::testing::ValuesIn(ALL_CA_TESTS));

#endif

extern "C" void nnMain()
{
    int                         argc = nnt::GetHostArgc();
    char                        **argv = nnt::GetHostArgv();

    //  Init the test harness, get it ready to go
    ::testing::InitGoogleTest(&argc, argv);

    //  Detect if the user has overridden the host mount location
    for (int i = 0; i < argc; i++)
    {

        if (strcmp(SIGLOHOST_OPTION, argv[i]) == 0)
        {
            //  Found the host option, grab it
            i++;
            HOST_MOUNT_PATH = argv[i];
            break;
        }
    }

    //  Mount the provided host path as the System mount location
    ::hostRootPath = nnsdkSetupHostSystemFs(MOUNT_POINT, HOST_MOUNT_PATH, 1);
    EXPECT_TRUE(NULL != ::hostRootPath);

    int result = RUN_ALL_TESTS();

    //  Testing is done, unmount the System mount location
    nnsdkShutdownHostFs();
    ::hostRootPath = NULL;

    nnt::Exit(result);
}
