﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/ssl.h>
#include <nn/ssl/ssl_Api.debug.h>
#include <nn/socket.h>

#include "ExecuterTestsBase.h"
#include "ExecuterTestsRunServerPkiImport.h"
#include "PkiResources.h"

// ------------------------------------------------------------------------------------------------
// ExecuterTestsRunServerPkiImport
// ------------------------------------------------------------------------------------------------
ExecuterTestsRunServerPkiImport::ExecuterTestsRunServerPkiImport(
    uint32_t count,
    uint32_t multiCount)
{
    m_RunCount  = (count > 0)?count:1;
    m_CertCount = multiCount;
}

ExecuterTestsRunServerPkiImport::~ExecuterTestsRunServerPkiImport()
{
}

void ExecuterTestsRunServerPkiImport::Run()
{
    nn::Result result;

    if (m_CertCount > PkiResources::ServerCertCount)
    {
        NN_LOG(" [ERROR] Specified server cert count exceeeds max (max:%d passed:%d)\n",
            PkiResources::ServerCertCount, m_CertCount);
        return;
    }

    nn::ssl::Debug::HeapTrackPoint tracker;
    StartHeapTrack(&tracker);

    for (uint32_t i = 0; i < m_RunCount; i++)
    {
        nn::ssl::Context* pSslContext[PkiResources::ServerCertCount] = {0};

        for (uint32_t j = 0; j < m_CertCount; j++)
        {
            pSslContext[j] = new nn::ssl::Context();
            if (pSslContext[j] == nullptr)
            {
                NN_LOG(" [ERROR] Faild to allocate a memory for a context.\n");
            }

            result = pSslContext[j]->Create(nn::ssl::Context::SslVersion_Auto);
            if (result.IsFailure())
            {
                break;
            }

            PkiResources pkiResources;

            nn::ssl::CertStoreId certStoreId;
            result = pSslContext[j]->ImportServerPki(
                &certStoreId,
                pkiResources.GetServerCertPointer(j),
                strlen(pkiResources.GetServerCertPointer(j)),
                nn::ssl::CertificateFormat_Pem);
            if (result.IsFailure())
            {
                NN_LOG(" [ERROR] Faild to import a server PKI.\n");
                break;
            }
            TEST_CASE_VERBOSE_LOG(" Imported a server pki (certStoreId:%llu)\n", certStoreId);
        }

        for (uint32_t j = 0; j < m_CertCount; j++)
        {
            if (pSslContext[j] != nullptr)
            {
                nn::ssl::SslContextId ctxId;
                pSslContext[j]->GetContextId(&ctxId);
                if (ctxId != 0)
                {
                    result = pSslContext[j]->Destroy();
                    if (result.IsFailure())
                    {
                        TEST_CASE_VERBOSE_LOG(" Failed to destroy a SSL context.\n");
                    }
                }

                delete pSslContext[j];
            }
        }

        NN_LOG(" count: %d (cert count: %d)\n", i + 1, m_CertCount);
        ExecuterTestsBase::SetResult(result.IsSuccess());
    }

    EndHeapTrack(&tracker);
    ExecuterTestsBase::PrintResult();
}

