﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/ssl.h>
#include <nn/ssl/ssl_Api.debug.h>
#include <nn/socket.h>

#include "ExecuterTestsBase.h"
#include "ExecuterTestsRunContextConnection.h"

// ------------------------------------------------------------------------------------------------
// ExecuterTestCaseRunContextConnection
// ------------------------------------------------------------------------------------------------
ExecuterTestsRunContextConnection::ExecuterTestsRunContextConnection(uint32_t count)
{
    m_RunCount = (count > 0)?count:1;
    m_HostName[0] = '\0';
    m_Port = 0;
}

ExecuterTestsRunContextConnection::ExecuterTestsRunContextConnection(
    const char* pInHostName,
    uint16_t port,
    uint32_t count)
{
    m_RunCount = (count > 0)?count:1;

    if (pInHostName != nullptr)
    {
        strncpy(m_HostName, pInHostName, sizeof(m_HostName));
    }
    else
    {
        m_HostName[0] = '\0';
    }

    if (port != 0)
    {
        m_Port = port;
    }
    else
    {
        m_Port = 443;
    }
}

ExecuterTestsRunContextConnection::~ExecuterTestsRunContextConnection()
{

}

void ExecuterTestsRunContextConnection::Run()
{
    nn::Result result;

    nn::ssl::Debug::HeapTrackPoint tracker;
    StartHeapTrack(&tracker);

    for (uint32_t i = 0; i <m_RunCount; i++)
    {
        nn::ssl::Context    *pSslContext;
        nn::ssl::Connection *pSslConnection;

        do
        {
            pSslContext = new nn::ssl::Context();
            pSslConnection = new nn::ssl::Connection();

            result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
            if (result.IsFailure())
            {
                break;
            }

            result = pSslConnection->Create(pSslContext);
            if (result.IsFailure())
            {
                break;
            }

            if (m_HostName[0] != '\0')
            {
                int socketFd = ExecuterTestsBase::CreateTcpSocket(true, m_Port, m_HostName, 0);
                if (socketFd < 0)
                {
                    break;
                }

                result = pSslConnection->SetSocketDescriptor(socketFd);
                if (result.IsFailure())
                {
                    break;
                }

                uint32_t hostNameLen = static_cast<uint32_t>(strlen(m_HostName));
                result = pSslConnection->SetHostName(m_HostName, hostNameLen);
                if (result.IsFailure())
                {
                    break;
                }

                result = pSslConnection->DoHandshake();
                if (result.IsFailure())
                {
                    nn::Result verifyError;
                    pSslConnection->GetVerifyCertError(&verifyError);
                    NN_LOG(" [WARN] DoHandshake failed (%d)(VerifyError:%d)\n",
                        result.GetDescription(),
                        verifyError.GetDescription());
                }
            }
        } while (NN_STATIC_CONDITION(false));

        if (pSslConnection != nullptr)
        {
            nn::ssl::SslConnectionId connId;
            pSslConnection->GetConnectionId(&connId);
            if (connId != 0)
            {
                result = pSslConnection->Destroy();
                if (result.IsFailure())
                {
                    TEST_CASE_VERBOSE_LOG(" Failed to destroy a SSL connection.\n");
                }
            }

            delete pSslConnection;
        }

        if (pSslContext != nullptr)
        {
            nn::ssl::SslContextId ctxId;
            pSslContext->GetContextId(&ctxId);
            if (ctxId != 0)
            {
                result = pSslContext->Destroy();
                if (result.IsFailure())
                {
                    TEST_CASE_VERBOSE_LOG(" Failed to destroy a SSL context.\n");
                }
            }

            delete pSslContext;
        }

        NN_LOG(" count: %d\n", i + 1);
    }

    EndHeapTrack(&tracker);

    ExecuterTestsBase::SetResult(result.IsSuccess());
    ExecuterTestsBase::PrintResult();
}

