﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/ssl.h>
#include <nn/ssl/ssl_Api.debug.h>
#include <nn/socket.h>

#include "ExecuterTestsBase.h"
#include "ExecuterTestsRunAbuse.h"
#include "PkiResources.h"
#include "WorkerUtil.h"

#define BREAK_ON_FAILURE(func,name)                                                               \
    result = func;                                                                                \
    if (result.IsFailure())                                                                       \
    {                                                                                             \
        break;                                                                                    \
    }                                                                                             \
    NN_LOG("  **** DONE - %s ****\n", name)

// ------------------------------------------------------------------------------------------------
// Parameters
// ------------------------------------------------------------------------------------------------
const char ExecuterTestsRunAbuse::TestHosts[][ExecuterTestsRunAbuse::HostNameLength] = {
    "www.facebook.com",
    "twitter.com",
    "www.yahoo.com",
    "www.linkedin.com",
    "www.microsoft.com",
    "www.nokia.com",
    "www.motorola.com",
    "www.ge.com",
    "www.siemens.com",
    "www.micron.com",
    "www.agilent.com",
    "www.qualcomm.com",
    "www.ti.com",
    "www.broadcom.com",
    "www.sandisk.com",
    "www.freescale.com",
    "www.amd.com",
    "www.apple.com",
    "www.xilinx.com",
    "www.forbes.com",
    "www.bloomberg.com",
    "www.whatsapp.com",
    "www.instagram.com",
    "www.skype.com",
    "www.snapchat.com",
    "www.amazon.com",
    "www.netflix.com",
    "www.meetup.com",
    "www.youtube.com",
    "www.ubisoft.com",
    "www.ea.com",
    "www.activision.com",
    "www.nintendo.com",
    "bethesda.net",
    "unity3d.com",
    "www.epicgames.com",
    "dena.com",
    "www.expedia.com",
    "www.hotels.com",
    "www.booking.com",
    "www.travelocity.com",
    "www.orbitz.com",
    "www.kayak.com",
    "www.tripadvisor.com",
    "www.united.com",
    "www.alaskaair.com",
    "www.aa.com",
    "www.southwest.com",
    "www.ana.co.jp",
    "www.jal.co.jp",
    "www.jetblue.com",
    "www.flyfrontier.com",
    "www.allegiantair.com",
};

const char ExecuterTestsRunAbuse::TestHostsEv[][ExecuterTestsRunAbuse::HostNameLength] = {
    "secure.comodo.com",                      // COMODO
    "addtrustexternalcaroot-ev.comodoca.com", // COMODO
    "www.cybertrust.ne.jp",                   // CyberTrust
    "evup.cybertrust.ne.jp",                  // Cybertrust
    "www.digicert.com",                       // DigiCert
    "assured-id-root-g2.digicert.com",        // DigiCert
    "global-root-g2.digicert.com",            // DigiCert
    "www.entrust.net",                        // Entrust
    "validg2.entrust.net",                    // Entrust
    "ssltest21.bbtest.net",                   // GeoTrust
    "www.geotrust.com",                       // GeoTrust
    "www.globalsign.com",                     // GlobalSign
    "2021.globalsign.com",                    // GlobalSign
    "2029.globalsign.com",                    // GlobalSign
    "www.godaddy.com",                        // Go Daddy
    "valid.gdig2.catest.godaddy.com",         // Go Daddy
    "www.starfieldtech.com",                  // Starfield
    "valid.sfig2.catest.starfieldtech.com",   // Starfield
    "www.thawte.com",                         // Thawte
    "ssltest8.bbtest.net",                    // Thawte
    "www.verisign.com",                       // VeriSign
    "ssltest6.jp.websecurity.symantec.com",   // VeriSign
    "www.amazontrust.com"
};

// Servers which policy OID is NOT in the built-in
const char ExecuterTestsRunAbuse::UserOidList[][ExecuterTestsRunAbuse::PolicyOidStringLength] = {
    "2.23.140.1.2.1", // www.amazontrust.com
};

const char ExecuterTestsRunAbuse::TestHostNintendo[] = "dauth-dd1.ndas.srv.nintendo.net";

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
class ScopedMutexLock
{
    NN_DISALLOW_COPY(ScopedMutexLock);

public:
    explicit ScopedMutexLock(nn::os::Mutex& unlockedMutex) NN_NOEXCEPT
        : m_mutex(unlockedMutex)
    {
        m_mutex.Lock();
    }

    ~ScopedMutexLock() NN_NOEXCEPT
    {
        m_mutex.Unlock();
    }

private:
    nn::os::Mutex& m_mutex;
};

// ------------------------------------------------------------------------------------------------
// Local functions
// ------------------------------------------------------------------------------------------------
namespace
{
    nn::Result CleanupContext(nn::ssl::Context* pInContext)
    {
        nn::Result result = nn::ResultSuccess();

        if (pInContext != nullptr)
        {
            nn::ssl::SslContextId ctxId;
            pInContext->GetContextId(&ctxId);
            if (ctxId != 0)
            {
                result = pInContext->Destroy();
                if (result.IsFailure())
                {
                    NN_LOG("  Failed to destroy a SSL context (desc:%d).\n",
                        result.GetDescription());
                }
            }

            delete pInContext;
        }

        return result;
    }

    nn::Result CleanupConnection(nn::ssl::Connection* pInConnection)
    {
        nn::Result result = nn::ResultSuccess();

        if (pInConnection != nullptr)
        {
            nn::ssl::SslConnectionId connId;
            pInConnection->GetConnectionId(&connId);
            if (connId != 0)
            {
                result = pInConnection->Destroy();
                if (result.IsFailure())
                {
                    NN_LOG("  Failed to destroy a SSL connection (desc:%d).\n",
                        result.GetDescription());
                }
            }
            delete pInConnection;
        }

        return result;
    }

    nn::Result ImportClientPki(nn::ssl::Context* pInContext)
    {
        nn::Result           result;
        nn::ssl::CertStoreId certStoreId;

        do
        {
            result = pInContext->ImportClientPki(
                &certStoreId,
                reinterpret_cast<const char*>(PkiResources::ClientPki1::Data),
                reinterpret_cast<const char*>(PkiResources::ClientPki1::Password),
                PkiResources::ClientPki1::DataSize,
                PkiResources::ClientPki1::PasswordLength);
            if (result.IsFailure())
            {
                NN_LOG("  Faild to import a client PKI (desc:%d).\n",
                    result.GetDescription());
                break;
            }

            NN_LOG("  Imported a client PKI.\n");
        } while (NN_STATIC_CONDITION(false));

        return result;
    }

    nn::Result ImportInternalPki(nn::ssl::Context* pInContext)
    {
        nn::Result           result;
        nn::ssl::CertStoreId certStoreId;

        do
        {
            result = pInContext->RegisterInternalPki(
                &certStoreId,
                nn::ssl::Context::InternalPki_DeviceClientCertDefault);
            if (result.IsFailure())
            {
                NN_LOG("  Faild to register the internal PKI (desc:%d).\n",
                    result.GetDescription());
                break;
            }

            NN_LOG("  Registered internal PKI.\n");
        } while (NN_STATIC_CONDITION(false));

        return result;
    }

    nn::Result ImportServerPkis(nn::ssl::Context* pInContext)
    {
        nn::Result   result;
        PkiResources pkiResources;

        do
        {
            for (int i = 0; i < PkiResources::ServerCertCount; i++)
            {
                nn::ssl::CertStoreId certStoreId;
                result = pInContext->ImportServerPki(
                    &certStoreId,
                    pkiResources.GetServerCertPointer(i),
                    strlen(pkiResources.GetServerCertPointer(i)),
                    nn::ssl::CertificateFormat_Pem);
                if (result.IsFailure())
                {
                    NN_LOG("  Faild to import a server PKI.\n",
                        result.GetDescription());
                    break;
                }
                NN_LOG("  Imported a server PKI.\n");
            }

            if (result.IsFailure())
            {
                break;
            }
        } while (NN_STATIC_CONDITION(false));

        return result;
    }

    nn::Result ImportPolicyOidStrings(ExecuterTestsRunAbuse* pInAbuseObj, nn::ssl::Context* pInContext)
    {
        nn::Result   result;

        for (int i = 0; i < ExecuterTestsRunAbuse::GetPolicyOidCount(); i++)
        {
            const char*  pCurPolicyOidString = nullptr;
            pCurPolicyOidString = ExecuterTestsRunAbuse::GetPolicyOidString(i);
            if (pCurPolicyOidString == nullptr)
            {
                break;
            }

            result = pInContext->AddPolicyOid(pCurPolicyOidString, static_cast<uint32_t>(strlen(pCurPolicyOidString) + 1));
            if (result.IsFailure())
            {
                NN_LOG("  Faild to import a policy OID string(%s)(desc:%d).\n",
                    pCurPolicyOidString, result.GetDescription());
                break;
            }

            NN_LOG("  Imported policy OID string (%s).\n", pCurPolicyOidString);
        }

        return result;
    }
} // Un-named namespace

nn::Result GetIndexPage(nn::ssl::Connection* pInConnection, const char* pInHostName)
{
    nn::Result result;
    char*      pReadBuffer        = nullptr;
    int        sentBytes          = 0;
    int        receivedBytes      = 0;
    int        totalReceivedBytes = 0;
    char       httpReqBuff[128]   = {0};

    do
    {
        sprintf(httpReqBuff, "GET / HTTP/1.1\r\nHost: %s\r\nConnection: Keep-Alive\r\n\r\n", pInHostName);

        nn::Result result = pInConnection->Write(
            httpReqBuff,
            &sentBytes,
            static_cast<uint32_t>(strlen(httpReqBuff)));

        if(result.IsFailure() || sentBytes < 0)
        {
            NN_LOG("  Failed to write data. (desc:%d)\n",
                result.GetDescription());
            break;
        }
        else
        {
            NN_LOG("  Sent HTTP request via SSL (%d bytes).\n", sentBytes);
        }

        pReadBuffer = new char[ExecuterTestsRunAbuse::ReadBufferSize];

        nn::ssl::Connection::PollEvent pollEvent;
        nn::ssl::Connection::PollEvent pollOutEvent;
        do
        {
            pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;  // reset
            pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Read; // set read event

            result = pInConnection->Poll(&pollOutEvent, &pollEvent, 5000);
            if(result.IsFailure())
            {
                if(nn::ssl::ResultIoTimeout::Includes(result))
                {
                    NN_LOG("  Finished receiving data for the timeout.\n");
                    result = nn::ResultSuccess(); // Reset error since this is not an error
                    break;
                }
                else
                {
                    NN_LOG("  Failed due to Poll error (Description:%d)\n",
                        result.GetDescription());
                    break;
                }
            }

            if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read)
               == nn::ssl::Connection::PollEvent::PollEvent_Read)
            {
                result = pInConnection->Read(
                    (pReadBuffer + totalReceivedBytes),
                    &receivedBytes,
                    ExecuterTestsRunAbuse::ReadBufferSize - totalReceivedBytes);
                if (result.IsFailure())
                {
                    if (nn::ssl::ResultIoWouldBlock::Includes(result))
                    {
                        continue;
                    }
                    else
                    {
                        NN_LOG("  Read failure (desc:%d).\n", result.GetDescription());
                        break;
                    }
                }
                else
                {
                    if (receivedBytes == 0)
                    {
                        NN_LOG("  Connection closed by the peer.\n");
                        break;
                    }

                    totalReceivedBytes += receivedBytes;

                    if (totalReceivedBytes >= ExecuterTestsRunAbuse::ReadBufferSize)
                    {
                        NN_LOG("  Read buffer size is too short.\n");
                        break;
                    }
                }
            }
        } while(NN_STATIC_CONDITION(true));
        NN_LOG("  Read %d bytes.\n", totalReceivedBytes);

    } while (NN_STATIC_CONDITION(false));

    if (pReadBuffer)
    {
        delete[] pReadBuffer;
    }

    return result;
}

nn::Result RunNormal(ExecuterTestsRunAbuse* pInAbuseObj)
{
    const int MaxConnectionCount = nn::ssl::MaxConnectionCount;

    ExecuterTestsRunAbuse::Config* pConfig = pInAbuseObj->GetConfig();
    nn::Result                     result;
    nn::ssl::Context*              pSslContext[MaxConnectionCount];
    nn::ssl::Connection*           pSslConnection[MaxConnectionCount];
    const char*                    pHostName[MaxConnectionCount];

    NN_LOG("  [RunNormal] start.\n");
    do
    {
        // Initialization
        for (int i = 0; i < pConfig->connectionCount; i++)
        {
            pSslContext[i]    = nullptr;
            pSslConnection[i] = nullptr;

            if (pConfig->isImportInternalPki)
            {
                pHostName[i] = ExecuterTestsRunAbuse::TestHostNintendo;
            }
            else
            {
                pHostName[i] = ExecuterTestsRunAbuse::GetCurrentHost();
            }
        }

        for (int i = 0; i < pConfig->connectionCount; i++)
        {
            pSslContext[i]    = new nn::ssl::Context();
            if (pSslContext[i] == nullptr)
            {
                NN_LOG("  [RunNormal] allocation failure - pSslContext.\n");
                break;
            }
            result = pSslContext[i]->Create(nn::ssl::Context::SslVersion_Auto);
            if (result.IsFailure())
            {
                NN_LOG("  [RunNormal] creation failure - pSslContext.\n");
                break;
            }

            pSslConnection[i] = new nn::ssl::Connection();
            if (pSslConnection[i] == nullptr)
            {
                NN_LOG("  [RunNormal] allocation failure - pSslConnection.\n");
                break;
            }
            result = pSslConnection[i]->Create(pSslContext[i]);
            if (result.IsFailure())
            {
                NN_LOG("  [RunNormal] creation failure - pSslConnection.\n");
                break;
            }
        }
        if (result.IsFailure())
        {
            break;
        }

        for (int i = 0; i < pConfig->connectionCount; i++)
        {
            if (pConfig->isImportClientPki)
            {
                result = ImportClientPki(pSslContext[i]);
                if (result.IsFailure())
                {
                    break;
                }
            }

            if (pConfig->isImportInternalPki)
            {
                result = ImportInternalPki(pSslContext[i]);
                if (result.IsFailure())
                {
                    break;
                }
            }

            if (pConfig->isImportServerPki)
            {
                result = ImportServerPkis(pSslContext[i]);
                if (result.IsFailure())
                {
                    break;
                }
            }
        }
        if (result.IsFailure())
        {
            break;
        }

        for (int i = 0; i < pConfig->connectionCount; i++)
        {
            result = pInAbuseObj->PerformHandshake(
                pHostName[i],
                pSslContext[i],
                pSslConnection[i],
                ExecuterTestsRunAbuse::NormalHttpsPort,
                pConfig->verifyOption,
                pConfig->sessionCacheMode,
                pConfig->ioMode,
                true,
                true,
                true,
                (pConfig->isGetServerCert)?(true):(false));
            if (result.IsFailure())
            {
                break;
            }
        }
        if (result.IsFailure())
        {
            break;
        }

        for (int i = 0; i < pConfig->connectionCount; i++)
        {
            result = GetIndexPage(pSslConnection[i], pHostName[i]);
            if (result.IsFailure())
            {
                break;
            }
        }

    } while (NN_STATIC_CONDITION(false));

    NN_LOG("  [RunNormal] Done with %s.\n", (result.IsFailure())?("failure"):("success"));


    // Finalization
    for (int i = 0; i < pConfig->connectionCount; i++)
    {
        if (pConfig->isClearSessionCache == true)
        {
            result = pSslConnection[i]->FlushSessionCache();
            if (result.IsFailure())
            {
                NN_LOG("  [RunNormal] failed to clear session cache (desc:%d)\n",
                    result.GetDescription());
            }
        }

        CleanupConnection(pSslConnection[i]);
        CleanupContext(pSslContext[i]);
    }

    return result;
} // NOLINT(impl/function_size)

nn::Result RunWithEv(ExecuterTestsRunAbuse* pInAbuseObj)
{
    ExecuterTestsRunAbuse::Config* pConfig = pInAbuseObj->GetConfig();
    nn::Result                     result;
    nn::ssl::Context*              pSslContext;
    nn::ssl::Connection*           pSslConnection;
    const char*                    pHostName;

    NN_LOG("  [RunWithEv] start.\n");
    do
    {
        for (int i = 0; i < ExecuterTestsRunAbuse::GetEvHostCount(); i++)
        {
            do
            {
                pHostName = ExecuterTestsRunAbuse::GetEvHost(i);

                pSslContext = new nn::ssl::Context();
                if (pSslContext == nullptr)
                {
                    NN_LOG("  [RunWithEv] allocation failure - pSslContext.\n");
                    break;
                }
                result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
                if (result.IsFailure())
                {
                    NN_LOG("  [RunWithEv] creation failure - pSslContext.\n");
                    break;
                }

                pSslConnection = new nn::ssl::Connection();
                if (pSslConnection == nullptr)
                {
                    NN_LOG("  [RunWithEv] allocation failure - pSslConnection.\n");
                    break;
                }
                result = pSslConnection->Create(pSslContext);
                if (result.IsFailure())
                {
                    NN_LOG("  [RunWithEv] creation failure - pSslConnection.\n");
                    break;
                }

                result = ImportPolicyOidStrings(pInAbuseObj, pSslContext);
                if (result.IsFailure())
                {
                    NN_LOG("  [RunWithEv] failed to add policy OID.\n");
                    break;
                }

                result = pInAbuseObj->PerformHandshake(
                    pHostName,
                    pSslContext,
                    pSslConnection,
                    ExecuterTestsRunAbuse::NormalHttpsPort,
                    pConfig->verifyOption,
                    pConfig->sessionCacheMode,
                    pConfig->ioMode,
                    true,
                    true,
                    true,
                    false);
                if (result.IsFailure())
                {
                    break;
                }

                result = GetIndexPage(pSslConnection, pHostName);
                if (result.IsFailure())
                {
                    break;
                }
            } while (NN_STATIC_CONDITION(false));

            NN_LOG("  [RunWithEv] Done with %s.\n", (result.IsFailure())?("failure"):("success"));

            if (pConfig->isClearSessionCache == true)
            {
                result = pSslConnection->FlushSessionCache();
                if (result.IsFailure())
                {
                    NN_LOG("  [RunWithEv] failed to clear session cache (desc:%d)\n",
                        result.GetDescription());
                }
            }

            CleanupConnection(pSslConnection);
            CleanupContext(pSslContext);

            if (result.IsFailure())
            {
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result RunJob(WorkerUtil::Func pFunc, ExecuterTestsRunAbuse* pInObj, int threadCount)
{
    nn::Result  result;
    WorkerUtil* pWorker[ExecuterTestsRunAbuse::ThreadMaxCount] = {0};

    do
    {
        for (int i = 0; i < threadCount; i++)
        {
            pWorker[i] = new WorkerUtil(pFunc,
                                        nullptr,
                                        pInObj,
                                        ExecuterTestsRunAbuse::DefaultThreadStackSize);
        }

        for (int i = 0; i < threadCount; i++)
        {
            WorkerUtil::Error err = pWorker[i]->Run(nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
            if (err != WorkerUtil::Error_Ok)
            {
                NN_LOG("  [RunJob]Failed to run the thread (desc:%p)\n", result.GetDescription());
                break;
            }
        }

        for (int i = 0; i < threadCount; i++)
        {
            pWorker[i]->WaitToBeDone();
            result = pWorker[i]->GetResult();
            delete pWorker[i];
        }

    } while (NN_STATIC_CONDITION(false));

    return result;
}

// ------------------------------------------------------------------------------------------------
// Test1
// - Performing SSL handshake without session ID
// ------------------------------------------------------------------------------------------------
nn::Result Test1Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_Blocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_None;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    config.connectionCount     = 1;
    config.isImportServerPki   = false;
    config.isImportClientPki   = false;
    config.isImportInternalPki = false;
    config.isClearSessionCache = false;
    config.isGetServerCert     = false;

    pObj->SetConfig(&config);

    return RunNormal(pObj);
}

nn::Result Test1(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test1Func, pInObj, 1 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// Test2
// - Performing SSL handshake with session ID, clears session cache when it completes
// ------------------------------------------------------------------------------------------------
nn::Result Test2Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_Blocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_SessionId;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    config.connectionCount     = 1;
    config.isImportServerPki   = false;
    config.isImportClientPki   = false;
    config.isImportInternalPki = false;
    config.isClearSessionCache = true;
    config.isGetServerCert     = false;

    pObj->SetConfig(&config);

    return RunNormal(pObj);
}

nn::Result Test2(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test2Func, pInObj, 1 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// Test3
// - Performing SSL handshake with the internal PKI, with Non-Blocking IO
// ------------------------------------------------------------------------------------------------
nn::Result Test3Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_NonBlocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_SessionId;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    config.connectionCount     = 1;
    config.isImportServerPki   = false;
    config.isImportClientPki   = false;
    config.isImportInternalPki = true;
    config.isClearSessionCache = false;
    config.isGetServerCert     = false;

    pObj->SetConfig(&config);

    return RunNormal(pObj);
}

nn::Result Test3(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test3Func, pInObj, 1 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// Test4
// - Performing SSL handshake with server cert buffer, with Non-Blocking IO
// ------------------------------------------------------------------------------------------------
nn::Result Test4Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_Blocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_SessionId;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    config.connectionCount     = 1;
    config.isImportServerPki   = false;
    config.isImportClientPki   = false;
    config.isImportInternalPki = false;
    config.isClearSessionCache = true;
    config.isGetServerCert     = true;

    pObj->SetConfig(&config);

    return RunNormal(pObj);
}

nn::Result Test4(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test4Func, pInObj, 1 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// Test5
// - Create 8 threads. On each thread:
//   - Create 1 SSL context
//   - Create 1 SSL connection
//   - Import 1 client PKI
//   - Import multiple server certificates
//   - Enable session ticket
//   - Enable all validation options
//   - Enable EV certificate validation
//   - Perform SSL handshake
//   - Write HTTP GET request
//   - Read the index page
// ------------------------------------------------------------------------------------------------
nn::Result Test5Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_Blocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_SessionTicket;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    config.connectionCount     = 1;
    config.isImportServerPki   = true;
    config.isImportClientPki   = true;
    config.isImportInternalPki = false;
    config.isClearSessionCache = false;
    config.isGetServerCert     = false;

    pObj->SetConfig(&config);

    return RunNormal(pObj);
}

nn::Result Test5(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test5Func, pInObj, 8 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// Test6
// - Run Test1 in Non-Blocking IO mode, clears session cache when completed
// ------------------------------------------------------------------------------------------------
nn::Result Test6Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_NonBlocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_SessionTicket;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All |
                                 nn::ssl::Connection::VerifyOption::VerifyOption_EvCertPartial;
    config.connectionCount     = 1;
    config.isImportServerPki   = true;
    config.isImportClientPki   = true;
    config.isImportInternalPki = false;
    config.isClearSessionCache = true;
    config.isGetServerCert     = false;

    pObj->SetConfig(&config);

    return RunNormal(pObj);
}

nn::Result Test6(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test6Func, pInObj, 8 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// Test7
// - Run Test1 in Non-Blocking IO mode, and session ID
// ------------------------------------------------------------------------------------------------
nn::Result Test7Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_NonBlocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_SessionId;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    config.connectionCount     = 1;
    config.isImportServerPki   = true;
    config.isImportClientPki   = true;
    config.isImportInternalPki = false;
    config.isClearSessionCache = false;
    config.isGetServerCert     = false;

    pObj->SetConfig(&config);

    return RunNormal(pObj);
}

nn::Result Test7(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test7Func, pInObj, 8 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// Test8
// - Run with EV certificate validation
// ------------------------------------------------------------------------------------------------
nn::Result Test8Func(void* ptr)
{
    ExecuterTestsRunAbuse*        pObj = reinterpret_cast<ExecuterTestsRunAbuse*>(ptr);
    ExecuterTestsRunAbuse::Config config;

    config.ioMode              = nn::ssl::Connection::IoMode_Blocking;
    config.sessionCacheMode    = nn::ssl::Connection::SessionCacheMode_SessionTicket;
    config.verifyOption        = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    config.connectionCount     = 0; // not used
    config.isImportServerPki   = true;
    config.isImportClientPki   = true;
    config.isImportInternalPki = false;
    config.isClearSessionCache = false;
    config.isGetServerCert     = false;

    pObj->SetConfig(&config);

    return RunWithEv(pObj);
}

nn::Result Test8(ExecuterTestsRunAbuse* pInObj)
{
    return RunJob(Test8Func, pInObj, 1 /* thread count */);
}

// ------------------------------------------------------------------------------------------------
// ExecuterTestsRunAbuse
// ------------------------------------------------------------------------------------------------
uint32_t      ExecuterTestsRunAbuse::g_TestHostIndex  = 0;
nn::os::Mutex ExecuterTestsRunAbuse::g_Mutex(false);

const char* ExecuterTestsRunAbuse::GetCurrentHost()
{
    ScopedMutexLock mutexLock(g_Mutex);

    uint32_t testUrlCount = sizeof(TestHosts) / sizeof(TestHosts[0]);

    if (g_TestHostIndex > testUrlCount - 1)
    {
        g_TestHostIndex = 0;
    }

    return TestHosts[g_TestHostIndex++];
}

void ExecuterTestsRunAbuse::SetConfig(Config* pInConfig)
{
    memcpy(&m_Config, pInConfig, sizeof(Config));
}

uint32_t ExecuterTestsRunAbuse::GetPolicyOidCount()
{
    return sizeof(UserOidList) / sizeof(UserOidList[0]);
}

const char* ExecuterTestsRunAbuse::GetPolicyOidString(uint32_t index)
{
    if (index < GetPolicyOidCount())
    {
        return UserOidList[index];
    }

    return nullptr;
}

ExecuterTestsRunAbuse::Config* ExecuterTestsRunAbuse::GetConfig()
{
    return &m_Config;
}

uint32_t ExecuterTestsRunAbuse::GetEvHostCount()
{
    return sizeof(TestHostsEv) / sizeof(TestHostsEv[0]);
}

const char* ExecuterTestsRunAbuse::GetEvHost(uint32_t index)
{
    if (index < GetEvHostCount())
    {
        return TestHostsEv[index];
    }

    return nullptr;
}

ExecuterTestsRunAbuse::ExecuterTestsRunAbuse(uint32_t count) :
    m_RunCount((count > 0)?count:1)
{
    ExecuterTestsBase::EnableVerbose();
}

ExecuterTestsRunAbuse::~ExecuterTestsRunAbuse()
{

}

nn::Result a(void* ptr)
{
    return nn::ResultSuccess();
}
nn::Result b(void* ptr)
{
    return nn::ResultSuccess();
}

void ExecuterTestsRunAbuse::Run()
{
    nn::Result                     result;
    nn::ssl::Debug::HeapTrackPoint tracker;

    StartHeapTrack(&tracker);
    do
    {
        for (int i = 0; i < m_RunCount; i++)
        {
            BREAK_ON_FAILURE(Test1(this), "Test1");
            BREAK_ON_FAILURE(Test2(this), "Test2");
            BREAK_ON_FAILURE(Test3(this), "Test3");
            BREAK_ON_FAILURE(Test4(this), "Test4");
            BREAK_ON_FAILURE(Test5(this), "Test5");
            BREAK_ON_FAILURE(Test6(this), "Test6");
            BREAK_ON_FAILURE(Test7(this), "Test7");
            BREAK_ON_FAILURE(Test8(this), "Test8");
        }
    } while (NN_STATIC_CONDITION(false));
    EndHeapTrack(&tracker);

    ExecuterTestsBase::SetResult(result.IsSuccess());
    ExecuterTestsBase::PrintResult();
}

