﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <curl/curl.h>
#include <nnt.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>

#define MIN(a, b) ((a) < (b) ? (a) : (b))

//-------------------------------------------------------------------------------------------------
//  Build flags
//-------------------------------------------------------------------------------------------------
#define RUN_CURL_POST_CHUNKED
#define RUN_CURL_POST_NORMAL
#define RUN_POST_WITHOUT_CURL

namespace
{
//-------------------------------------------------------------------------------------------------
//  Params
//-------------------------------------------------------------------------------------------------
SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

int                      g_SendBuffSize = 1024 * 8;
char*                    g_UrlString = nullptr;
size_t                   g_DataSize  = 0;
int                      g_ExecCount = 0;

//-------------------------------------------------------------------------------------------------
//  PostData
//-------------------------------------------------------------------------------------------------
class PostData
{
    char*  pData;
    char*  pDataItr;
    size_t leftBytes;
public:
    PostData() : pData(nullptr), pDataItr(nullptr), leftBytes(0){}
    ~PostData(){}

    size_t GetLeftSize() {return leftBytes;}
    char*  GetDataPtr()  {return pData;}
    char*  GetDataItr()  {return pDataItr;}
    void   UpdateDataItr(size_t updateSize)
    {
        pDataItr  += updateSize;
        leftBytes -= updateSize;
    }

    bool Initialize(size_t dataSize)
    {
        pData = new char[dataSize];
        if (pData)
        {
            for (uint64_t i = 0; i < dataSize - 2; i++)
            {
                pData[i] = '0' + i % 10;
            }
            pData[dataSize - 1] = '\0';
            pDataItr = pData;
            leftBytes = dataSize;
        }

        return (pData)?true:false;
    }

    void Finalize()
    {
        if (pData)
        {
            delete pData;
        }
        pData    = nullptr;
        pDataItr = nullptr;
    }
};

size_t CurlSslContextCallback(CURL* pCurl, void* pSslContext, void* pUserData)
{
    // Obtain pointer to the SSL context passed by CURLOPT_SSL_CTX_FUNCTION
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);

    // Create SSL context
    nn::Result result = pContext->Create(nn::ssl::Context::SslVersion_Auto);
    if( result.IsFailure() )
    {
        return (size_t) - 1;
    }

    return 0;
}

size_t readCallback(void *ptr, size_t size, size_t nmemb, void *userp)
{
    PostData* pData = (PostData*)userp;

    if (size*nmemb < 1)
        return 0;

    if (pData->GetLeftSize()) {
        size_t readsize = MIN(size*nmemb, (size_t)pData->GetLeftSize());
        memcpy(ptr, pData->GetDataItr(), readsize);
        pData->UpdateDataItr(readsize);
#if defined (NN_BUILD_CONFIG_OS_WIN)
        printf("readCallback %lu, left:%lu\n", (unsigned long)readsize, (unsigned long)pData->GetLeftSize());
#else
        printf("readCallback %zu, left:%zu\n", readsize, (size_t)pData->GetLeftSize());
#endif
        return readsize;
    }

    /* no more data left to deliver */
    return 0;
}

int traceCallback(CURL *handle, curl_infotype type, char *tracedata, size_t size, void *userp)
{
    NN_UNUSED(userp);
    NN_UNUSED(handle);

    const char *text;

    switch (type) {
    case CURLINFO_TEXT:
        NN_LOG("== Info: %s", tracedata);
    default: /* in case a new one is introduced to shock us */
        return 0;

    case CURLINFO_HEADER_OUT:
        text = "=> Send header";
        break;
    case CURLINFO_DATA_OUT:
        text = "=> Send data";
        break;
    case CURLINFO_SSL_DATA_OUT:
        text = "=> Send SSL data";
        break;
    case CURLINFO_HEADER_IN:
        text = "<= Recv header";
        break;
    case CURLINFO_DATA_IN:
        text = "<= Recv data";
        break;
    case CURLINFO_SSL_DATA_IN:
        text = "<= Recv SSL data";
        break;
    }
    NN_LOG("%s\n", text);

    return 0;
}

curl_socket_t sockOptCallback(void *clientp, curl_socket_t curlfd, curlsocktype purpose)
{
    int rval     = 0;
    int buffSize = 0;
    nn::socket::SockLenT optLen   = sizeof(buffSize);

    rval = nn::socket::GetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &buffSize, &optLen);
    if (rval < 0)
    {
        NN_LOG("GetSockOpt failed (errno:%d)\n", nn::socket::GetLastError());
        return CURL_SOCKOPT_ERROR;
    }
    NN_LOG("sockOptCallback: default - SNDBUF:%dbytes curlfd:%d\n", buffSize, curlfd);
    if (g_SendBuffSize != 0)
    {
        buffSize = g_SendBuffSize;
        optLen   = sizeof(buffSize);
        rval = nn::socket::SetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &buffSize, optLen);
        if (rval < 0)
        {
            NN_LOG("SetSockOpt failed (errno:%d)\n", nn::socket::GetLastError());
            return CURL_SOCKOPT_ERROR;
        }

        rval = nn::socket::GetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &buffSize, &optLen);
        if (rval < 0)
        {
            NN_LOG("GetSockOpt failed (errno:%d)\n", nn::socket::GetLastError());
            return CURL_SOCKOPT_ERROR;
        }

        NN_LOG("sockOptCallback: SNDBUF:%dbytes curlfd:%d\n", buffSize, curlfd);
    }
    return CURL_SOCKOPT_OK;
}

int CreateTcpSocket(
    bool bEstablishConn,
    uint16_t portNumber,
    const char* pInHostName,
    uint32_t ipAddress)
{
    int                tcpSocket;
    nn::socket::InAddr     inetAddr;
    nn::socket::SockAddrIn serverAddr;

    memset(&inetAddr, 0x00, sizeof(inetAddr));

    if(pInHostName == nullptr && ipAddress == 0)
    {
        NN_LOG("Host name and IP address was not passed.\n");
        return -1;
    }

    tcpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if(tcpSocket < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d)\n", errno);
        return -1;
    }
    NN_LOG("Created TCP socket (sockfd: %d).\n", tcpSocket);

    if (bEstablishConn != true)
    {
        return tcpSocket;
    }

    if(ipAddress == 0)
    {
        NN_LOG("Resolving %s\n", pInHostName);
        nn::socket::HostEnt *pHostEnt = nn::socket::GetHostEntByName(pInHostName);
        if(pHostEnt == nullptr)
        {
            NN_LOG("Failed to resolve host name (errno:%d)\n", errno);
            return -1;
        }

        // Just pick the first one
        memcpy(&inetAddr, pHostEnt->h_addr_list[0], sizeof(nn::socket::InAddr));
        serverAddr.sin_addr.S_addr = inetAddr.S_addr;
    }
    else
    {
        NN_LOG("Use 0x%x for server IP.\n", ipAddress);
        serverAddr.sin_addr.S_addr = ipAddress;
    }

    serverAddr.sin_family = nn::socket::Family::Af_Inet;
    serverAddr.sin_port   = nn::socket::InetHtons(portNumber);

    int rval = nn::socket::Connect(tcpSocket, (nn::socket::SockAddr*)&serverAddr, sizeof(serverAddr));
    if(rval < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d).\n", errno);
        nn::socket::Close(tcpSocket);
        return -1;
    }

    NN_LOG("Established TCP connection (addr:0x%x on port :%d).\n",
         serverAddr.sin_addr.S_addr, nn::socket::InetNtohs(serverAddr.sin_port));

    return tcpSocket;
}

} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
    ASSERT_TRUE(curl_global_init(CURL_GLOBAL_DEFAULT) == CURLE_OK);

    int    argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();
    if (argc != 5)
    {
        NN_LOG("Url tring, data size to send, send buffer size and exec loop count need to be passed (e.g. https://ntd-net-server1.ntd.nintendo.com 65536 8192 10)\n");
        ASSERT_TRUE(NN_STATIC_CONDITION(false));
    }

    g_UrlString    = argv[1];
    g_DataSize     = atoi(argv[2]);
    g_SendBuffSize = atoi(argv[3]);
    g_ExecCount    = atoi(argv[4]);
    NN_LOG("Test parameters\nURL:%s\nData size:%d\nSNDBUFF size:%d\nLoop count:%d\n",
           g_UrlString, g_DataSize, g_SendBuffSize, g_ExecCount);
}

#ifdef RUN_CURL_POST_CHUNKED
TEST(NormalPostChunked, Success)
{
    for (int i = 0; i < g_ExecCount; i++)
    {
        NN_LOG("Size of data to POST = %d bytes\n", g_DataSize);

        CURL*    curl;
        CURLcode res;
        PostData postData;

        ASSERT_TRUE(postData.Initialize(g_DataSize));
        curl = curl_easy_init();
        ASSERT_TRUE(curl != nullptr);

        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
        curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextCallback);

        curl_easy_setopt(curl, CURLOPT_URL, g_UrlString);
        curl_easy_setopt(curl, CURLOPT_POST, 1L);

        curl_easy_setopt(curl, CURLOPT_SOCKOPTFUNCTION, sockOptCallback);

        curl_easy_setopt(curl, CURLOPT_READFUNCTION, readCallback);
        curl_easy_setopt(curl, CURLOPT_READDATA, &postData);

        //curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
        //curl_easy_setopt(curl, CURLOPT_NOPROXY, "*");

        curl_easy_setopt(curl, CURLOPT_DEBUGFUNCTION, traceCallback);

        {
            struct curl_slist *chunk = NULL;
            chunk = curl_slist_append(chunk, "Expect:");
            res = curl_easy_setopt(curl, CURLOPT_HTTPHEADER, chunk);
            char tempBuf[128];
#if defined (NN_BUILD_CONFIG_OS_WIN)
            sprintf(tempBuf, "Content-Length: %lu", (unsigned long)g_DataSize);
#else
            sprintf(tempBuf, "Content-Length: %zu", g_DataSize);
#endif
            chunk = curl_slist_append(chunk, tempBuf);
            chunk = curl_slist_append(chunk, "application/x-www-form-urlencoded");
            curl_easy_setopt(curl, CURLOPT_HTTPHEADER, chunk);
        }

        res = curl_easy_perform(curl);
        if (res != CURLE_OK)
        {
            NN_LOG("curl_easy_perform() failed: %d %s\n", res, curl_easy_strerror(res));
        }
        else
        {
            NN_LOG("curl_easy_perform(): SUCCESS\n");
        }

        curl_easy_cleanup(curl);
        postData.Finalize();
    }
}
#endif // RUN_CURL_POST_CHUNKED

#ifdef RUN_CURL_POST_NORMAL
TEST(NormalPost, Success)
{
    for (int i = 0; i < g_ExecCount; i++)
    {
        CURL*    curl;
        CURLcode res;
        PostData postData;

        ASSERT_TRUE(postData.Initialize(g_DataSize));
        NN_LOG("Size of data to POST = %d bytes\n", postData.GetLeftSize());

        curl = curl_easy_init();
        ASSERT_TRUE(curl != nullptr);

        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
        curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextCallback);

        curl_easy_setopt(curl, CURLOPT_URL, g_UrlString);
        curl_easy_setopt(curl, CURLOPT_POST, 1L);

        curl_easy_setopt(curl, CURLOPT_SOCKOPTFUNCTION, sockOptCallback);

        curl_easy_setopt(curl, CURLOPT_READFUNCTION, readCallback);
        curl_easy_setopt(curl, CURLOPT_READDATA, &postData);

        //curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
        curl_easy_setopt(curl, CURLOPT_DEBUGFUNCTION, traceCallback);
        curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, postData.GetLeftSize());

        res = curl_easy_perform(curl);
        if (res != CURLE_OK)
        {
            NN_LOG("curl_easy_perform() failed: %d %s\n", res, curl_easy_strerror(res));
        }
        else
        {
            NN_LOG("curl_easy_perform(): SUCCESS\n");
        }

        curl_easy_cleanup(curl);
        postData.Finalize();
    }
}
#endif // RUN_CURL_POST_NORMAL

#ifdef RUN_POST_WITHOUT_CURL
TEST(SslPost, Success)
{
    int i = 0;

    int   portNumber      = 443;
    char  hostString[256] = {0};
    char* pTmp            = g_UrlString;
    if (strstr(pTmp, "https://"))
    {
        pTmp += strlen("https://");
    }

    do
    {
        hostString[i] = pTmp[i];
        i++;
    } while ((pTmp[i] != ':') && (pTmp[i] != '\0'));

    if (pTmp[i] == ':')
    {
        pTmp = &pTmp[i + 1];
        portNumber = atoi(pTmp);
    }

    for (i = 0; i < g_ExecCount; i++)
    {
        NN_LOG("Host:%s Port:%d\n", hostString, portNumber);

        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(portNumber), hostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        if (g_SendBuffSize > 0)
        {
            int buffSize = g_SendBuffSize;
            nn::socket::SockLenT optLen   = sizeof(buffSize);
            int rval = nn::socket::SetSockOpt(socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &buffSize, optLen);
            ASSERT_TRUE(rval >= 0);

            rval = nn::socket::GetSockOpt(socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &buffSize, &optLen);
            ASSERT_TRUE(rval >= 0);
            NN_LOG("Socket send buff size:%d\n", buffSize);
        }

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
        ASSERT_TRUE(result.IsSuccess());

        // Handshake **********************************************************************************
        uint32_t asyncHandshakeCount      = 0;
        uint32_t MaxAsyncDoHandshakeRetry = 30;
        do
        {
            result = sslConnection.DoHandshake();
            if(result.IsSuccess())
            {
                NN_LOG("SSL Handshake completed.\n");
                break;
            }
            else if(nn::ssl::ResultIoWouldBlock::Includes(result))
            {
                if(asyncHandshakeCount++ < MaxAsyncDoHandshakeRetry)
                {
                    NN_LOG("Handshake would block, try again after 100 msec (retry count:%d)\n",
                        asyncHandshakeCount);
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    continue;
                }
                else
                {
                    NN_LOG("Handshake failed after %d times of execution\n", asyncHandshakeCount);
                    break;
                }
            }
            else
            {
                NN_LOG("SSL Handshake failed (Description:%d)\n", result.GetDescription());
                if(nn::ssl::ResultVerifyCertFailed::Includes(result))
                {
                    nn::Result verifyCertError;
                    result = sslConnection.GetVerifyCertError(&verifyCertError);
                    if(result.IsFailure())
                    {
                        NN_LOG("Failed to get verify cert error (Description:%d)\n",
                            result.GetDescription());
                    }
                    else
                    {
                        NN_LOG("Certificate validation failed (Description:%d)\n",
                            verifyCertError.GetDescription());
                        result = verifyCertError;
                    }
                }
                break;
            }
        }
        while (NN_STATIC_CONDITION(true));
        ASSERT_TRUE(result.IsSuccess());

        // Send POST request **************************************************************************
        {
            char httpReqBuff[256] = {0};
#if defined (NN_BUILD_CONFIG_OS_WIN)
            sprintf(httpReqBuff, "POST / HTTP/1.1\r\nHost: %s\r\nAccept: */*\r\nContent-Length: %lu\r\nContent-Type: application/x-www-form-urlencoded\r\n\r\n",
                    hostString, (unsigned long)g_DataSize);
#else
            sprintf(httpReqBuff, "POST / HTTP/1.1\r\nHost: %s\r\nAccept: */*\r\nContent-Length: %zu\r\nContent-Type: application/x-www-form-urlencoded\r\n\r\n",
                    hostString, g_DataSize);
#endif
            int sentBytes     = 0;
            result = sslConnection.Write(httpReqBuff, &sentBytes, (uint32_t)strlen(httpReqBuff));
            if(result.IsFailure() || sentBytes < 0)
            {
                NN_LOG("Failed to write data. (Description:%d)\n", result.GetDescription());
            }
            else
            {
                NN_LOG("Sent HTTP request via SSL (%d bytes).\n", sentBytes);
            }
        }

        // Send Data **********************************************************************************
        int wouldBlockRetry = 0;

        PostData postData;
        ASSERT_TRUE(postData.Initialize(g_DataSize));
        NN_LOG("Size of data to POST = %d bytes \n", postData.GetLeftSize());
        do
        {
            if(postData.GetLeftSize() == 0)
            {
                NN_LOG("Sent all the data.\n");
                break;
            }
#if 0
            uint32_t msecPollTimeout  = 5000;
            nn::ssl::Connection::PollEvent pollEvent;
            nn::ssl::Connection::PollEvent pollOutEvent;

            pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
            pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Write;
            result = sslConnection.Poll(&pollOutEvent, &pollEvent, msecPollTimeout);
            if(result.IsFailure())
            {
                if(nn::ssl::ResultIoTimeout::Includes(result))
                {
                    NN_LOG("Finished receiving data for the timeout.\n");
                    break;
                }
                else
                {
                    NN_LOG("Failed due to Poll error (Description:%d)\n", result.GetDescription());
                    break;
                }
            }
            if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Write) !=
               nn::ssl::Connection::PollEvent::PollEvent_Write)
            {
                NN_LOG("NOT ready to write.\n");
            }
#endif
#if !defined (NN_BUILD_CONFIG_OS_WIN)
            int space = 0;
            int rval = nn::socket::Ioctl(socketFd, nn::socket::IoctlCommand::FionSpace, &space, sizeof(space));
            ASSERT_TRUE(rval >= 0);
            NN_LOG("nn::socket::IoctlCommand::FionSpace:%d\n", space);
#endif
            int sentBytes = 0;
            NN_LOG("Calling write: size:%d \n", postData.GetLeftSize());
            result = sslConnection.Write(postData.GetDataItr(), &sentBytes, postData.GetLeftSize());
            if(result.IsFailure() || sentBytes <= 0)
            {
                if(nn::ssl::ResultIoWouldBlock::Includes(result))
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    if (wouldBlockRetry++ > 300)
                    {
                        NN_LOG("Failed with max WOULDBLOCK retry.\n");
                        break;
                    }
                    continue;
                }
                else
                {
                    NN_LOG("Failed to write data (Description:%d)\n", result.GetDescription());
                    break;
                }
            }
            else
            {
                postData.UpdateDataItr(sentBytes);
                NN_LOG("---->Sent %d bytes (left:%d bytes)\n", sentBytes, postData.GetLeftSize());
            }
        } while(NN_STATIC_CONDITION(true));
        postData.Finalize();

        // Read until connection gets closed or timeout ***********************************************
        int receivedTotalBytes = 0;
        int readRetryMax = 10;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;
#if 0
            uint32_t msecPollTimeout = 500;
            nn::ssl::Connection::PollEvent pollEvent;
            nn::ssl::Connection::PollEvent pollOutEvent;
            pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
            pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Read;

            result = sslConnection.Poll(&pollOutEvent, &pollEvent, msecPollTimeout);
            if(nn::ssl::ResultIoTimeout::Includes(result))
            {
                NN_LOG("Finished receiving data for the timeout.\n");
                break;
            }
            if(result.IsFailure())
            {
                NN_LOG("Failed due to Poll error (Description:%d)\n", result.GetDescription());
                break;
            }

            if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read) !=
               nn::ssl::Connection::PollEvent::PollEvent_Read)
            {
                NN_LOG("NOT ready to read\n");
                continue;
            }
#endif
            result = sslConnection.Read(
                readBuff,
                &receivedBytes,
                sizeof(readBuff));
            if(nn::ssl::ResultIoWouldBlock::Includes(result))
            {
                if (--readRetryMax == 0)
                {
                    NN_LOG("Finished receiving data for max retry.\n");
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }

            if(result.IsFailure() || receivedBytes < 0)
            {
                NN_LOG("Failed to read data (Description:%d)\n", result.GetDescription());
                break;
            }
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
} // NOLINT(impl/function_size)
#endif

TEST(FinalizeTest, Success)
{
    curl_global_cleanup();
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
