﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>

#include <Common/testCommonUtil.h>

// ------------------------------------------------------------------------------------------------
// Macros
// ------------------------------------------------------------------------------------------------
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#define BREAK_ON_CURL_FAILURE(res)    \
    do{                               \
        if(res != CURLE_OK) {         \
            break;                    \
        }                             \
    } while (NN_STATIC_CONDITION(0))

// ------------------------------------------------------------------------------------------------
// Grobal parameters
// ------------------------------------------------------------------------------------------------
namespace
{
const char* NintendoServerAddress = "https://dauth-dd1.ndas.srv.nintendo.net/v1/device_auth_token";
const char* ProxyAddress          = "http://proxy.nintendo.co.jp";
const char  PostData[]            = "system_version=00000001&client_id=16e96f76850156d1\0";

SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
uint32_t                 g_PostDataSizeLeft = 0;
uint32_t                 g_PostDataSize     = 0;

// ------------------------------------------------------------------------------------------------
// Libcurl callbacks
// ------------------------------------------------------------------------------------------------
size_t CurlSslContextCallback(CURL* pCurl, void* pSslContext, void* pUserData)
{
    // Obtain pointer to the SSL context passed by CURLOPT_SSL_CTX_FUNCTION
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);

    // Create SSL context
    nn::Result result = pContext->Create(nn::ssl::Context::SslVersion_Auto);
    if( result.IsFailure() )
    {
        NN_LOG("Create failed (Desc:%d)\n", result.GetDescription());
        return (size_t) - 1;
    }

    nn::ssl::CertStoreId certStoreId;
    result = pContext->RegisterInternalPki(
        &certStoreId,
        nn::ssl::Context::InternalPki_DeviceClientCertDefault);
    if (result.IsFailure())
    {
        NN_LOG("Failed to register internal pki (Desc:%d)\n", result.GetDescription());
        return (size_t) - 1;
    }

    return 0;
}

size_t ReadCallback(void *ptr, size_t size, size_t nmemb, void *userp)
{
    size_t readsize = 0;

    if (size*nmemb < 1)
    {
        return 0;
    }

    if (g_PostDataSizeLeft > 0)
    {
        readsize = MIN(size*nmemb, g_PostDataSizeLeft);
        memcpy(ptr, &PostData[g_PostDataSize - g_PostDataSizeLeft], readsize);
        g_PostDataSizeLeft -= readsize;
    }

    return readsize;
}

} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());

    ASSERT_TRUE(nn::ssl::Initialize().IsSuccess());
}


TEST(InternalPkiWithNintendoServer, Success)
{
    CURL*    curl;
    CURLcode res;
    bool     isSuccess  = false;
    bool     isUseProxy = false;
    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (argc == 2)
    {
        if (strcmp(argv[1], "--use_proxy") == 0)
        {
            NN_LOG(" use_proxy option is enabled.\n");
            isUseProxy = true;
        }
        else
        {
            NN_LOG(" Unkown option was passed:%s\n", argv[1]);
        }
    }

    ASSERT_TRUE(curl_global_init(CURL_GLOBAL_DEFAULT) == CURLE_OK);
    do
    {
        curl = curl_easy_init();
        ASSERT_TRUE(curl != nullptr);

        BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L));
        BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_URL, NintendoServerAddress));
        BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 15L));
        BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L));
        BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L));
        BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextCallback));

        if (isUseProxy == true)
        {
            BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_PROXYAUTOCONFIG, 0));
            BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_PROXYAUTH, CURLAUTH_BASIC));
            BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_PROXY, ProxyAddress));
            BREAK_ON_CURL_FAILURE(curl_easy_setopt(curl, CURLOPT_PROXYPORT, 8080));
        }

        // Setup POST data
        g_PostDataSize     = strlen(PostData);
        g_PostDataSizeLeft = g_PostDataSize;
        curl_easy_setopt(curl, CURLOPT_POST, 1L);
        curl_easy_setopt(curl, CURLOPT_READFUNCTION, ReadCallback);
        curl_easy_setopt(curl, CURLOPT_READDATA, PostData);
        curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, g_PostDataSize);

        // Ready to perform
        res = curl_easy_perform(curl);
        if (res != CURLE_OK)
        {
            NN_LOG("curl_easy_perform FAILED: curl error: %d\n", res);
        }
        else
        {
            // Verify response code
            long httpResponseCode = 0;
            BREAK_ON_CURL_FAILURE(curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &httpResponseCode));
            EXPECT_TRUE(httpResponseCode == 200);

            NN_LOG("curl_easy_perform SUCCEEDED.\n");
            isSuccess = true;
        }
    } while (NN_STATIC_CONDITION(0));

    curl_easy_cleanup(curl);
    curl_global_cleanup();
    EXPECT_TRUE(isSuccess);
}


TEST(FinalizeTest, Success)
{
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
