﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <Common/testCommonUtil.h>

#define ENABLE_VERIFY
#define ENABLE_SESSION_CACHE

namespace
{
const char               g_ServerName[] = "www.google.com";
SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
} // Un-named namespace

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup -> %p\n", (void *)nninitStartup);

    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}


TEST(Handshake2x, Success)
{
    NN_LOG("nninitStartup:%p\n", (void *)nninitStartup);

    nn::Result result;

    NN_LOG("Init socket...\n");
    result = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
                                    nn::socket::DefaultSocketMemoryPoolSize,
                                    nn::socket::MinSocketAllocatorSize,
                                    nn::socket::DefaultConcurrencyLimit);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Init SSL...\n");
    result = nn::ssl::Initialize();
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::Context *pCtx = nullptr;
    nn::ssl::Connection *pCn = nullptr;

    NN_LOG("Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Create SSL connection...\n");
    pCn = new nn::ssl::Connection();
    ASSERT_TRUE(pCn != nullptr);
    result = pCn->Create(pCtx);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Create socket...\n");
    int socketFd =
        g_CommonUtil.CreateTcpSocket(true,
                                     443,
                                     g_ServerName,
                                     0);
    ASSERT_TRUE(socketFd >= 0);

    NN_LOG("Set SSL socket...\n");
    result = pCn->SetSocketDescriptor(socketFd);
    ASSERT_TRUE(result.IsSuccess());

    result = pCn->SetHostName(g_ServerName, strlen(g_ServerName));
    ASSERT_TRUE(result.IsSuccess());

#ifdef ENABLE_VERIFY
    NN_LOG("Set SSL verify to DEFAULT...\n");
    result = pCn->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_Default);
    ASSERT_TRUE(result.IsSuccess());
#else
    NN_LOG("Set SSL verify to NONE...\n");
    result = pCn->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
    ASSERT_TRUE(result.IsSuccess());
    result = pCn->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
    ASSERT_TRUE(result.IsSuccess());
#endif

#ifndef ENABLE_SESSION_CACHE
    NN_LOG("Set SSL cache mode to NONE...\n");
    result = pCn->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode_None);
    ASSERT_TRUE(result.IsSuccess());
#endif

    NN_LOG("Do handshake...\n");
    result = pCn->DoHandshake();
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Do handshake again...\n");
    result = pCn->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    pCn = new nn::ssl::Connection();
    result = pCn->Create(pCtx);
    ASSERT_TRUE(result.IsSuccess());
    socketFd =
        g_CommonUtil.CreateTcpSocket(true,
                                     443,
                                     g_ServerName,
                                     0);
    ASSERT_TRUE(socketFd >= 0);

    NN_LOG("Set SSL socket...\n");
    result = pCn->SetSocketDescriptor(socketFd);
    ASSERT_TRUE(result.IsSuccess());

    result = pCn->SetHostName(g_ServerName, strlen(g_ServerName));
    ASSERT_TRUE(result.IsSuccess());

#ifdef ENABLE_VERIFY
    NN_LOG("Set SSL verify to DEFAULT...\n");
    result = pCn->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_Default);
    ASSERT_TRUE(result.IsSuccess());
#else
    result = pCn->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
    ASSERT_TRUE(result.IsSuccess());
    NN_LOG("Set SSL verify to NONE...\n");
    result = pCn->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
    ASSERT_TRUE(result.IsSuccess());
#endif

#ifndef ENABLE_SESSION_CACHE
    NN_LOG("Set SSL cache mode to NONE...\n");
    result = pCn->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode_None);
    ASSERT_TRUE(result.IsSuccess());
#endif

    result = pCn->DoHandshake();
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Tear down SSL connection...\n");
    result = pCn->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCn;
    pCn = nullptr;

    NN_LOG("Tear down SSL context...\n");
    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;

    NN_LOG("Finalize SSL...\n");
    result = nn::ssl::Finalize();
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Finalize socket...\n");
    nn::socket::Finalize();
}
