﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>


#define ENABLE_VERIFY
//#define ENABLE_VERIFY_HOSTNAME
#define ENABLE_SESSION_CACHE

#define ENABLE_IMPORT_CRL
#define ENABLE_IMPORT_CRL_NO_ISSUER
#define ENABLE_IMPORT_CRL_BAD_SIG
#define ENABLE_IMPORT_CRL_REVOKED_TRUST_CERT
#define ENABLE_IMPORT_CRL_CTX_BLEED

namespace
{
const char               g_GoogleSrv[] = "google.com";

const char               *g_TestSrv = ServerName;

const int                g_NumTestSrvCerts = 2;
const char               *g_TestSrvCerts[g_NumTestSrvCerts] =
{
    g_pTestRvkRootCa,
    g_pTestRvkIntermediateCa,
};

const size_t             g_TestSrvCertSizes[g_NumTestSrvCerts] =
{
    strlen(g_pTestRvkRootCa),
    strlen(g_pTestRvkIntermediateCa),
};


SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];


nn::Result CreateSslConnection(nn::ssl::Connection **pOutCn,
                               nn::ssl::Context    *pInCtx,
                               const char          *srvName,
                               short               srvPort)
{
    nn::Result                  result;
    nn::ssl::Connection         *pCn;

    do
    {
        *pOutCn = nullptr;
        pCn = new nn::ssl::Connection();
        if (pCn == nullptr)
        {
            result = nn::ssl::ResultInsufficientMemory();
            break;
        }

        result = pCn->Create(pInCtx);
        if (result.IsFailure())
        {
            break;
        }

        int socketFd =
            g_CommonUtil.CreateTcpSocket(true, srvPort, srvName, 0);
        if (socketFd < 0)
        {
            result = nn::ssl::ResultNoTcpConnection();
            break;
        }

        result = pCn->SetSocketDescriptor(socketFd);
        if (result.IsFailure())
        {
            break;
        }

        result = pCn->SetHostName(srvName, strlen(srvName));
        if (result.IsFailure())
        {
            break;
        }

        nn::ssl::Connection::VerifyOption   verify =
            nn::ssl::Connection::VerifyOption::VerifyOption_None;

#ifdef ENABLE_VERIFY
        verify |= nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa |
                  nn::ssl::Connection::VerifyOption::VerifyOption_DateCheck;
#ifdef ENABLE_VERIFY_HOSTNAME
        verify |= nn::ssl::Connection::VerifyOption::VerifyOption_HostName;
#endif
        result = pCn->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
        if(result.IsFailure())
        {
            break;
        }

        result = pCn->SetVerifyOption(verify);
        if (result.IsFailure())
        {
            break;
        }
#endif

#ifndef ENABLE_SESSION_CACHE
        result = pCn->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode_None);
        if (result.IsFailure())
        {
            break;
        }
#endif

        *pOutCn = pCn;
    } while(NN_STATIC_CONDITION(false));

    if (*pOutCn == nullptr)
    {
        if (pCn != nullptr)
        {
            pCn->Destroy();
            delete pCn;
            pCn = nullptr;
        }
    }

    return result;
}

} // Un-named namespace

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup -> %p\n", (void *)nninitStartup);

    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}


TEST(Initialize, Success)
{
    nn::Result result;

    NN_LOG("Setup network...\n");
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());

    NN_LOG("Init socket...\n");
    result = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
                                    nn::socket::DefaultSocketMemoryPoolSize,
                                    nn::socket::MinSocketAllocatorSize,
                                    nn::socket::DefaultConcurrencyLimit);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Init SSL...\n");
    result = nn::ssl::Initialize();
    ASSERT_TRUE(result.IsSuccess());
}


#ifdef ENABLE_IMPORT_CRL
TEST(ImportCrl, Success)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[ImportCrl, Success] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->ImportServerPki(&srvCertIds[i],
                                       g_TestSrvCerts[i],
                                       g_TestSrvCertSizes[i],
                                       nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[ImportCrl, Success] import srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    NN_LOG("[ImportCrl, Success] Import IA CRL...\n");
    result = pCtx->ImportCrl(&crlId,
                             reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                             sizeof(g_pTestCrlRvkServer));
    ASSERT_TRUE(result.IsSuccess());

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}


TEST(RemoveCrl, Success)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[RemoveCrl, Success] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->ImportServerPki(&srvCertIds[i],
                                       g_TestSrvCerts[i],
                                       g_TestSrvCertSizes[i],
                                       nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[RemoveCrl, Success] import srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    NN_LOG("[RemoveCrl, Success] Import IA CRL...\n");
    result = pCtx->ImportCrl(&crlId,
                             reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                             sizeof(g_pTestCrlRvkServer));
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[RemoveCrl, Success] Remove test CRL...\n");
    result = pCtx->RemovePki(crlId);
    ASSERT_TRUE(result.IsSuccess());

    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->RemovePki(srvCertIds[i]);
        ASSERT_TRUE(result.IsSuccess());
    }

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}


TEST(ImportCrlHandshake, NotRevoked)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::Connection         *pCn = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[ImportCrlHandshake, NotRevoked] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->ImportServerPki(&srvCertIds[i],
                                       g_TestSrvCerts[i],
                                       g_TestSrvCertSizes[i],
                                       nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[ImportCrlHandshake, NotRevoked] import srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    NN_LOG("[ImportCrlHandshake, NotRevoked] Import IA CRL...\n");
    result = pCtx->ImportCrl(&crlId,
                             reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                             sizeof(g_pTestCrlRvkServer));
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlHandshake, NotRevoked] Create SSL connection...\n");
    result = CreateSslConnection(&pCn, pCtx, g_GoogleSrv, 443);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlHandshake, NotRevoked] Do handshake...\n");
    result = pCn->DoHandshake();
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlHandshake, NotRevoked] Tear down SSL connection...\n");
    result = pCn->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCn;
    pCn = nullptr;

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}


TEST(ImportCrlHandshake, Revoked)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::Connection         *pCn = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[ImportCrlHandshake, Revoked] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->ImportServerPki(&srvCertIds[i],
                                       g_TestSrvCerts[i],
                                       g_TestSrvCertSizes[i],
                                       nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[ImportCrlHandshake, Revoked] import srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    NN_LOG("[ImportCrlHandshake, Revoked] Import IA CRL...\n");
    result = pCtx->ImportCrl(&crlId,
                             reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                             sizeof(g_pTestCrlRvkServer));
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlHandshake, Revoked] Create SSL connection...\n");
    result = CreateSslConnection(&pCn, pCtx, g_TestSrv, ServerPort_SrvRevoked);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlHandshake, Revoked] Do handshake...\n");
    result = pCn->DoHandshake();
    ASSERT_TRUE(result.IsFailure());
    ASSERT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));
    nn::Result verifyResult;
    result = pCn->GetVerifyCertError(&verifyResult);
    ASSERT_TRUE(result.IsSuccess());
    ASSERT_TRUE(nn::ssl::ResultSslErrorRevokedCertificate::Includes(verifyResult));

    NN_LOG("[ImportCrlHandshake, Revoked] Tear down SSL connection...\n");
    result = pCn->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCn;
    pCn = nullptr;

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}


TEST(ImportCrl, SingleCtxLimit)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::CertStoreId        crlId;
    uint32_t                    count;

    NN_LOG("[ImportCrl, SingleCtxLimit] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    //  Import server PKI so our CRL is recognized
    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->ImportServerPki(&srvCertIds[i],
                                       g_TestSrvCerts[i],
                                       g_TestSrvCertSizes[i],
                                       nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[ImportCrl, SingleCtxLimit] import srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    //  Repeatedly import CRL, ensur we are limited by the lib
    for (count = 0;
         count <= nn::ssl::MaxAppCrlImportCount;
         count++)
    {
        NN_LOG("[ImportCrl, SingleCtxLimit] Import IA CRL (%d)..\n",
               count);
        result = pCtx->ImportCrl(&crlId,
                                 reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                                 sizeof(g_pTestCrlRvkServer));
        if (count < nn::ssl::MaxAppCrlImportCount)
        {
            ASSERT_TRUE(result.IsSuccess());
        }
        else
        {
            ASSERT_TRUE(result.IsFailure());
            ASSERT_TRUE(nn::ssl::ResultResourceMax::Includes(result));
        }
    }

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}


TEST(ImportCrl, MultiCtxLimit)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx[2] = { nullptr, nullptr };
    nn::ssl::CertStoreId        crlId;
    uint32_t                    count;
    int                         j;

    for (j = 0; j < 2; j++)
    {
        NN_LOG("[ImportCrl, MultiCtxLimit] Create SSL context %d...\n", j);
        pCtx[j] = new nn::ssl::Context();
        ASSERT_TRUE(pCtx[j] != nullptr);
        result = pCtx[j]->Create(nn::ssl::Context::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());
    }

    //  Import server PKI so our CRL is recognized
    for (j = 0; j < 2; j++)
    {
        nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
        for (int i = 0; i < g_NumTestSrvCerts; i++)
        {
            result = pCtx[j]->ImportServerPki(&srvCertIds[i],
                                              g_TestSrvCerts[i],
                                              g_TestSrvCertSizes[i],
                                              nn::ssl::CertificateFormat_Pem);
            ASSERT_TRUE(result.IsSuccess());
            NN_LOG("[ImportCrl, MultiCtxLimit] import srv %d, id %llu\n",
                   i,
                   srvCertIds[i]);
        }
    }

    //  Repeatedly import CRL, ensur we are limited by the lib
    for (count = 0;
         count <= nn::ssl::MaxAppCrlImportCount;
         count++)
    {
        j = count % 2;

        NN_LOG("[ImportCrl, MultiCtxLimit] Import IA CRL (%d) on ctx %d..\n",
               count,
               j);

        result = pCtx[j]->ImportCrl(&crlId,
                                    reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                                    sizeof(g_pTestCrlRvkServer));
        if (count < nn::ssl::MaxAppCrlImportCount)
        {
            ASSERT_TRUE(result.IsSuccess());
        }
        else
        {
            ASSERT_TRUE(result.IsFailure());
            ASSERT_TRUE(nn::ssl::ResultResourceMax::Includes(result));
        }
    }

    for (j = 0; j < 2; j++)
    {
        result = pCtx[j]->Destroy();
        ASSERT_TRUE(result.IsSuccess());
        delete pCtx[j];
        pCtx[j] = nullptr;
    }
}


namespace
{
struct CrlImportThreadArgs
{
    int                         threadNum;
    nn::ssl::Context            *pCtx;
    int                         importCount;
    bool                        fatalError;
    bool                        running;
};

static const int                g_MULTI_THREAD_COUNT = 2;
static const uint32_t           g_MULTI_THREAD_IMPORT_ITERATIONS = 1000;
static const size_t             g_THREAD_STACK_SIZE = 8192;

NN_OS_ALIGNAS_GUARDED_STACK  uint8_t  threadStacks[g_MULTI_THREAD_COUNT][g_THREAD_STACK_SIZE];


//  MultiThreadCtxLimit thread entry point
void CrlImportThreadCb(void *pArg)
{
    int                         i;
    CrlImportThreadArgs         *pImportArgs = reinterpret_cast<CrlImportThreadArgs *>(pArg);
    nn::ssl::CertStoreId        crlIds[nn::ssl::MaxAppCrlImportCount];
    nn::Result                  result;

    //NN_LOG("[CrlImportThreadCb:%d] start\n", pImportArgs->threadNum);

    //  Attempt to import CRL repeatedly, up to the max number.  If we fail,
    //  then unroll and exit.
    pImportArgs->fatalError = false;
    memset(&crlIds, 0, sizeof(crlIds));
    for (i = 0; i < nn::ssl::MaxAppCrlImportCount; i++)
    {
        result =
            pImportArgs->pCtx->ImportCrl(&crlIds[i],
                                         reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                                         sizeof(g_pTestCrlRvkServer));
        if (result.IsFailure())
        {
            if (!nn::ssl::ResultResourceMax::Includes(result))
            {
                NN_LOG("[CrlImportThreadCb:%d] FATAL ERROR DURING IMPORT: %d-%d\n",
                       pImportArgs->threadNum,
                       result.GetModule(),
                       result.GetDescription());
                pImportArgs->fatalError = true;
            }

            //  Normal case where resources are maxed, break out
            break;
        }

        pImportArgs->importCount++;
        nn::os::YieldThread();
    }

    //  Reverse walk it, remove the CRLs which were just imported.  We need to
    //  init i to -1 of what is currently is because if import completed
    //  successfully it is at the max (beyond our array size) and if it failed
    //  then it is at an index which does not have a valid id.
    for (i--; i >= 0; i--)
    {
        result = pImportArgs->pCtx->RemovePki(crlIds[i]);
        if (result.IsFailure())
        {
            NN_LOG("[CrlImportThreadCb] unable to release CRL id %X: %d-%d\n",
                   crlIds[i],
                   result.GetModule(),
                   result.GetDescription());

            crlIds[i] = 0;

            //  Do not break just continue
        }
    }
}

}

TEST(ImportCrl, MultiThreadCtxLimit)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx[g_MULTI_THREAD_COUNT];
    int                         i;
    int                         j;
    nn::os::MultiWaitType       waiter;
    nn::os::ThreadType          threads[g_MULTI_THREAD_COUNT];
    nn::os::MultiWaitHolderType threadMw[g_MULTI_THREAD_COUNT];
    uint32_t                    compCount[g_MULTI_THREAD_COUNT];
    CrlImportThreadArgs         threadArgs[g_MULTI_THREAD_COUNT];
    nn::os::MultiWaitHolderType *pDoneWaiter;
    int                         waitCount;

    //  Similar to the multi-context limit, except this test uses multiple
    //  threads to do the imports repeatedly in order to exercise any
    //  race condition handling as well as import limits.
    for (j = 0; j < g_MULTI_THREAD_COUNT; j++)
    {
        NN_LOG("[ImportCrl, MultiThreadCtxLimit] Create SSL context %d...\n", j);
        pCtx[j] = new nn::ssl::Context();
        ASSERT_TRUE(pCtx[j] != nullptr);
        result = pCtx[j]->Create(nn::ssl::Context::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());
    }

    //  Import server PKI so our CRL is recognized
    for (j = 0; j < g_MULTI_THREAD_COUNT; j++)
    {
        nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
        for (i = 0; i < g_NumTestSrvCerts; i++)
        {
            result = pCtx[j]->ImportServerPki(&srvCertIds[i],
                                              g_TestSrvCerts[i],
                                              g_TestSrvCertSizes[i],
                                              nn::ssl::CertificateFormat_Pem);
            ASSERT_TRUE(result.IsSuccess());
            NN_LOG("[ImportCrl, MultiThreadCtxLimit] import srv %d, id %llu\n",
                   i,
                   srvCertIds[i]);
        }
    }

    InitializeMultiWait(&waiter);
    memset(&threadArgs, 0, sizeof(threadArgs));
    memset(&compCount, 0, sizeof(compCount));

    waitCount = 0;
    while (NN_STATIC_CONDITION(true))
    {
        for (i = 0; i < g_MULTI_THREAD_COUNT; i++)
        {
            //  If we aren't already waiting for this thread, create it
            //  and get it going.
            if ((compCount[i] > g_MULTI_THREAD_IMPORT_ITERATIONS) ||
                threadArgs[i].running)
            {
                continue;
            }

            threadArgs[i].fatalError = false;
            threadArgs[i].pCtx = pCtx[i];
            threadArgs[i].threadNum = i;
            result = CreateThread(&threads[i],
                                  CrlImportThreadCb,
                                  &threadArgs[i],
                                  &threadStacks[i][0],
                                  g_THREAD_STACK_SIZE,
                                  nn::os::DefaultThreadPriority);
            ASSERT_TRUE(result.IsSuccess());

            threadArgs[i].running = true;
            nn::os::InitializeMultiWaitHolder(&threadMw[i], &threads[i]);
            nn::os::LinkMultiWaitHolder(&waiter, &threadMw[i]);
            //NN_LOG("[ImportCrl, MultiThreadCtxLimit] start thread %d\n", i);
            nn::os::StartThread(&threads[i]);
            waitCount++;
        }

        if (waitCount == 0)
        {
            //  done
            break;
        }

        //  Wait any
        pDoneWaiter = nn::os::WaitAny(&waiter);

        //  Find the match, unlink, update running flag, destroy thread
        for (i = 0; i < g_MULTI_THREAD_COUNT; i++)
        {
            if (pDoneWaiter != &threadMw[i])
            {
                continue;
            }

            ASSERT_TRUE(threadArgs[i].fatalError == false);
            nn::os::UnlinkMultiWaitHolder(&threadMw[i]);
            nn::os::FinalizeMultiWaitHolder(&threadMw[i]);
            threadArgs[i].running = false;
            nn::os::DestroyThread(&threads[i]);
            compCount[i]++;
            waitCount--;

            if ((compCount[i] % 100) == 0)
            {
                NN_LOG("[ImportCrl, MultiThreadCtxLimit] thread %d, iteration %d, good imports %d\n",
                       i,
                       compCount[i],
                       threadArgs[i].importCount);
            }
        }
    }

    //  Cleanup
    nn::os::FinalizeMultiWait(&waiter);

    for (i = 0; i < g_MULTI_THREAD_COUNT; i++)
    {
        result = pCtx[i]->Destroy();
        ASSERT_TRUE(result.IsSuccess());
        delete pCtx[i];
        pCtx[i] = nullptr;
    }
}
#endif    //  ENABLE_IMPORT_CRL


#ifdef ENABLE_IMPORT_CRL_NO_ISSUER
TEST(ImportCrl, NoTrustedIssuer)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[ImportCrl, NoTrustedIssuer] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrl, NoTrustedIssuer] Import IA CRL...\n");
    result = pCtx->ImportCrl(&crlId,
                             reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                             sizeof(g_pTestCrlRvkServer));
    ASSERT_TRUE(result.IsFailure());
    ASSERT_TRUE(nn::ssl::ResultUnknownCrlIssuer::Includes(result));

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}
#endif    //  ENABLE_IMPORT_CRL_NO_ISSUER


#ifdef ENABLE_IMPORT_CRL_BAD_SIG
TEST(ImportCrl, BadSignature)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[ImportCrl, BadSignature] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->ImportServerPki(&srvCertIds[i],
                                       g_TestSrvCerts[i],
                                       g_TestSrvCertSizes[i],
                                       nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[ImportCrl, BadSignature] import srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    NN_LOG("[ImportCrl, BadSignature] Import ROOT (BAD) CRL...\n");
    result = pCtx->ImportCrl(&crlId,
                             reinterpret_cast<const char *>(g_pTestCrlRvkIntermediateCaBadSig),
                             sizeof(g_pTestCrlRvkIntermediateCaBadSig));
    ASSERT_TRUE(result.IsFailure());
    ASSERT_TRUE(nn::ssl::ResultCrlSignatureVerifyFail::Includes(result));

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}
#endif    //  ENABLE_IMPORT_CRL_BAD_SIG


#ifdef ENABLE_IMPORT_CRL_REVOKED_TRUST_CERT
TEST(ImportCrlHandshake, RevokedIa)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx = nullptr;
    nn::ssl::Connection         *pCn = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[ImportCrlHandshake, RevokedIa] Create SSL context...\n");
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);
    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx->ImportServerPki(&srvCertIds[i],
                                       g_TestSrvCerts[i],
                                       g_TestSrvCertSizes[i],
                                       nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[ImportCrlHandshake, RevokedIa] import srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    NN_LOG("[ImportCrlHandshake, RevokedIa] Import ROOT CRL...\n");
    result = pCtx->ImportCrl(&crlId,
                             reinterpret_cast<const char *>(g_pTestCrlRvkIntermediateCa),
                             sizeof(g_pTestCrlRvkIntermediateCa));
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlHandshake, RevokedIa] Create SSL connection...\n");
    result = CreateSslConnection(&pCn, pCtx, g_TestSrv, ServerPort_SrvRevoked);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlHandshake, RevokedIa] Do handshake...\n");
    result = pCn->DoHandshake();
    ASSERT_TRUE(result.IsFailure());
    ASSERT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));
    nn::Result verifyResult;
    result = pCn->GetVerifyCertError(&verifyResult);
    ASSERT_TRUE(result.IsSuccess());
    ASSERT_TRUE(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyResult));

    NN_LOG("[ImportCrlHandshake, RevokedIa] Tear down SSL connection...\n");
    result = pCn->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCn;
    pCn = nullptr;

    result = pCtx->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx;
    pCtx = nullptr;
}
#endif    //  ENABLE_IMPORT_CRL_REVOKED_TRUST_CERT


#ifdef ENABLE_IMPORT_CRL_CTX_BLEED
TEST(ImportCrlCtxBleed, Success)
{
    nn::Result                  result;
    nn::ssl::Context            *pCtx1 = nullptr;
    nn::ssl::Context            *pCtx2 = nullptr;
    nn::ssl::Connection         *pCn = nullptr;
    nn::ssl::CertStoreId        crlId;

    NN_LOG("[ImportCrlCtxBleed, Success] Create SSL context 1...\n");
    pCtx1 = new nn::ssl::Context();
    ASSERT_TRUE(pCtx1 != nullptr);
    result = pCtx1->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlCtxBleed, Success] Create SSL context 2...\n");
    pCtx2 = new nn::ssl::Context();
    ASSERT_TRUE(pCtx2 != nullptr);
    result = pCtx2->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId        srvCertIds[g_NumTestSrvCerts];
    for (int i = 0; i < g_NumTestSrvCerts; i++)
    {
        result = pCtx1->ImportServerPki(&srvCertIds[i],
                                        g_TestSrvCerts[i],
                                        g_TestSrvCertSizes[i],
                                        nn::ssl::CertificateFormat_Pem);
        ASSERT_TRUE(result.IsSuccess());
        NN_LOG("[ImportCrlCtxBleed, Success] import on context 1 srv %d, id %llu\n",
               i,
               srvCertIds[i]);
    }

    NN_LOG("[ImportCrlCtxBleed, Success] Import IA CRL on context 2...\n");
    result = pCtx2->ImportCrl(&crlId,
                              reinterpret_cast<const char *>(g_pTestCrlRvkServer),
                              sizeof(g_pTestCrlRvkServer));
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlCtxBleed, Success] Create SSL connection...\n");
    result = CreateSslConnection(&pCn, pCtx1, g_TestSrv, ServerPort_SrvRevoked);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlCtxBleed, Success] Do handshake...\n");
    result = pCn->DoHandshake();
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("[ImportCrlCtxBleed, Success] Tear down SSL connection...\n");
    result = pCn->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCn;
    pCn = nullptr;

    result = pCtx2->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx2;
    pCtx2 = nullptr;

    result = pCtx1->Destroy();
    ASSERT_TRUE(result.IsSuccess());
    delete pCtx1;
    pCtx1 = nullptr;
}
#endif    //  ENABLE_IMPORT_CRL_CTX_BLEED


TEST(Finalize, Success)
{
    nn::Result result;

    NN_LOG("Finalize SSL...\n");
    result = nn::ssl::Finalize();
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("Finalize socket...\n");
    nn::socket::Finalize();
}
