﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <nnc/nn_Macro.h>
#include <nnc/nn_Result.h>
#include <nnc/ssl/ssl.h>
#include <nnc/socket/socket_Api.h>

#include "util.h"
// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
#define MIN(x,y) ((x > y)?(y):(x))
#define RESULT_IS_FAILURE(result) (nnResultIsFailure(result))
#define REPORT_ERROR_AND_RETURN_UPON_FAILURE(errString, result)                                   \
    if(RESULT_IS_FAILURE(result))                                                                 \
    {                                                                                             \
        UTIL_LOG("FAILED - %s (result:%d desc:%d)\n",                                             \
           errString, result._value, nnResultGetDescription(result));                             \
        return false;                                                                             \
    }

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
int CreateTcpSocket(bool bEstablishConn, const char* pInHostName)
{
    int                rval;
    int                tcpSocket;
    struct in_addr     inetAddr;
    struct hostent*    pHostEnt;
    struct sockaddr_in serverAddr;
    memset(&inetAddr, 0x00, sizeof(inetAddr));

    tcpSocket = nnsocketSocket(AF_INET, SOCK_STREAM, 0);
    if(tcpSocket < 0)
    {
        UTIL_LOG("Failed to create TCP socket (errno: %d)\n", errno);
        return -1;
    }
    UTIL_LOG("Created TCP socket (sockfd: %d).\n", tcpSocket);

    if (bEstablishConn)
    {
        UTIL_LOG("Resolving %s\n", pInHostName);
        pHostEnt = nnsocketGetHostByName(pInHostName);

        if(pHostEnt == NULL)
        {
            UTIL_LOG("Failed to resolve host name (errno:%d)\n", errno);
            return -1;
        }

        // Just pick the first one
        memcpy(&inetAddr, pHostEnt->h_addr_list[0], sizeof(struct in_addr));
        UTIL_LOG("pInHostName: 0x%x\n", inetAddr.s_addr);

        serverAddr.sin_addr.s_addr = inetAddr.s_addr;
        serverAddr.sin_family      = AF_INET;
        serverAddr.sin_port        = nnsocketInetHtons(443);

        rval = nnsocketConnect(tcpSocket, (struct sockaddr*)&serverAddr, sizeof(serverAddr));
        if(tcpSocket < 0)
        {
            UTIL_LOG("Failed to create TCP socket (errno: %d)\n", errno);
            return -1;
        }
        UTIL_LOG("Established connection.\n");
    }

    return tcpSocket;
}

bool CUtilInitializeSsl()
{
    REPORT_ERROR_AND_RETURN_UPON_FAILURE(
        "Failed to initialize the SSL library.\n",
        nnsslInitialize());

    return true;
}

bool CUtilFinalizeSsl()
{
    REPORT_ERROR_AND_RETURN_UPON_FAILURE(
        "Failed to finalize the SSL library.\n",
        nnsslFinalize());

    return true;
}

// Perform SSL data transfer by C interface
bool CUtilPerformSslTransfer(const char* pInHostName)
{
    nnResult result;
    int sockFd;
    nnsslSslConnectionType sslConnection;
    nnsslSslContextType    sslContext;
    nnsslSslContextId      contextId;
    nnsslSslConnectionId   connectionId;
    char                   pServerCertBuffer[1024 * 16] = {0};

    if(CheckAndDumpNnSslClassSize() != true)
    {
        return false;
    }

    sockFd = CreateTcpSocket(true, pInHostName);
    if (sockFd < 0)
    {
        return false;
    }

    result = nnsslContextCreate(&sslContext, nnsslContextSslVersion_Auto);
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslContextCreate", result);

    result = nnsslContextGetContextId(&sslContext, &contextId);
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslContextGetContextId", result);
    UTIL_LOG("Created context (ID:%llX)\n", (unsigned long long)contextId);

    result = nnsslConnectionCreate(&sslConnection, &sslContext);
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionCreate", result);

    // Just check C linkage validity (they'll fail but intentionally called here)
    {
        int      tmpSize;
        char     tmpBuff[32];
        uint32_t tmpBuffSize = sizeof(tmpBuff);
        nnResult tmpResult;

        result = nnsslConnectionDoHandshake(&sslConnection);
        result = nnsslConnectionPending(&sslConnection, &tmpSize);
        result = nnsslConnectionPeek(&sslConnection, tmpBuff, &tmpSize, tmpBuffSize);
        result = nnsslConnectionGetVerifyCertError(&sslConnection, &tmpResult);
    }

    // Set/Get socket descriptor
    {
        int tmpFd = 0;
        result = nnsslConnectionSetSocketDescriptor(&sslConnection, sockFd);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetSocketDescriptor", result);
        result = nnsslConnectionGetSocketDescriptor(&sslConnection, &tmpFd);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionGetSocketDescriptor", result);
        UTIL_LOG("(Set/Get) Socket descriptor: %s\n", (tmpFd == sockFd)?"OK":"NG");
    }

    // Set/Get host name
    {
        char     pTmpName[64]   = {0};
        uint32_t hostNameLen    = sizeof(pTmpName);
        uint32_t hostNameOutLen = 0;

        result = nnsslConnectionSetHostName(&sslConnection, pInHostName, strlen(pInHostName));
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetHostName", result);
        result = nnsslConnectionGetHostName(&sslConnection, pTmpName, &hostNameOutLen, hostNameLen);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionGetHostName", result);
        UTIL_LOG("(Set/Get) Host name: %s\n",
            (strncmp(pInHostName, pTmpName, MIN(strlen(pInHostName), hostNameLen)) == 0)?"OK":"NG");
    }

    // Set/Get verify option
    {
        nnsslConnectionVerifyOption tmpOption = 0;

        result = nnsslConnectionSetOption(&sslConnection, nnsslConnectionOptionType_SkipDefaultVerify, true);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetOption", result);
        result = nnsslConnectionSetVerifyOption(&sslConnection, nnsslConnectionVerifyOption_PeerCa);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetVerifyOption", result);
        result = nnsslConnectionGetVerifyOption(&sslConnection, &tmpOption);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionGetVerifyOption", result);
        UTIL_LOG("(Set/Get) Verify option: %s\n", (tmpOption == nnsslConnectionVerifyOption_PeerCa)?"OK":"NG");
        result = nnsslConnectionSetVerifyOption(&sslConnection, nnsslConnectionVerifyOption_None);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetVerifyOption", result);
    }

    // Set /Get I/O mode
    {
        nnsslConnectionIoMode tmpIoMode = 0;

        result = nnsslConnectionSetIoMode(&sslConnection, nnsslConnectionIoMode_NonBlocking);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetIoMode", result);
        result = nnsslConnectionGetIoMode(&sslConnection, &tmpIoMode);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionGetIoMode", result);
        UTIL_LOG("(Set/Get) I/O mode: %s\n", (tmpIoMode == nnsslConnectionIoMode_NonBlocking)?"OK":"NG");
        result = nnsslConnectionSetIoMode(&sslConnection, nnsslConnectionIoMode_Blocking);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetIoMode", result);
    }

    result = nnsslConnectionSetServerCertBuffer(&sslConnection, pServerCertBuffer, sizeof(pServerCertBuffer));
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionSetServerCertBuffer", result);

    // Get context ID
    result = nnsslConnectionGetContextId(&sslConnection, &contextId);
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionGetContextId", result);
    UTIL_LOG("nnsslConnectionGetContextId - %llX\n", (unsigned long long)contextId);

    // Get connection ID
    result = nnsslConnectionGetConnectionId(&sslConnection, &connectionId);
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionGetConnectionId", result);
    UTIL_LOG("nnsslConnectionGetConnectionId - %llX\n", (unsigned long long)connectionId);


    // Perform actual SSL Handshake
    {
        uint32_t outServerCertSize = 0;
        result = nnsslConnectionDoHandshakeWithCertBuffer(&sslConnection, &outServerCertSize, NULL);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionDoHandshakeWithCertBuffer", result);
        UTIL_LOG("nnsslConnectionDoHandshakeWithCertBuffer - server cert size: %d\n", outServerCertSize);
    }

    // Write
    {
        char     httpReqBuff[64] = {0};
        uint32_t httpReqSize = 0;
        int      writtenSize = 0;
        snprintf(httpReqBuff, sizeof(httpReqBuff), "GET / HTTP/1.0\r\nHost: %s\r\n\r\n", pInHostName);
        httpReqSize = (uint32_t)strlen(httpReqBuff);
        result = nnsslConnectionWrite(&sslConnection, httpReqBuff, &writtenSize, httpReqSize);
        REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionWrite", result);
        UTIL_LOG("Sent HTTP request via SSL (%d bytes).\n", writtenSize);
    }

    // Read
    {
        const int msecPollTimeout = 5000;
        const int readBufferSize  = 1024 * 1024;
        int receivedTotalBytes    = 0;
        char* readBuf = malloc(readBufferSize);
        if(readBuf == NULL)
        {
            UTIL_LOG("Failed to allocate read buffer.\n");
            return false;
        }
        memset(readBuf, 0x00, readBufferSize);

        do {
            nnsslConnectionPollEvent pollEvent = nnsslConnectionPollEvent_None;
            nnsslConnectionPollEvent pollOutEvent = nnsslConnectionPollEvent_None;

            pollEvent |= nnsslConnectionPollEvent_Read;
            result = nnsslConnectionPoll(&sslConnection, &pollOutEvent, &pollEvent, msecPollTimeout);
            REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionPoll", result);
            if(pollOutEvent & nnsslConnectionPollEvent_Read)
            {
                char readTmpBuff[1024] = {0};
                int readSize = 0;
                result = nnsslConnectionRead(&sslConnection, readTmpBuff, &readSize, sizeof(readTmpBuff));
                REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionRead", result);
                if(readSize == 0)
                {
                    UTIL_LOG("Connection closed by the server.\n");
                    break;
                }

                memcpy(&readBuf[receivedTotalBytes], readTmpBuff, readSize);
                receivedTotalBytes += readSize;
            }
        } while(1);

        if(readBuf != NULL)
        {
            free(readBuf);
        }
        UTIL_LOG("Received %d bytes.\n", receivedTotalBytes);
    }

    result = nnsslConnectionDestroy(&sslConnection);
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslConnectionDestroy", result);

    result = nnsslContextDestroy(&sslContext);
    REPORT_ERROR_AND_RETURN_UPON_FAILURE("nnsslContextDestroy", result);

    return true;
} // NOLINT(impl/function_size)
