﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <Common/testCommonUtil.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_TimeZoneApi.h>
#endif

#include "PkiInfo.h"
#include "Common.h"


// ------------------------------------------------------------------------------------------------
// Note
// ------------------------------------------------------------------------------------------------
/*

[APIs in nn::ssl]
--------------------+-------------+----------------------------------------------------------------
  namespace/class   |     From    | ID  | API
--------------------+-------------+----------------------------------------------------------------
nn::ssl             | ----------- | --- |
                    | Pre-Install | 100 | nn::Result Initialize() NN_NOEXCEPT;
                    | Pre-Install | 101 | nn::Result Initialize(uint32_t concurLimit) NN_NOEXCEPT;
                    | Pre-Install | 102 | nn::Result Finalize() NN_NOEXCEPT;
                    | Pre-Install | 103 | nn::Result GetSslResultFromValue(
                    | ----------- | --- |     nn::Result* pOutResult,
                    | ----------- | --- |     const char* pInValue,
                    | ----------- | --- |     uint32_t inValueSize) NN_NOEXCEPT;
--------------------+-------------+----------------------------------------------------------------
nn::ssl::Context    | ----------- | --- |
                    | Pre-Install | 200 | nn::Result Create(SslVersion version) NN_NOEXCEPT;
                    | Pre-Install | 201 | nn::Result Destroy() NN_NOEXCEPT;
                    | Pre-Install | 202 | nn::Result ImportServerPki(
                    | ----------- | --- |     CertStoreId* pOutCertId,
                    | ----------- | --- |     const char* pInCertData,
                    | ----------- | --- |     uint32_t certDataSize,
                    | ----------- | --- |     CertificateFormat certFormat) NN_NOEXCEPT;
                    | Pre-Install | 203 | nn::Result RemovePki(CertStoreId certId) NN_NOEXCEPT;
                    | Pre-Install | 204 | nn::Result GetContextId(SslContextId* pOutValue) NN_NOEXCEPT;
--------------------+-------------+----------------------------------------------------------------
nn::ssl::Connection | ----------- | --- |
                    | Pre-Install | 300 | nn::Result Create(Context* pInSslContext) NN_NOEXCEPT;
                    | Pre-Install | 301 | nn::Result Destroy() NN_NOEXCEPT;
                    | Pre-Install | 302 | nn::Result SetSocketDescriptor(int socketDescriptor) NN_NOEXCEPT;
                    | Pre-Install | 303 | nn::Result SetHostName(const char* pInHostName, uint32_t hostNameLength) NN_NOEXCEPT;
                    | Pre-Install | 304 | nn::Result SetVerifyOption(VerifyOption optionValue) NN_NOEXCEPT;
                    | Pre-Install | 305 | nn::Result SetServerCertBuffer(char* pOutBuffer, uint32_t bufferLength) NN_NOEXCEPT;
                    | Pre-Install | 306 | nn::Result SetIoMode(IoMode mode) NN_NOEXCEPT;
                    | Pre-Install | 307 | nn::Result SetSessionCacheMode(SessionCacheMode mode) NN_NOEXCEPT;
                    | Pre-Install | 308 | nn::Result SetRenegotiationMode(RenegotiationMode mode) NN_NOEXCEPT;
                    | Pre-Install | 309 | nn::Result GetSocketDescriptor(int* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 310 | nn::Result GetHostName(char* pOutHostNameBuffer, uint32_t* pOutHostNameLength, uint32_t hostNameBufferLength) NN_NOEXCEPT;
                    | Pre-Install | 311 | nn::Result GetVerifyOption(VerifyOption* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 312 | nn::Result GetIoMode(IoMode* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 313 | nn::Result GetSessionCacheMode(SessionCacheMode* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 314 | nn::Result GetRenegotiationMode(RenegotiationMode* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 315 | nn::Result FlushSessionCache() NN_NOEXCEPT;
                    | Pre-Install | 316 | nn::Result DoHandshake() NN_NOEXCEPT;
                    | Pre-Install | 317 | nn::Result DoHandshake(uint32_t* pOutServerCertSize, uint32_t* pOutServerCertCount) NN_NOEXCEPT;
                    | Pre-Install | 318 | int        Read(char* pOutBuffer, uint32_t bufferLength) NN_NOEXCEPT;
                    | Pre-Install | 319 | nn::Result Read(char* pOutBuffer, int* pOutReadSizeCourier, uint32_t bufferLength) NN_NOEXCEPT;
                    | Pre-Install | 320 | int        Write(const char* pInBuffer, uint32_t bufferLength) NN_NOEXCEPT;
                    | Pre-Install | 321 | nn::Result Write(const char* pInBuffer, int* pOutWrittenSizeCourier, uint32_t bufferLength) NN_NOEXCEPT;
                    | Pre-Install | 322 | int        Pending() NN_NOEXCEPT;
                    | Pre-Install | 323 | nn::Result Pending(int* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 324 | nn::Result Peek(char* pOutBuffer, int* pOutReadSizeCourier, uint32_t bufferLength) NN_NOEXCEPT;
                    | Pre-Install | 325 | nn::Result Poll(PollEvent* pOutEvent, PollEvent* pInEvent, uint32_t msecTimeout) NN_NOEXCEPT;
                    | Pre-Install | 326 | nn::Result GetLastError(nn::Result* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 327 | nn::Result GetVerifyCertError(nn::Result* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 328 | nn::Result GetNeededServerCertBufferSize(uint32_t *pOutValue);
                    | Pre-Install | 329 | nn::Result GetContextId(SslContextId* pOutValue) NN_NOEXCEPT;
                    | Pre-Install | 330 | nn::Result GetConnectionId(SslConnectionId* pOutValue) NN_NOEXCEPT;
--------------------+-------------+----------------------------------------------------------------

*/

namespace
{
// ------------------------------------------------------------------------------------------------
// Grobal parameters
// ------------------------------------------------------------------------------------------------
const char     TestHost[]           = "natf.com";
const uint16_t HttpsPortNumber      = 441;

SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

// ------------------------------------------------------------------------------------------------
// Test result params
// ------------------------------------------------------------------------------------------------

TestParam g_testResult[] = {
    {false, 100, "Pre-Install", "nn::ssl::Initialize()"},
    {false, 101, "Pre-Install", "nn::ssl::Initialize(uint32_t concurLimit)"},
    {false, 102, "Pre-Install", "nn::ssl::Finalize"},
    {false, 103, "Pre-Install", "nn::ssl::GetSslResultFromValue"},
    {false, 200, "Pre-Install", "nn::ssl::Context::Create"},
    {false, 201, "Pre-Install", "nn::ssl::Context::Destroy"},
    {false, 202, "Pre-Install", "nn::ssl::Context::ImportServerPki"},
    {false, 203, "Pre-Install", "nn::ssl::Context::RemovePki"},
    {false, 204, "Pre-Install", "nn::ssl::Context::GetContextId"},
    {false, 300, "Pre-Install", "nn::ssl::Connection::Create"},
    {false, 301, "Pre-Install", "nn::ssl::Connection::Destroy"},
    {false, 302, "Pre-Install", "nn::ssl::Connection::SetSocketDescriptor"},
    {false, 303, "Pre-Install", "nn::ssl::Connection::SetHostName"},
    {false, 304, "Pre-Install", "nn::ssl::Connection::SetVerifyOption"},
    {false, 305, "Pre-Install", "nn::ssl::Connection::SetServerCertBuffer"},
    {false, 306, "Pre-Install", "nn::ssl::Connection::SetIoMode"},
    {false, 307, "Pre-Install", "nn::ssl::Connection::SetSessionCacheMode"},
    {false, 308, "Pre-Install", "nn::ssl::Connection::SetRenegotiationMode"},
    {false, 309, "Pre-Install", "nn::ssl::Connection::GetSocketDescriptor"},
    {false, 310, "Pre-Install", "nn::ssl::Connection::GetHostName"},
    {false, 311, "Pre-Install", "nn::ssl::Connection::GetVerifyOption"},
    {false, 312, "Pre-Install", "nn::ssl::Connection::GetIoMode"},
    {false, 313, "Pre-Install", "nn::ssl::Connection::GetSessionCacheMode"},
    {false, 314, "Pre-Install", "nn::ssl::Connection::GetRenegotiationMode"},
    {false, 315, "Pre-Install", "nn::ssl::Connection::FlushSessionCache"},
    {false, 316, "Pre-Install", "nn::ssl::Connection::DoHandshake()"},
    {false, 317, "Pre-Install", "nn::ssl::Connection::DoHandshake(uint32_t* pOutServerCertSize, uint32_t* pOutServerCertCount)"},
    {false, 318, "Pre-Install", "nn::ssl::Connection::Read(char* pOutBuffer, uint32_t bufferLength)"},
    {false, 319, "Pre-Install", "nn::ssl::Connection::Read(char* pOutBuffer, int* pOutReadSizeCourier, uint32_t bufferLength)"},
    {false, 320, "Pre-Install", "nn::ssl::Connection::Write(const char* pInBuffer, uint32_t bufferLength)"},
    {false, 321, "Pre-Install", "nn::ssl::Connection::Write(const char* pInBuffer, int* pOutWrittenSizeCourier, uint32_t bufferLength)"},
    {false, 322, "Pre-Install", "nn::ssl::Connection::Pending()"},
    {false, 323, "Pre-Install", "nn::ssl::Connection::Pending(int* pOutValue)"},
    {false, 324, "Pre-Install", "nn::ssl::Connection::Peek"},
    {false, 325, "Pre-Install", "nn::ssl::Connection::Poll"},
    {false, 326, "Pre-Install", "nn::ssl::Connection::GetLastError"},
    {false, 327, "Pre-Install", "nn::ssl::Connection::GetVerifyCertError"},
    {false, 328, "Pre-Install", "nn::ssl::Connection::GetNeededServerCertBufferSize"},
    {false, 329, "Pre-Install", "nn::ssl::Connection::GetContextId"},
    {false, 330, "Pre-Install", "nn::ssl::Connection::GetConnectionId"},
};

const int TestIdCount = sizeof(g_testResult) / sizeof(g_testResult[0]);

} // Un-named namespace

//-------------------------------------------------------------------------------------------------
// InitTest - Initialization of the test
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());

    // Testing ID - 100 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    // Testing ID - 101 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::Result result = nn::ssl::Initialize();
    ASSERT_TRUE(result.IsSuccess());
    SetTestDoneFlag(g_testResult, TestIdCount, 100, result.IsSuccess());
    SetTestDoneFlag(g_testResult, TestIdCount, 101, result.IsSuccess());

#ifndef NN_BUILD_CONFIG_OS_WIN32
    nn::time::PosixTime posixTime;
    nn::time::CalendarTime calendarTime;

    result = nn::time::Initialize();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    result = nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    result = nn::time::ToCalendarTime(&calendarTime, nullptr, posixTime);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    NN_LOG(
        "\n Date/Time:  %04d-%02d-%02d %02d:%02d:%02d\n\n",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);

    nn::time::Finalize();
#endif
}

//-------------------------------------------------------------------------------------------------
// WithoutHandshake - Test cases which doesn't need the SSL handshake
//-------------------------------------------------------------------------------------------------
TEST(WithoutHandshake, Success)
{
    nn::Result result;
    int socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestHost, 0);
    ASSERT_TRUE(socketFd >= 0);

    // Testing ID - 200 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::Context* pSslContext = new nn::ssl::Context();
    ASSERT_TRUE(pSslContext != nullptr);
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 200, result.IsSuccess());

    // Testing ID - 204 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::SslContextId contextId;
    result = pSslContext->GetContextId(&contextId);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    ASSERT_TRUE(contextId != 0);
    SetTestDoneFlag(g_testResult, TestIdCount, 204, result.IsSuccess());

    // Testing ID - 300 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();
    ASSERT_TRUE(pSslConnection != nullptr);
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 300, result.IsSuccess());

    // Testing ID - 330 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::SslConnectionId connectionId;
    result = pSslConnection->GetConnectionId(&connectionId);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(connectionId != 0);
    SetTestDoneFlag(g_testResult, TestIdCount, 330, result.IsSuccess());

    // Testing ID - 329 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::SslContextId contextId2;
    result = pSslConnection->GetContextId(&contextId2);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(contextId2 != 0);
    EXPECT_TRUE(contextId2 == contextId);
    SetTestDoneFlag(g_testResult, TestIdCount, 329, result.IsSuccess());

    // Testing ID - 302 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 302, result.IsSuccess());

    // Testing ID - 309 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    int tmpSocketFd = 0;
    result = pSslConnection->GetSocketDescriptor(&tmpSocketFd);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(socketFd == tmpSocketFd);
    SetTestDoneFlag(g_testResult, TestIdCount, 309, result.IsSuccess());

    // Testing ID - 307 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::Connection::SessionCacheMode sessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;
    result = pSslConnection->GetSessionCacheMode(&sessionCacheMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(sessionCacheMode == nn::ssl::Connection::SessionCacheMode_SessionId);
    sessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;
    result = pSslConnection->SetSessionCacheMode(sessionCacheMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 307, result.IsSuccess());

    // Testing ID - 313 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->GetSessionCacheMode(&sessionCacheMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(sessionCacheMode == nn::ssl::Connection::SessionCacheMode_None);
    SetTestDoneFlag(g_testResult, TestIdCount, 313, result.IsSuccess());

    // Testing ID - 308 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::Connection::RenegotiationMode renegotiationMode = nn::ssl::Connection::RenegotiationMode_None;
    result = pSslConnection->GetRenegotiationMode(&renegotiationMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(renegotiationMode == nn::ssl::Connection::RenegotiationMode_Secure);
    renegotiationMode = nn::ssl::Connection::RenegotiationMode_None;
    result = pSslConnection->SetRenegotiationMode(renegotiationMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 308, result.IsSuccess());

    // Testing ID - 314 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->GetRenegotiationMode(&renegotiationMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(renegotiationMode == nn::ssl::Connection::RenegotiationMode_None);
    SetTestDoneFlag(g_testResult, TestIdCount, 314, result.IsSuccess());

    // Testing ID - 303 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    char     tmpHostName[64] = {0};
    uint32_t tmpHostNameLen    = sizeof(tmpHostName);
    uint32_t tmpHostNameOutLen = 0;
    result = pSslConnection->SetHostName(TestHost, static_cast<uint32_t>(strlen(TestHost)));
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 303, result.IsSuccess());

    // Testing ID - 310 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->GetHostName(tmpHostName, &tmpHostNameOutLen, tmpHostNameLen);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(strcmp(TestHost,tmpHostName) == 0);
    EXPECT_TRUE(strlen(TestHost) == tmpHostNameOutLen);
    SetTestDoneFlag(g_testResult, TestIdCount, 310, result.IsSuccess());

    // Testing ID - 304 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::Connection::VerifyOption tmpVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
    result = pSslConnection->GetVerifyOption(&tmpVerifyOption);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(tmpVerifyOption == nn::ssl::Connection::VerifyOption::VerifyOption_Default);
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 304, result.IsSuccess());

    // Testing ID - 311 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    tmpVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
    result = pSslConnection->GetVerifyOption(&tmpVerifyOption);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(tmpVerifyOption == nn::ssl::Connection::VerifyOption::VerifyOption_All);
    SetTestDoneFlag(g_testResult, TestIdCount, 311, result.IsSuccess());

    // Testing ID - 306 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::ssl::Connection::IoMode tmpIoMode = nn::ssl::Connection::IoMode_NonBlocking;
    result = pSslConnection->GetIoMode(&tmpIoMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(tmpIoMode == nn::ssl::Connection::IoMode_Blocking);
    tmpIoMode = nn::ssl::Connection::IoMode_Blocking;
    result = pSslConnection->SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 306, result.IsSuccess());

    // Testing ID - 312 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->GetIoMode(&tmpIoMode);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(tmpIoMode == nn::ssl::Connection::IoMode_NonBlocking);
    SetTestDoneFlag(g_testResult, TestIdCount, 312, result.IsSuccess());

    // Testing ID - 301 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->Destroy();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    delete pSslConnection;
    SetTestDoneFlag(g_testResult, TestIdCount, 301, result.IsSuccess());

    // Testing ID - 201 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslContext->Destroy();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    delete pSslContext;
    SetTestDoneFlag(g_testResult, TestIdCount, 201, result.IsSuccess());
} // NOLINT(impl/function_size)

//-------------------------------------------------------------------------------------------------
// WithHandshake - Test cases which needs the SSL handshake
//-------------------------------------------------------------------------------------------------
TEST(WithHandshake1, Success)
{
    nn::Result           result;
    nn::Result           verifyError;
    int                  socketFd;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;

    // ------------------------------------------------------------------------
    // 1st connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestHost);

    // Testing ID - 316 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->DoHandshake();
    CHECK_HANDSHAKE_RESULT(nn::ssl::ResultVerifyCertFailed::Includes(result), result, pSslConnection);
    SetTestDoneFlag(g_testResult, TestIdCount, 316, nn::ssl::ResultVerifyCertFailed::Includes(result));

    // Testing ID - 103 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::Result resultFromValue = nn::ResultSuccess();
    uint64_t resultValue = static_cast<uint64_t>(result.GetInnerValueForDebug());
    result = nn::ssl::GetSslResultFromValue(
        &resultFromValue,
        reinterpret_cast<char*>(&resultValue),
        sizeof(resultValue));
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE_AND_LOG(nn::ssl::ResultVerifyCertFailed::Includes(resultFromValue), resultFromValue);
    SetTestDoneFlag(g_testResult, TestIdCount, 103, result.IsSuccess());

    // Testing ID - 327 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->GetVerifyCertError(&verifyError);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE_AND_LOG(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyError), verifyError);
    SetTestDoneFlag(g_testResult, TestIdCount, 327, result.IsSuccess());

    CleanupSsl(pSslContext, pSslConnection);
}

TEST(WithHandshake2, Success)
{
    nn::Result           result;
    nn::Result           verifyError;
    nn::ssl::CertStoreId certStoreId;
    int                  socketFd;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;
    // ------------------------------------------------------------------------
    // 2nd connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestHost);

    result = pSslContext->ImportServerPki(
        &certStoreId,
        reinterpret_cast<const char*>(NatfServerCa),
        NatfServerCaLen,
        nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    // Testing ID - 203 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslContext->RemovePki(certStoreId);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 203, result.IsSuccess());

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE_AND_LOG(nn::ssl::ResultVerifyCertFailed::Includes(result), result);
    result = pSslConnection->GetVerifyCertError(&verifyError);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE_AND_LOG(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyError), verifyError);
    result = pSslConnection->FlushSessionCache();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    CleanupSsl(pSslContext, pSslConnection);
}

TEST(WithHandshake3, Success)
{
    nn::Result           result;
    nn::Result           verifyError;
    nn::ssl::CertStoreId certStoreId;
    int                  socketFd;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;
    uint32_t             certSize = 0;

    // ------------------------------------------------------------------------
    // 3rd connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestHost);

    // Testing ID - 202 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslContext->ImportServerPki(
        &certStoreId,
        reinterpret_cast<const char*>(NatfServerCa),
        NatfServerCaLen,
        nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 202, result.IsSuccess());

    // Testing ID - 305 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    char* pServerCertBuff = new char[1024 * 4];
    result = pSslConnection->SetServerCertBuffer(pServerCertBuff, 1024 * 4);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 305, result.IsSuccess());

    // Testing ID - 317 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    certSize = 0;
    result = pSslConnection->DoHandshake(&certSize, nullptr);
    CHECK_HANDSHAKE_RESULT(result.IsSuccess(), result, pSslConnection);
    EXPECT_TRUE(certSize > 0);
    delete[] pServerCertBuff;
    SetTestDoneFlag(g_testResult, TestIdCount, 317, result.IsSuccess());

    // Testing ID - 315 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->FlushSessionCache();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 315, result.IsSuccess());

    CleanupSsl(pSslContext, pSslConnection);
}

TEST(WithHandshake4, Success)
{
    nn::Result           result;
    nn::Result           verifyError;
    int                  socketFd;
    nn::ssl::CertStoreId certStoreId;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;
    uint32_t             certSize = 0;

    // ------------------------------------------------------------------------
    // 4th connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestHost);

    result = pSslContext->ImportServerPki(
        &certStoreId,
        reinterpret_cast<const char*>(NatfServerCa),
        NatfServerCaLen,
        nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    char pInvalidServerCert;
    result = pSslConnection->SetServerCertBuffer(&pInvalidServerCert, sizeof(pInvalidServerCert));
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    result = pSslConnection->DoHandshake(&certSize, nullptr);
    CHECK_HANDSHAKE_RESULT(nn::ssl::ResultInsufficientServerCertBuffer::Includes(result), result, pSslConnection);

    // Testing ID - 328 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    uint32_t neededServerCertBuffSize = 0;
    result = pSslConnection->GetNeededServerCertBufferSize(&neededServerCertBuffSize);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    EXPECT_TRUE(neededServerCertBuffSize > 0);
    SetTestDoneFlag(g_testResult, TestIdCount, 328, result.IsSuccess());

    result = pSslConnection->FlushSessionCache();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    CleanupSsl(pSslContext, pSslConnection);
}

TEST(WithHandshake5, Success)
{
    nn::Result           result;
    nn::Result           verifyError;
    int                  socketFd;
    nn::ssl::CertStoreId certStoreId;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;
    char                 httpReqBuff[64] = {0};
    uint32_t             httpReqBuffLen = 0;
    int                  sentBytes = 0;

    // ------------------------------------------------------------------------
    // 5th connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestHost);

    result = pSslContext->ImportServerPki(
        &certStoreId,
        reinterpret_cast<const char*>(NatfServerCa),
        NatfServerCaLen,
        nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    result = pSslConnection->DoHandshake();
    CHECK_HANDSHAKE_RESULT(result.IsSuccess(), result, pSslConnection);

    MY_SNPRINTF(httpReqBuff, sizeof(httpReqBuff), "GET / HTTP/1.0\r\nHost: %s\r\n\r\n", TestHost);
    httpReqBuffLen = static_cast<uint32_t>(strlen(httpReqBuff));

    // Testing ID - 320 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    sentBytes = 0;
    sentBytes = pSslConnection->Write(httpReqBuff, httpReqBuffLen);
    ASSERT_TRUE(sentBytes > 0);
    SetTestDoneFlag(g_testResult, TestIdCount, 320, !!(sentBytes > 0));

    do
    {
        char tmpBuff[1024] = {0};
        // Testing ID - 322 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        int pendingBytes = pSslConnection->Pending();
        EXPECT_TRUE(pendingBytes >= 0);
        SetTestDoneFlag(g_testResult, TestIdCount, 322, !!(pendingBytes >= 0));

        // Testing ID - 318 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        int receivedBytes = pSslConnection->Read(tmpBuff, sizeof(tmpBuff));
        EXPECT_TRUE(receivedBytes >= 0);
        SetTestDoneFlag(g_testResult, TestIdCount, 318, !!(receivedBytes >= 0));
        if(receivedBytes < 0)
        {
            NN_LOG("nn::ssl::Read failed!\n");
            break;
        }
        if(receivedBytes == 0)
        {
            break;
        }
    } while(NN_STATIC_CONDITION(true));

    result = pSslConnection->Destroy();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    // Testing ID - 326 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    {
        char tmpBuff;
        int receivedBytes = pSslConnection->Read(&tmpBuff, sizeof(tmpBuff));
        EXPECT_TRUE(receivedBytes == -1);
        nn::Result errLastIoResult = nn::ResultSuccess();
        result = pSslConnection->GetLastError(&errLastIoResult);
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        EXPECT_TRUE(errLastIoResult.IsFailure());
        SetTestDoneFlag(g_testResult, TestIdCount, 326, errLastIoResult.IsFailure());
    }
    delete pSslConnection;
    result = pSslContext->Destroy();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    delete pSslContext;
}

TEST(WithHandshake6, Success)
{
    nn::Result           result;
    nn::Result           verifyError;
    int                  socketFd;
    nn::ssl::CertStoreId certStoreId;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;
    char                 httpReqBuff[64] = {0};
    uint32_t             httpReqBuffLen = 0;
    int                  sentBytes = 0;

    // ------------------------------------------------------------------------
    // 6th connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestHost);

    result = pSslContext->ImportServerPki(
        &certStoreId,
        reinterpret_cast<const char*>(NatfServerCa),
        NatfServerCaLen,
        nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    result = pSslConnection->DoHandshake();
    CHECK_HANDSHAKE_RESULT(result.IsSuccess(), result, pSslConnection);

    memset(httpReqBuff, 0x00, sizeof(httpReqBuff));
    MY_SNPRINTF(httpReqBuff, sizeof(httpReqBuff), "GET / HTTP/1.0\r\nHost: %s\r\n\r\n", TestHost);
    httpReqBuffLen = static_cast<uint32_t>(strlen(httpReqBuff));

    // Testing ID - 321 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslConnection->Write(httpReqBuff, &sentBytes, httpReqBuffLen);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    ASSERT_TRUE(sentBytes > 0);
    SetTestDoneFlag(g_testResult, TestIdCount, 321, result.IsSuccess());

    do
    {
        char tmpBuff[1024] = {0};

        // Testing ID - 325 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        nn::ssl::Connection::PollEvent pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
        nn::ssl::Connection::PollEvent pollOutEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
        pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Read;
        result = pSslConnection->Poll(&pollOutEvent, &pollEvent, 1000);
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        SetTestDoneFlag(g_testResult, TestIdCount, 325, result.IsSuccess());
        if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read)
           != nn::ssl::Connection::PollEvent::PollEvent_Read)
        {
            continue;
        }

        // Testing ID - 323 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        int pendingBytes = 0;
        result = pSslConnection->Pending(&pendingBytes);
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        EXPECT_TRUE(pendingBytes >= 0);
        SetTestDoneFlag(g_testResult, TestIdCount, 323, result.IsSuccess());

        // Testing ID - 324 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        int peekBytes = 0;
        result = pSslConnection->Peek(tmpBuff, &peekBytes, sizeof(tmpBuff));
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        EXPECT_TRUE(peekBytes >= 0);
        SetTestDoneFlag(g_testResult, TestIdCount, 324, result.IsSuccess());

        // Testing ID - 319 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        int receivedBytes = 0;
        result = pSslConnection->Read(tmpBuff, &receivedBytes, sizeof(tmpBuff));
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        EXPECT_TRUE(receivedBytes >= 0);
        SetTestDoneFlag(g_testResult, TestIdCount, 319, result.IsSuccess());
        if(receivedBytes < 0)
        {
            NN_LOG("nn::ssl::Read failed!\n");
            break;
        }
        if(receivedBytes == 0)
        {
            break;
        }
    } while(NN_STATIC_CONDITION(true));

    CleanupSsl(pSslContext, pSslConnection);
}

//-------------------------------------------------------------------------------------------------
// FinalizeTest - Finalization of the test
//-------------------------------------------------------------------------------------------------
TEST(FinalizeTest, Success)
{
    // Testing ID - 102 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    nn::Result result = nn::ssl::Finalize();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 102, result.IsSuccess());

    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();

    DumpTestResult(g_testResult, TestIdCount);
}

//-------------------------------------------------------------------------------------------------
// nninitStartup
//-------------------------------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}
