﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ro.h>
#include <nn/ro/ro_Types.h>

#include "util_Common.h"

#include <nn/nn_Log.h>
#include <nn/svc/svc_Base.h>

namespace {
    const char* BinWeak1Name = "nro/BinWeak1.nro";
    const char* BinWeak2Name = "nro/BinWeak2.nro";

    typedef uintptr_t (*ReturnUintptr)();
    typedef int (*ReturnInt)(int);

    const char* GetNotDefinedWeakFuncAddressName = "_Z28GetNotDefinedWeakFuncAddressv";
    const char* CallWeakFuncName = "_Z12CallWeakFunci";
    const char* GetWeakVariableAddressName = "_Z22GetWeakVariableAddressv";

    class WeakTest : public ::testing::TestWithParam<nn::ro::BindFlag>
    {
    protected:
        virtual void SetUp()
        {
            m_Allocator = &TestAllocator::GetInstance();
            m_Nro1.SetUp(BinWeak1Name, m_Allocator->GetAllocator());
            m_Nro2.SetUp(BinWeak2Name, m_Allocator->GetAllocator());
            nn::ro::Initialize();
        }

        virtual void TearDown()
        {
            nn::ro::Finalize();
        }

        TestAllocator* m_Allocator;
        TestNro m_Nro1;
        TestNro m_Nro2;
    };
} // namespace

INSTANTIATE_TEST_CASE_P(ManualDll, WeakTest, ::testing::Values(nn::ro::BindFlag_Lazy, nn::ro::BindFlag_Now));

TEST_P(WeakTest, NotDefined)
{
    auto result = m_Nro1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    result = nn::ro::LookupSymbol(&addr, GetNotDefinedWeakFuncAddressName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnUintptr getAddr = reinterpret_cast<ReturnUintptr>(addr);

    ASSERT_EQ(getAddr(), 0);

    m_Nro1.Unload();
}

TEST_P(WeakTest, BeforeWeakSymbol)
{
    auto result = m_Nro1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_Nro2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    result = nn::ro::LookupSymbol(&addr, CallWeakFuncName);
    ASSERT_RESULT_SUCCESS(result);

    ReturnInt func = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(func(1), 1);

    result = nn::ro::LookupSymbol(&addr, GetWeakVariableAddressName);
    ASSERT_RESULT_SUCCESS(result);

    ReturnUintptr getAddr = reinterpret_cast<ReturnUintptr>(addr);
    int* ptr = reinterpret_cast<int*>(getAddr());
    ASSERT_EQ(*ptr, 1);

    m_Nro1.Unload();
    m_Nro2.Unload();
}

TEST_P(WeakTest, AfterWeakSymbol)
{
    auto result = m_Nro2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_Nro1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    result = nn::ro::LookupSymbol(&addr, CallWeakFuncName);
    ASSERT_RESULT_SUCCESS(result);

    ReturnInt func = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(func(1), 2);

    result = nn::ro::LookupSymbol(&addr, GetWeakVariableAddressName);
    ASSERT_RESULT_SUCCESS(result);

    ReturnUintptr getAddr = reinterpret_cast<ReturnUintptr>(addr);
    int* ptr = reinterpret_cast<int*>(getAddr());
    ASSERT_EQ(*ptr, 2);

    m_Nro2.Unload();

    if (GetParam() == nn::ro::BindFlag_Lazy)
    {
        // 遅延解決の場合、未解決に一度戻されるので、
        // unload 後にアクセスすると、weak で定義した関数にアクセスする。
        result = nn::ro::LookupSymbol(&addr, CallWeakFuncName);
        ASSERT_RESULT_SUCCESS(result);

        ReturnInt func = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(func(1), 1);
    }

    m_Nro1.Unload();
}

