﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ro.h>
#include <nn/ro/ro_Types.h>

#include "util_Common.h"

#include <nn/nn_Log.h>

namespace {
    const char* BinProtectedName = "nro/BinProtected.nro";
    const char* BinAccessProtectedName = "nro/BinAccessProtected.nro";

    typedef int (*ReturnInt)();
    typedef uintptr_t (*ReturnUintptr)();
    typedef void (*SetInt)(int);
    const char* ProtectedFuncName = "_Z13ProtectedFuncv";
    const char* CallProtectedFuncName = "_Z17CallProtectedFuncv";
    const char* GetProtectedVariableAddressName = "_Z27GetProtectedVariableAddressv";
    const char* ProtectedValueName = "g_ProtectedVariable";
    const char* ProtectedFuncName2 = "_Z14ProtectedFunc2v";
    const char* CallProtectedFuncName2 = "_Z18CallProtectedFunc2v";
    const char* GetProtectedVariableAddressName2 = "_Z28GetProtectedVariableAddress2v";
    const char* ProtectedValueName2 = "g_ProtectedVariable2";

    class ProtectedTest : public ::testing::TestWithParam<nn::ro::BindFlag>
    {
    protected:
        virtual void SetUp()
        {
            m_Allocator = &TestAllocator::GetInstance();
            m_Nro1.SetUp(BinProtectedName, m_Allocator->GetAllocator());
            m_Nro2.SetUp(BinAccessProtectedName, m_Allocator->GetAllocator());
            nn::ro::Initialize();
        }

        virtual void TearDown()
        {
            nn::ro::Finalize();
        }

        TestAllocator* m_Allocator;
        TestNro m_Nro1;
        TestNro m_Nro2;
    };

} // namespace

INSTANTIATE_TEST_CASE_P(ManualDll, ProtectedTest, ::testing::Values(nn::ro::BindFlag_Lazy, nn::ro::BindFlag_Now));

TEST_P(ProtectedTest, Bind)
{
    auto result = m_Nro1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_Nro2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    // Protected は外部参照が出来る
    {
        result = nn::ro::LookupSymbol(&addr, ProtectedFuncName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t funcAddr1 = addr;

        result = nn::ro::LookupSymbol(&addr, ProtectedValueName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t valAddr1 = addr;

        result = nn::ro::LookupSymbol(&addr, CallProtectedFuncName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t callAddr1 = addr;

        result = nn::ro::LookupSymbol(&addr, GetProtectedVariableAddressName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t getAddrFunc1 = addr;

        result = m_Nro1.FindSymbol(&addr, ProtectedFuncName);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(funcAddr1, addr);
        ReturnInt func1 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(0xf, func1());

        result = m_Nro1.FindSymbol(&addr, ProtectedValueName);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(valAddr1, addr);
        int* pVal1 = reinterpret_cast<int*>(addr);
        ASSERT_EQ(0xf, *pVal1);

        result = m_Nro1.FindSymbol(&addr, CallProtectedFuncName);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(callAddr1, addr);
        ReturnInt call1 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(0xf, call1());

        result = m_Nro1.FindSymbol(&addr, GetProtectedVariableAddressName);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(getAddrFunc1, addr);
        ReturnUintptr getAddr1 = reinterpret_cast<ReturnUintptr>(addr);
        ASSERT_EQ(valAddr1, getAddr1());

        result = m_Nro2.FindSymbol(&addr, CallProtectedFuncName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t callAddr2 = addr;
        ASSERT_NE(callAddr1, callAddr2);
        ReturnInt call2 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(call1(), call2());

        result = m_Nro2.FindSymbol(&addr, GetProtectedVariableAddressName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t getAddrFunc2 = addr;
        ASSERT_NE(getAddrFunc1, getAddrFunc2);
        ReturnUintptr getAddr2 = reinterpret_cast<ReturnUintptr>(addr);
        ASSERT_EQ(getAddr1(), getAddr2());
    }

    // 同じシンボルがあった場合、自分のシンボルを参照する
    {
        result = nn::ro::LookupSymbol(&addr, ProtectedFuncName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t funcAddr1 = addr;

        result = nn::ro::LookupSymbol(&addr, ProtectedValueName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t valAddr1 = addr;

        result = nn::ro::LookupSymbol(&addr, CallProtectedFuncName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t callAddr1 = addr;

        result = nn::ro::LookupSymbol(&addr, GetProtectedVariableAddressName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t getAddrFunc1 = addr;

        result = m_Nro1.FindSymbol(&addr, ProtectedFuncName2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(funcAddr1, addr);
        ReturnInt func1 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(0x1, func1());

        result = m_Nro1.FindSymbol(&addr, ProtectedValueName2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(valAddr1, addr);
        int* pVal1 = reinterpret_cast<int*>(addr);
        ASSERT_EQ(0x1, *pVal1);

        result = m_Nro1.FindSymbol(&addr, CallProtectedFuncName2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(callAddr1, addr);
        ReturnInt call1 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(0x1, call1());

        result = m_Nro1.FindSymbol(&addr, GetProtectedVariableAddressName2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(getAddrFunc1, addr);
        ReturnUintptr getAddr1 = reinterpret_cast<ReturnUintptr>(addr);
        ASSERT_EQ(valAddr1, getAddr1());

        result = m_Nro2.FindSymbol(&addr, ProtectedFuncName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t funcAddr2 = addr;
        ASSERT_NE(funcAddr1, funcAddr2);
        ReturnInt func2 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_NE(func1(), func2());

        result = m_Nro2.FindSymbol(&addr, ProtectedValueName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t valAddr2 = addr;
        ASSERT_NE(valAddr1, valAddr2);
        int* pVal2 = reinterpret_cast<int*>(addr);
        ASSERT_NE(*pVal1, *pVal2);

        result = m_Nro2.FindSymbol(&addr, CallProtectedFuncName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t callAddr2 = addr;
        ASSERT_NE(callAddr1, callAddr2);
        ReturnInt call2 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_NE(call1(), call2());

        result = m_Nro2.FindSymbol(&addr, GetProtectedVariableAddressName2);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t getAddrFunc2 = addr;
        ASSERT_NE(getAddrFunc1, getAddrFunc2);
        ReturnUintptr getAddr2 = reinterpret_cast<ReturnUintptr>(addr);
        ASSERT_NE(getAddr1(), getAddr2());
    }

    m_Nro2.Unload();
    m_Nro1.Unload();
} // NOLINT(impl/function_size)

