﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ro.h>
#include <nn/ro/ro_Types.h>

#include "util_Common.h"

#include <nn/nn_Log.h>


namespace {
    const char* BinDefineExternName = "nro/BinDefineExtern.nro";
    const char* BinAccessExternName = "nro/BinAccessExtern.nro";
    const char* BinDuplicateExternName = "nro/BinDuplicateExtern.nro";
    typedef int (*ReturnInt)();
    const char* ExternalFunctionName = "_Z16ExternalFunctionv";
    const char* CallExternalFunctionName = "_Z20CallExternalFunctionv";
    const char* AccessExternValueFunctionName = "_Z17AccessExternValuev";
    const char* ExternalValueName = "g_ExternValue";

    class ExternTest : public ::testing::TestWithParam<nn::ro::BindFlag>
    {
    protected:

        virtual void SetUp()
        {
            m_Allocator = &TestAllocator::GetInstance();
            m_Callee.SetUp(BinDefineExternName, m_Allocator->GetAllocator());
            m_Caller.SetUp(BinAccessExternName, m_Allocator->GetAllocator());
            m_Redefine.SetUp(BinDuplicateExternName, m_Allocator->GetAllocator());
            nn::ro::Initialize();
        }

        virtual void TearDown()
        {
            nn::ro::Finalize();
        }

        TestAllocator* m_Allocator;
        TestNro m_Callee;
        TestNro m_Caller;
        TestNro m_Redefine;
    };

    void CheckDllFunction(bool isFirst)
    {
        uintptr_t addr;

        auto result = nn::ro::LookupSymbol(&addr, ExternalFunctionName);
        ASSERT_RESULT_SUCCESS(result);

        ReturnInt func = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(func(), (isFirst ? 10 : 400));

        result = nn::ro::LookupSymbol(&addr, CallExternalFunctionName);
        ASSERT_RESULT_SUCCESS(result);

        func = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(func(), ((isFirst) ? 10 : 400));

        result = nn::ro::LookupSymbol(&addr, ExternalValueName);
        ASSERT_RESULT_SUCCESS(result);

        int* ptr = reinterpret_cast<int*>(addr);
        ASSERT_EQ(*ptr, ((isFirst) ? 10 : 20));
        *ptr = 30;
        ASSERT_EQ(func(), ((isFirst) ? 30 : 900));

        result = nn::ro::LookupSymbol(&addr, AccessExternValueFunctionName);
        ASSERT_RESULT_SUCCESS(result);
        ReturnInt func2 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(func2(), 30);
        *ptr = 10;
        ASSERT_EQ(func(), ((isFirst) ? 10 : 100));
        ASSERT_EQ(func2(), 10);
    }
} // namespace

INSTANTIATE_TEST_CASE_P(ManualDll, ExternTest, ::testing::Values(nn::ro::BindFlag_Lazy, nn::ro::BindFlag_Now));

// Global の関数 と変数を参照することが出来る
TEST_P(ExternTest, Bind)
{
    auto result = m_Callee.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_Caller.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_Redefine.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    CheckDllFunction(true);

    // 参照元がアンロードされると、参照がリセットされる
    m_Callee.Unload();

    CheckDllFunction(false);

    m_Caller.Unload();
    m_Redefine.Unload();
}

