﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

#include "RidCompatibilityVerifier_MenuUpdateTest.h"

namespace
{
    //! ダウンロード状況のラベル
    struct MenuUpdateStatePair
    {
        nn::rid::MenuUpdateState state;
        std::string stateStr;
    } MenuUpdateStateLabelList[] = {
        { nn::rid::MenuUpdateState::DoNothing, "DoNothing" },
        { nn::rid::MenuUpdateState::Downloading, "Downloading" },
        { nn::rid::MenuUpdateState::Completed, "Completed" },
        { nn::rid::MenuUpdateState::Failed, "Failed" },
        { nn::rid::MenuUpdateState::NeedNoUpdate, "NeedNoUpdate" },
    };
}


RidTestMenuUpdater::RidTestMenuUpdater() NN_NOEXCEPT
    : m_IsStartUpdate(false), m_IsMenuUpdateDone(false)
{
}

RidTestMenuUpdater::~RidTestMenuUpdater() NN_NOEXCEPT
{
    if (m_IsStartUpdate)
    {
        //! 実行スレッドの破棄
        nn::os::WaitThread(&m_UpdateExecutionThread);
        nn::os::DestroyThread(&m_UpdateExecutionThread);
    }
}

void RidTestMenuUpdater::Start() NN_NOEXCEPT
{
    static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_UpdateExecutionThread, [](void * p)
    {
        auto self = reinterpret_cast<RidTestMenuUpdater*>(p);
        NN_LOG("Call MenuUpdater.Execute() Start\n");
        self->m_MenuUpdateResult = self->m_MenuUpdater.Execute();
        NN_LOG("Call MenuUpdater.Execute() End\n");

        self->m_IsMenuUpdateDone = true;
    }
    , this, s_Stack, sizeof(s_Stack), nn::os::DefaultThreadPriority));

    // Menu更新処理の開始
    nn::os::StartThread(&m_UpdateExecutionThread);

    // Menu更新処理が着実に開始されるまで待つ
    while (NN_STATIC_CONDITION(true))
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        const auto state = m_MenuUpdater.GetState();
        if (state != nn::rid::MenuUpdateState::DoNothing)
        {
            break;
        }
    }

    m_IsStartUpdate = true;
}

std::string RidTestMenuUpdater::GetState() NN_NOEXCEPT
{
    const auto state = m_MenuUpdater.GetState();

    for (auto& musl : MenuUpdateStateLabelList)
    {
        if (musl.state == state)
        {
            return musl.stateStr;
        }
    }
    // ありえないはずだが該当する状態が無い場合は空文字を返しておく
    return "";
}
