﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

#include "RidCompatibilityVerifier_ApplicationUpdateTest.h"

namespace
{
    //! ダウンロード状況のラベル
    struct ApplicationUpdateStatePair
    {
        nn::rid::ApplicationUpdateProgress::State state;
        std::string stateStr;
    }  ApplicationUpdateStateLabelList[] = {
        { nn::rid::ApplicationUpdateProgress::State::DoNothing, "DoNothing" },
        { nn::rid::ApplicationUpdateProgress::State::Downloading, "Downloading" },
        { nn::rid::ApplicationUpdateProgress::State::Completed, "Completed" },
        { nn::rid::ApplicationUpdateProgress::State::Failed, "Failed" },
        { nn::rid::ApplicationUpdateProgress::State::Cancelled, "Cancelled" },
    };
}


RidTestApplicationUpdater::RidTestApplicationUpdater() NN_NOEXCEPT
    : m_IsStartUpdate(false), m_IsApplicationUpdateDone(false)
{
}

RidTestApplicationUpdater::~RidTestApplicationUpdater() NN_NOEXCEPT
{
    if (m_IsStartUpdate)
    {
        //! 実行スレッドの破棄
        nn::os::WaitThread(&m_UpdateExecutionThread);
        nn::os::DestroyThread(&m_UpdateExecutionThread);
    }
}

void RidTestApplicationUpdater::Start() NN_NOEXCEPT
{
    static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_UpdateExecutionThread, [](void * p)
    {
        auto self = reinterpret_cast<RidTestApplicationUpdater*>(p);

        // テスト用の固定値を設定する
        const int MaxAssetListLength = 32;
        std::unique_ptr<nn::rid::ApplicationAsset[]> pAssetList(new nn::rid::ApplicationAsset[MaxAssetListLength]);
        int assetListCount = 0;
        self->CreateTestApplicationAsset(pAssetList.get(), assetListCount);

        nn::rid::DownloadResult resultList[MaxAssetListLength + 1];
        int outResultCount = 0;
        // 不要な Asset 等は削除する設定にする
        bool isDelete = true;

        NN_LOG("Call ApplicationUpdater.Execute() Start\n");
        self->m_ApplicationUpdateResult = self->m_ApplicationUpdater.Execute(
            &outResultCount, resultList, MaxAssetListLength + 1, pAssetList.get(), assetListCount, isDelete);
        NN_LOG("Call ApplicationUpdater.Execute() End : outResultCount=%d\n", outResultCount);

        for (int i = 0; i < outResultCount; ++i)
        {
            auto& dResult = resultList[i];
            NN_LOG("  resultList[%d] : id=0x%016llx, result=0x%08x\n", i, dResult.applicationId.value, dResult.result.GetInnerValueForDebug());
        }

        self->m_IsApplicationUpdateDone = true;
    }
    , this, s_Stack, sizeof(s_Stack), nn::os::DefaultThreadPriority));

    // Application更新処理の開始
    nn::os::StartThread(&m_UpdateExecutionThread);

    // Application更新処理が着実に開始されるまで待つ
    while (NN_STATIC_CONDITION(true))
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        const auto progress = m_ApplicationUpdater.GetProgress();
        if (progress.state != nn::rid::ApplicationUpdateProgress::State::DoNothing)
        {
            break;
        }
    }

    m_IsStartUpdate = true;
}

std::string RidTestApplicationUpdater::GetState() NN_NOEXCEPT
{
    const auto progress = m_ApplicationUpdater.GetProgress();

    for (auto& ausl : ApplicationUpdateStateLabelList)
    {
        if (ausl.state == progress.state)
        {
            return ausl.stateStr;
        }
    }
    // ありえないはずだが該当する状態が無い場合は空文字を返しておく
    return "";
}

void RidTestApplicationUpdater::GetProgress(int64_t& outDone, int64_t& outTotal) NN_NOEXCEPT
{
    const auto progress = m_ApplicationUpdater.GetProgress();
    if (progress.state == nn::rid::ApplicationUpdateProgress::State::Downloading)
    {
        outDone = progress.done;
        outTotal = progress.total;
    }
}

void RidTestApplicationUpdater::CreateTestApplicationAsset(nn::rid::ApplicationAsset outAssetList[], int& outAssetListCount) NN_NOEXCEPT
{
    // テスト用の固定値定義
    outAssetList[0].index = 1;
    outAssetList[0].demoApplicationId = { 0x01001a500005e012 };
    outAssetList[0].aocApplicationId  = { 0x01001a500005e011 };

    outAssetList[1].index = 10;
    outAssetList[1].demoApplicationId = { nn::rid::InvalidApplicationId };
    outAssetList[1].aocApplicationId  = { 0x01001a500005e011 };

    outAssetListCount = 2;
}
