﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_ErrorResult.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <type_traits>
#include <cstdint>
#include <nn/nn_Abort.h>

#define NNT_ASSERT_DEATH_BY_UNHANDLED_RESULT(s) ASSERT_DEATH(s, "")

namespace {

    NN_DEFINE_ABSTRACT_ERROR_RANGE_RESULT(ResultAll, 10, 0, 300);
        NN_DEFINE_ABSTRACT_ERROR_RANGE_RESULT(ResultA, 10, 0, 100);
            NN_DEFINE_ERROR_RANGE_RESULT(ResultA1, 10, 1, 2);
            NN_DEFINE_ERROR_RANGE_RESULT(ResultA2, 10, 2, 3);
        NN_DEFINE_ERROR_RANGE_RESULT(ResultB, 10, 100, 200);
            NN_DEFINE_ERROR_RANGE_RESULT(ResultB1, 10, 101, 102);
        NN_DEFINE_ERROR_RANGE_RESULT(ResultC, 10, 200, 300);
    NN_DEFINE_ERROR_RANGE_RESULT(ResultOther, 20, 0, 500);

    NN_DEFINE_ERROR_RANGE_RESULT(ResultModuleMax, 511, 0, 1);
    NN_DEFINE_ERROR_RANGE_RESULT(ResultDescriptionMax, 1, 8191, 8192);

    template <typename ResultValue, typename InnerType>
    struct ResultValueTest
    {
        static_assert(std::is_same<typename ResultValue::InnerType, InnerType>::value, "");
        static_assert(std::is_convertible<ResultValue, nn::Result>::value, "");

        ResultValue value;
        bool isSuccess;
        int module;
        int description;
        InnerType innerValue;

        explicit ResultValueTest(const ResultValue& value) : value(value) {}

        void Test()
        {
            ASSERT_EQ(value.IsSuccess(), isSuccess);
            ASSERT_EQ(value.GetModule(), module);
            ASSERT_EQ(value.GetDescription(), description);
            ASSERT_EQ(value.GetInnerValueForDebug(), innerValue);
        }

        NN_DISALLOW_COPY(ResultValueTest);
    };

    template <typename ResultHolder>
    struct ResultHolderTest
    {
        static_assert(std::is_convertible<nn::Result, ResultHolder>::value, "");

        nn::Result result;
        bool canAccept;

        explicit ResultHolderTest(nn::Result result) : result(result) {}

        void Test()
        {
            ASSERT_EQ(ResultHolder::CanAccept(result), canAccept);
            if (!ResultHolder::CanAccept(result))
            {
                NNT_ASSERT_DEATH_BY_UNHANDLED_RESULT(static_cast<ResultHolder>(result));
            }
        }
    };

}

TEST(result, ConceptCheck_Result)
{
    {
        ResultValueTest<nn::Result, std::uint32_t> t((nn::ResultSuccess()));
        t.isSuccess = true;
        t.module = 0;
        t.description = 0;
        t.innerValue = 0;
        t.Test();
    }
    {
        ResultValueTest<nn::Result, std::uint32_t> t((ResultA1()));
        t.isSuccess = false;
        t.module = 10;
        t.description = 1;
        t.innerValue = ResultA1::InnerValue;
        t.Test();
    }
    {
        ResultHolderTest<nn::Result> t((nn::ResultSuccess()));
        t.canAccept = true;
        t.Test();
    }
    {
        ResultHolderTest<nn::Result> t((ResultA1()));
        t.canAccept = true;
        t.Test();
    }
}

TEST(result, ConceptCheck_ResultSuccess)
{
    {
        ResultValueTest<nn::ResultSuccess, std::uint32_t> t((nn::ResultSuccess()));
        t.isSuccess = true;
        t.module = 0;
        t.description = 0;
        t.innerValue = 0;
        t.Test();
    }
    {
        ResultHolderTest<nn::ResultSuccess> t((nn::ResultSuccess()));
        t.canAccept = true;
        t.Test();
    }
    {
        ResultHolderTest<nn::ResultSuccess> t((ResultA1()));
        t.canAccept = false;
        t.Test();
    }
}

TEST(result, ConceptCheck_ResultRange)
{
    {
        ResultValueTest<ResultA1, std::uint32_t> t((ResultA1()));
        t.isSuccess = false;
        t.module = 10;
        t.description = 1;
        t.innerValue = ResultA1::InnerValue;
        t.Test();
    }
}

// エラーレンジのテスト

#define NNT_RESULT_ASSERT_INCLUDES(type, result) ASSERT_TRUE(type::Includes(static_cast<nn::Result>(result)))
#define NNT_RESULT_ASSERT_NOT_INCLUDES(type, result) ASSERT_TRUE(!(type::Includes(static_cast<nn::Result>(result))))

TEST(result, ErrorRange)
{
    NNT_RESULT_ASSERT_INCLUDES(ResultAll, ResultA1());
    NNT_RESULT_ASSERT_INCLUDES(ResultAll, ResultA2());
    NNT_RESULT_ASSERT_INCLUDES(ResultAll, ResultB());
    NNT_RESULT_ASSERT_INCLUDES(ResultAll, ResultC());
    NNT_RESULT_ASSERT_INCLUDES(ResultA, ResultA1());
    NNT_RESULT_ASSERT_INCLUDES(ResultA, ResultA2());
    NNT_RESULT_ASSERT_INCLUDES(ResultA1, ResultA1());

    NNT_RESULT_ASSERT_NOT_INCLUDES(ResultAll, nn::ResultSuccess());
    NNT_RESULT_ASSERT_NOT_INCLUDES(ResultA1, nn::ResultSuccess());
    NNT_RESULT_ASSERT_NOT_INCLUDES(ResultOther, ResultB());
    NNT_RESULT_ASSERT_NOT_INCLUDES(ResultAll, ResultOther());
    NNT_RESULT_ASSERT_NOT_INCLUDES(ResultB1, ResultB());
}

// マクロのテスト

TEST(result, Macro_NN_RESULT_SUCCESS)
{
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_SUCCESS;
        };
        ASSERT_TRUE(f().IsSuccess());
    }
    {
        auto f = [] () -> nn::ResultSuccess
        {
            NN_RESULT_SUCCESS;
        };
        ASSERT_TRUE(f().IsSuccess());
    }
}

TEST(result, Macro_NN_RESULT_THROW)
{
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_THROW(ResultA1());
        };
        NNT_RESULT_ASSERT_INCLUDES(ResultA1, f());
    }
    {
        auto f = [] () -> nn::ResultSuccess
        {
            NN_RESULT_THROW(ResultA1());
        };
        NNT_ASSERT_DEATH_BY_UNHANDLED_RESULT(f());
    }
    {
        auto f = [] () -> int
        {
            NN_RESULT_THROW(ResultA1());
        };
        NNT_ASSERT_DEATH_BY_UNHANDLED_RESULT(f());
    }
}

TEST(result, Macro_NN_RESULT_THROW_UNLESS)
{
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_THROW_UNLESS(NN_STATIC_CONDITION(true), ResultA1());
            NN_RESULT_SUCCESS;
        };
        ASSERT_TRUE(f().IsSuccess());
    }
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_THROW_UNLESS(NN_STATIC_CONDITION(false), ResultA1());
            NN_RESULT_SUCCESS;
        };
        NNT_RESULT_ASSERT_INCLUDES(ResultA1, f());
    }
}

TEST(result, Macro_NN_RESULT_DO)
{
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_DO(nn::ResultSuccess());
            NN_RESULT_SUCCESS;
        };
        ASSERT_TRUE(f().IsSuccess());
    }
    {
        auto f = [] () -> nn::ResultSuccess
        {
            NN_RESULT_DO(nn::ResultSuccess());
            NN_RESULT_SUCCESS;
        };
        ASSERT_TRUE(f().IsSuccess());
    }
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_DO(ResultA1());
            NN_RESULT_SUCCESS;
        };
        NNT_RESULT_ASSERT_INCLUDES(ResultA1, f());
    }
    {
        auto f = [] () -> nn::ResultSuccess
        {
            NN_RESULT_DO(ResultA1());
            NN_RESULT_SUCCESS;
        };
        NNT_ASSERT_DEATH_BY_UNHANDLED_RESULT(f());
    }
    {
        auto f = [] () -> int
        {
            NN_RESULT_DO(ResultA1());
            return 0;
        };
        NNT_ASSERT_DEATH_BY_UNHANDLED_RESULT(f());
    }
    {
        auto f = [] () -> int
        {
            NN_RESULT_DO(nn::ResultSuccess());
            return 1;
        };
        ASSERT_EQ(f(), 1);
    }
}

TEST(result, Macro_NN_RESULT_TRY)
{
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_TRY(nn::ResultSuccess())
            NN_RESULT_END_TRY
            NN_RESULT_SUCCESS;
        };
        ASSERT_TRUE(f().IsSuccess());
    }
    {
        auto f = [] () -> int
        {
            NN_RESULT_TRY(nn::ResultSuccess())
            NN_RESULT_END_TRY
            return 1;
        };
        ASSERT_EQ(f(), 1);
    }
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_TRY(ResultA1())
            NN_RESULT_END_TRY
            NN_RESULT_SUCCESS;
        };
        NNT_RESULT_ASSERT_INCLUDES(ResultA1, f());
    }
    {
        auto f = [] () -> int
        {
            NN_RESULT_TRY(ResultA1())
            NN_RESULT_END_TRY
            return 0;
        };
        NNT_ASSERT_DEATH_BY_UNHANDLED_RESULT(f());
    }
}

TEST(result, Macro_NN_RESULT_CATCH)
{
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(nn::ResultSuccess())
                NN_RESULT_CATCH(ResultA)
                {
                    return false;
                }
            NN_RESULT_END_TRY
            return true;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH(ResultA) { return true; }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH(ResultA) { return true; }
                NN_RESULT_CATCH(ResultA1) { return false; }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH(ResultA2)
                {
                    return false;
                }
                NN_RESULT_CATCH(ResultA1)
                {
                    return true;
                }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH(ResultOther)
                {
                    return false;
                }
                NN_RESULT_CATCH(ResultA)
                {
                    return true;
                }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
}

TEST(result, Macro_NN_RESULT_CATCH_ALL)
{
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(nn::ResultSuccess())
                NN_RESULT_CATCH(ResultA)
                {
                    return false;
                }
                NN_RESULT_CATCH_ALL
                {
                    return false;
                }
            NN_RESULT_END_TRY
            return true;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH_ALL
                {
                    return true;
                }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH(ResultA)
                {
                    return true;
                }
                NN_RESULT_CATCH_ALL
                {
                    return false;
                }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH(ResultOther)
                {
                    return false;
                }
                NN_RESULT_CATCH_ALL
                {
                    return true;
                }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
    {
        auto f = [] () -> bool
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH_ALL
                {
                    return true;
                }
                NN_RESULT_CATCH(ResultA)
                {
                    return false;
                }
            NN_RESULT_END_TRY
            return false;
        };
        ASSERT_TRUE(f());
    }
}

TEST(result, Macro_NN_RESULT_RETHROW)
{
    {
        auto f = [] () -> nn::Result
        {
            NN_RESULT_TRY(ResultA1())
                NN_RESULT_CATCH(ResultA)
                {
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            NN_RESULT_SUCCESS;
        };
        NNT_RESULT_ASSERT_INCLUDES(ResultA1, f());
    }
}

TEST(result, Macro_NN_RESULT_TRY_Goto)
{
    {
        auto f = [] () -> bool
        {
            {
                NN_RESULT_TRY(ResultA1())
                    NN_RESULT_CATCH(ResultA)
                    {
                        goto l;
                    }
                NN_RESULT_END_TRY
                return false;
            }
        l:
            return true;
        };
        ASSERT_TRUE(f());
    }
}

TEST(result, Macro_NN_RESULT_TRY_LoopBreak)
{
    {
        auto f = [] () -> bool
        {
            while (NN_STATIC_CONDITION(true))
            {
                NN_RESULT_TRY(ResultA1())
                    NN_RESULT_CATCH(ResultA)
                    {
                        break;
                    }
                NN_RESULT_END_TRY
                return false;
            }
            return true;
        };
        ASSERT_TRUE(f());
    }
}

TEST(result, Macro_NN_RESULT_TRY_LoopContinue)
{
    {
        auto f = [] () -> bool
        {
            auto i = 0;
            while (NN_STATIC_CONDITION(true))
            {
                if (i == 0)
                {
                    NN_RESULT_TRY(ResultA1())
                        NN_RESULT_CATCH(ResultA)
                        {
                            i = 1;
                            continue;
                        }
                    NN_RESULT_END_TRY
                    return false;
                }
                else if (i == 1)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
            return false;
        };
        ASSERT_TRUE(f());
    }
}

TEST(result, Macro_NN_RESULT_TRY_SwitchBreak)
{
    {
        auto f = [] () -> bool
        {
            switch (0)
            {
                case 0:
                    NN_RESULT_TRY(ResultA1())
                        NN_RESULT_CATCH(ResultA)
                        {
                            break;
                        }
                    NN_RESULT_END_TRY
                    return false;
                default:
                    return false;
            }
            return true;
        };
        ASSERT_TRUE(f());
    }
}
