﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/pwm/pwm.h>

#include <nnt/gtest/gtest.h>

#include "../Common/testPwm_Helper.h"
#include "../TargetDevices/testPwm_Util-hardware.nx.h"

#include <nn/fan/fan.h>
#include <nn/fan/fan_ControlDev.h>
#include <nn/fan/impl/fan.h>

TEST( TargetSpecificNx, LcdBacklightUpDown )
{
    bool isPreviouslyEnabled = nnt::pwm::SetLcdBacklightEnabled(true);

    nn::pwm::Initialize();

    nn::pwm::ChannelSession session;
    nn::pwm::OpenSession(&session, nn::pwm::ChannelName::ChannelName_LcdBacklight);

    // 初期設定
    const nn::TimeSpan TargetPeriod = nn::TimeSpan::FromNanoSeconds((1 * 1000 * 1000 * 1000) / nnt::pwm::LcdBacklightFrequency);
    nn::pwm::SetPeriod(&session, TargetPeriod);
    nn::pwm::SetDuty(&session, 0);

    ASSERT_TRUE(nnt::pwm::IsTimeSpanNear(nn::pwm::GetPeriod(&session), TargetPeriod, nnt::pwm::DefaultAllowableErrorPercent));
    ASSERT_EQ(nn::pwm::GetDuty(&session), 0);

    nn::pwm::SetEnabled(&session, true);

    const nn::TimeSpan Interval = nn::TimeSpan::FromMilliSeconds(500);
    const int DutyTable[] = {
        0, 25, 50, 75, 100, 125, 150, 175, 200, 225, 250,
        225, 200, 175, 150, 125, 100, 75, 50, 25
        };
    for (auto duty : DutyTable)
    {
        nn::pwm::SetDuty(&session, duty);
        ASSERT_EQ(nn::pwm::GetDuty(&session), duty);

        nn::os::SleepThread(Interval);
    }

    nn::pwm::CloseSession(&session);

    nn::pwm::Finalize();

    nnt::pwm::SetLcdBacklightEnabled(isPreviouslyEnabled);
}

TEST( TargetSpecificNx, FanUpDown )
{
    bool isPreviouslyEnabled = nnt::pwm::SetFanEnabled(true);

    nn::pwm::Initialize();

    nn::pwm::ChannelSession session;
    nn::pwm::OpenSession(&session, nn::pwm::ChannelName::ChannelName_CpuFan);

    // 初期設定
    const nn::TimeSpan TargetPeriod = nn::TimeSpan::FromNanoSeconds((1 * 1000 * 1000 * 1000) / nnt::pwm::FanFrequency);
    nn::pwm::SetPeriod(&session, TargetPeriod);
    nn::pwm::SetDuty(&session, 0);

    ASSERT_TRUE(nnt::pwm::IsTimeSpanNear(nn::pwm::GetPeriod(&session), TargetPeriod, nnt::pwm::DefaultAllowableErrorPercent));
    ASSERT_EQ(nn::pwm::GetDuty(&session), 0);

    nn::pwm::SetEnabled(&session, true);

    const nn::TimeSpan Interval = nn::TimeSpan::FromMilliSeconds(500);
    const int DutyTable[] = {
        0, 25, 50, 75, 100, 125, 150, 175, 200, 225, 250,
        225, 200, 175, 150, 125, 100, 75, 50, 25
        };
    for (auto duty : DutyTable)
    {
        nn::pwm::SetDuty(&session, duty);
        ASSERT_EQ(nn::pwm::GetDuty(&session), duty);

        nn::os::SleepThread(Interval);
    }

    nn::pwm::CloseSession(&session);

    nn::pwm::Finalize();

    nnt::pwm::SetFanEnabled(isPreviouslyEnabled);
}

TEST( TargetSpecificNx, Led )
{
    nn::pwm::Initialize();

    nn::pwm::ChannelSession session;
    nn::pwm::OpenSession(&session, nn::pwm::ChannelName::ChannelName_BlinkLed);

    nn::pwm::SetEnabled(&session, true);

    const nn::TimeSpan Interval = nn::TimeSpan::FromMilliSeconds(500);

    const nn::TimeSpan PeriodTable[] = {
        nn::TimeSpan::FromMilliSeconds(100),
        nn::TimeSpan::FromMilliSeconds(10),
    };

    const int DutyTable[] = {0, 25, 50, 75, 100};

    for (auto period : PeriodTable)
    {
        nn::pwm::SetPeriod(&session, period);
        ASSERT_EQ(nn::pwm::GetPeriod(&session), period);

        for (auto duty : DutyTable)
        {
            nn::pwm::SetDuty(&session, duty);
            ASSERT_EQ(nn::pwm::GetDuty(&session), duty);

            nn::os::SleepThread(Interval);
        }
    }

    nn::pwm::CloseSession(&session);

    nn::pwm::Finalize();
}
