﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nnt/gtest/gtest.h>

#include <nn/fan/fan.h>
#include <nn/fan/fan_ControlDev.h>
#include <nn/fan/impl/fan.h>

#include <nn/pwm/pwm.h>
#include <nn/pwm/pwm_ChannelDev.h>

#include "../Common/testPwm_Helper.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include "../TargetDevices/testPwm_Util-hardware.nx.h"
#endif

namespace {

#if   defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)

// テスト対象のチャンネル番号
const int TargetChannelTable[] = {0, 1, 2, 3};

// 対象ターゲットボードで精度要求を満たすべきターゲット周波数
const nn::TimeSpan TargetPeriodTable[] = {
    nn::TimeSpan::FromMilliSeconds(10),     // 100Hz 程度
    nn::TimeSpan::FromMilliSeconds(1),      //  1kHz 程度
    nn::TimeSpan::FromNanoSeconds(200000),  //  5kHz 程度
    nn::TimeSpan::FromNanoSeconds(100000),  // 10kHz 程度
    nn::TimeSpan::FromNanoSeconds(50000),   // 20kHz 程度
    nn::TimeSpan::FromNanoSeconds(40000),   // 25kHz 程度
    nn::TimeSpan::FromNanoSeconds(33333),   // 30kHz 程度
    nn::TimeSpan::FromNanoSeconds(28571),   // 35kHz 程度
};

#elif defined(NN_BUILD_CONFIG_HARDWARE_NX)

// テスト対象のチャンネル番号
const int TargetChannelTable[] = {0, 1, 2, 3};

// 対象ターゲットボードで精度要求を満たすべきターゲット周波数
const nn::TimeSpan TargetPeriodTable[] = {
    nn::TimeSpan::FromMilliSeconds(10),     // 100Hz 程度
    nn::TimeSpan::FromMilliSeconds(1),      //  1kHz 程度
    nn::TimeSpan::FromNanoSeconds(200000),  //  5kHz 程度
    nn::TimeSpan::FromNanoSeconds(100000),  // 10kHz 程度
    nn::TimeSpan::FromNanoSeconds(50000),   // 20kHz 程度
    nn::TimeSpan::FromNanoSeconds(40000),   // 25kHz 程度
    nn::TimeSpan::FromNanoSeconds(33333),   // 30kHz 程度
    nn::TimeSpan::FromNanoSeconds(28571),   // 35kHz 程度
};

#else
    #error NN_BUILD_CONFIG_HARDWARE_ not selected or supported
#endif

const int MaxChannelCount = sizeof(TargetChannelTable) / sizeof(TargetChannelTable[0]);

// 非同期操作を伴うテスト用のスレッドリソース
const size_t SubThreadStackSize = 1 * 1024 * 1024;
const int SubThreadCount = MaxChannelCount;
NN_ALIGNAS(4096) char s_SubThreadStack[SubThreadCount][SubThreadStackSize];
nn::os::ThreadType s_SubThread[SubThreadCount];

// テスト全体の終了フラグ
bool s_IsRunning = true;

void StartSubThread(nn::os::ThreadType* pThread, char* pStack, size_t stackSize, nn::os::ThreadFunction function, void *arg, int priority)
{
    nn::Result result = nn::os::CreateThread( pThread, function, arg, pStack, stackSize, priority );
    ASSERT_TRUE(result.IsSuccess());
    nn::os::StartThread( pThread );
}

void WaitAndFinishSubThread(nn::os::ThreadType* pThread)
{
    nn::os::WaitThread( pThread );
    nn::os::DestroyThread( pThread );
}

void ConcurrentTestMain(void* arg)
{
    const int channelIndex = *(reinterpret_cast<int*>(arg));

    // スレッドごとに少しばらつかせる
    const nn::TimeSpan Interval = nn::TimeSpan::FromMilliSeconds(50 + 10 * channelIndex);

    const int DutyTable[] = {
        0, 30, 60, 100,
    };

    while (s_IsRunning)
    {
        nn::pwm::ChannelSession session;
        nn::pwm::OpenSessionForDev(&session, channelIndex);

        // 適当な初期値
        nn::pwm::SetEnabled(&session, false);
        nn::pwm::SetPeriod(&session, nn::TimeSpan::FromMicroSeconds(100));
        nn::pwm::SetDuty(&session, 0);

        for (auto period : TargetPeriodTable)
        {
            nn::pwm::SetPeriod(&session, period);
            for (auto duty : DutyTable)
            {
                nn::pwm::SetEnabled(&session, true);
                ASSERT_TRUE(nn::pwm::GetEnabled(&session));

                nn::os::SleepThread(Interval);

                nn::pwm::SetDuty(&session, duty);
                ASSERT_EQ(nn::pwm::GetDuty(&session), duty);

                nn::pwm::SetEnabled(&session, false);
                ASSERT_FALSE(nn::pwm::GetEnabled(&session));

                nn::os::SleepThread(Interval);
            }
            ASSERT_TRUE(nnt::pwm::IsTimeSpanNear(nn::pwm::GetPeriod(&session), period, nnt::pwm::DefaultAllowableErrorPercent));
        }

        nn::pwm::CloseSession(&session);
    }
}

}

TEST( Basic, Concurrent )
{
    nn::pwm::Initialize();

    for (int threadIndex=0; threadIndex < SubThreadCount; ++threadIndex)
    {
        StartSubThread(&s_SubThread[threadIndex], s_SubThreadStack[threadIndex], SubThreadStackSize,
                        ConcurrentTestMain,
                        reinterpret_cast<void*>(const_cast<int*>(&TargetChannelTable[threadIndex])),
                        nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()) - 1);
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    s_IsRunning = false; // 終了をシグナル
    for (int threadIndex=0; threadIndex < SubThreadCount; ++threadIndex)
    {
        WaitAndFinishSubThread(&s_SubThread[threadIndex]);
    }

    nn::pwm::Finalize();
}

TEST( Basic, PeriodAccuracy )
{
    nn::pwm::Initialize();

    nn::pwm::ChannelSession session;
    nn::pwm::OpenSessionForDev(&session, 0);

    // 適当な初期値
    nn::pwm::SetEnabled(&session, false);
    nn::pwm::SetPeriod(&session, nn::TimeSpan::FromMicroSeconds(100));
    nn::pwm::SetDuty(&session, 0);

    for (auto period : TargetPeriodTable)
    {
        nn::pwm::SetPeriod(&session, period);
        ASSERT_TRUE(nnt::pwm::IsTimeSpanNear(nn::pwm::GetPeriod(&session), period, nnt::pwm::DefaultAllowableErrorPercent));
    }

    nn::pwm::CloseSession(&session);
    nn::pwm::Finalize();
}
