﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Assert.h>

#include <nn/os.h>

#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/usb/pd/usb_Pd.h>
#include <nn/util/util_IntrusiveList.h>

namespace {

nn::os::SystemEventType g_SystemEvent;

// Psm が見ている Error, Pdo, Rdo の値のみ考慮すればいい。
nn::usb::pd::Status g_Status;

bool g_StubInitialized = false;
bool g_PowerRequestNoticeEnabled = true;

struct StaticMutex
{
    ::nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&mutex);
    }
};

StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

// StandardAllocator を使用するためには 4KB アライン、16KB 以上のバッファが必要とのこと。
NN_ALIGNAS(0x1000) uint8_t g_BufferForAllocator[0x40000];
nn::mem::StandardAllocator g_StandardAllocator;

class SessionContext : public nn::util::IntrusiveListBaseNode<SessionContext>
{
public:
    SessionContext() NN_NOEXCEPT
    : m_Notice()
    {
    }

public:
    nn::usb::pd::Notice m_Notice;
};

nn::util::IntrusiveList<SessionContext, nn::util::IntrusiveListBaseNodeTraits<SessionContext>> g_SessionContextList;

SessionContext* FindSessionContextInList(SessionContext* pSessionContext) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( &(*itr) == pSessionContext )
        {
            return &(*itr);
        }
    }

    return nullptr;
}

const size_t MaxSessions = 1;

SessionContext* AddSessionContextToList() NN_NOEXCEPT
{
    NN_ASSERT(g_SessionContextList.size() < MaxSessions);

    SessionContext* pSessionContext = static_cast<SessionContext*>(g_StandardAllocator.Allocate(sizeof(SessionContext)));
    new (pSessionContext) SessionContext();

    (pSessionContext->m_Notice).Clear();

    g_SessionContextList.push_front(*pSessionContext);

    return pSessionContext;
}

bool RemoveSessionContextFromList(SessionContext* pSessionContext) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( &(*itr) == pSessionContext )
        {
            // delete で解放しないので明示的にデストラクタを呼ぶ。
            pSessionContext->~SessionContext();
            g_SessionContextList.erase(itr);
            g_StandardAllocator.Free(pSessionContext);
            return true;
        }
    }

    return false;
}

void SetNoticeAll(::nn::usb::pd::Notice notice) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        (*itr).m_Notice.SetMaskedBits(0xff, notice.GetMaskedBits(0xff) | (*itr).m_Notice.GetMaskedBits(0xff));
    }
}

SessionContext* FindSessionContextInList(nn::usb::pd::Session* pSession) NN_NOEXCEPT
{
    return FindSessionContextInList(static_cast<SessionContext*>(pSession->_handle));
}

void AddSessionContextToList(nn::usb::pd::Session* pOutSession) NN_NOEXCEPT
{
    pOutSession->_handle = static_cast<void*>(AddSessionContextToList());
    pOutSession->_pEvent = nullptr;

    NN_ASSERT_NOT_NULL(pOutSession->_handle);
}

void RemoveSessionContextFromList(nn::usb::pd::Session* pSession) NN_NOEXCEPT
{
    NN_ASSERT(RemoveSessionContextFromList(static_cast<SessionContext*>(pSession->_handle)));
}

void BindEvent(nn::os::SystemEventType* pEvent, nn::usb::pd::Session* pSession) NN_NOEXCEPT
{
    SessionContext* pSessionContext = FindSessionContextInList(pSession);
    NN_ASSERT_NOT_NULL(pSessionContext);

    NN_ASSERT(pSession->_pEvent == nullptr);
    NN_ASSERT_NOT_NULL(pEvent);

    nn::os::NativeHandle handle = nn::os::GetReadableHandleOfSystemEvent(&g_SystemEvent);
    nn::os::AttachReadableHandleToSystemEvent(pEvent, handle, false, nn::os::EventClearMode_ManualClear);
    pSession->_pEvent = pEvent;
}

void UnbindEvent(nn::usb::pd::Session* pSession) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(FindSessionContextInList(pSession));
    NN_ASSERT_NOT_NULL(pSession->_pEvent);

    nn::os::DestroySystemEvent(pSession->_pEvent);
    pSession->_pEvent = nullptr;
}

} // namespace

namespace nn { namespace usb { namespace pd {

void Initialize() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(!g_StubInitialized);
    NN_ASSERT(g_SessionContextList.empty());

    g_StubInitialized = true;
    g_StandardAllocator.Initialize(g_BufferForAllocator, 0x40000);
    nn::os::CreateSystemEvent(&g_SystemEvent, nn::os::EventClearMode_ManualClear, true);
    g_Status.Clear();
}

void Finalize() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT(g_SessionContextList.empty());

    g_StubInitialized = false;
    g_StandardAllocator.Finalize();
    nn::os::DestroySystemEvent(&g_SystemEvent);
}

void OpenSession(Session* pOutSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pOutSession);

    AddSessionContextToList(pOutSession);
}

void CloseSession(Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    RemoveSessionContextFromList(pSession);
}

Result BindNoticeEvent(nn::os::SystemEventType* pEvent, Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);
    NN_ASSERT_NOT_NULL(pEvent);

    BindEvent(pEvent, pSession);

    NN_RESULT_SUCCESS;
}

void UnbindNoticeEvent(Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    UnbindEvent(pSession);
}

Result GetStatus(Status* pStatus, Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);
    NN_ASSERT_NOT_NULL(FindSessionContextInList(pSession));

    NN_ASSERT_NOT_NULL(pStatus);
    *pStatus = g_Status;

    NN_RESULT_SUCCESS;
}

Result GetNotice(Notice* pNotice, Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    NN_ASSERT_NOT_NULL(pNotice);

    SessionContext* pSessionContext = FindSessionContextInList(pSession);

    NN_ASSERT_NOT_NULL(pSessionContext);

    *pNotice = pSessionContext->m_Notice;
    (pSessionContext->m_Notice).Clear();

    NN_RESULT_SUCCESS;
}

Result EnablePowerRequestNotice(Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);
    NN_ASSERT_NOT_NULL(FindSessionContextInList(pSession));

    g_PowerRequestNoticeEnabled = true;

    NN_RESULT_SUCCESS;
}

Result DisablePowerRequestNotice(Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);
    NN_ASSERT_NOT_NULL(FindSessionContextInList(pSession));

    g_PowerRequestNoticeEnabled = false;

    NN_RESULT_SUCCESS;
}

// 値はスルーする。
Result ReplyPowerRequest(Session* pSession, bool isSuccess) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);
    NN_ASSERT_NOT_NULL(FindSessionContextInList(pSession));

    NN_UNUSED(isSuccess);

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::usb::pd

namespace nnt { namespace usb { namespace pd {

namespace {

//! ステータスの差分があるとき通知を送ります。
bool SendNotice(::nn::usb::pd::Status& newStatus, ::nn::usb::pd::Status& oldStatus) NN_NOEXCEPT
{
    ::nn::usb::pd::Notice notice;

    notice.Set<::nn::usb::pd::Notice::ActiveNotice>(newStatus.IsActive() != oldStatus.IsActive());
    notice.Set<::nn::usb::pd::Notice::ErrorNotice>(newStatus.GetError() != ::nn::usb::pd::StatusError_None);
    notice.Set<::nn::usb::pd::Notice::DataRoleNotice>(newStatus.GetDataRole() != oldStatus.GetDataRole());
    notice.Set<::nn::usb::pd::Notice::PowerRoleNotice>(newStatus.GetPowerRole() != oldStatus.GetPowerRole());

    // DeviceNotice : 未実装
    notice.Set<::nn::usb::pd::Notice::DeviceNotice>(false);

    // ConsumerContractNotice : 未実装
    notice.Set<::nn::usb::pd::Notice::ConsumerContractNotice>(false);

    // ProviderContractNotice : 未実装
    notice.Set<::nn::usb::pd::Notice::ProviderContractNotice>(false);

    notice.Set<::nn::usb::pd::Notice::RequestNotice>(newStatus.GetRequest() != oldStatus.GetRequest());

    if ( !g_PowerRequestNoticeEnabled )
    {
        // TBD: 取り消した RequestNotice は後で追加されるべきか。
        notice.Set<::nn::usb::pd::Notice::RequestNotice>(0);
    }

    if ( notice.IsAnyBitOn(0xff) )
    {
        SetNoticeAll(notice);
        nn::os::SignalSystemEvent(&g_SystemEvent);

        return true;
    }

    return false;
}

} // namespace

bool SetStatus(::nn::usb::pd::Status status) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);

    ::nn::usb::pd::Status oldStatus = g_Status;
    g_Status = status;

    return SendNotice(g_Status, oldStatus);
}

}}} // namespace nnt::usb::pd
