﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/ovln/format/ovln_PowerManagementMessage.h>
#include <nn/psm/psm_System.h>

#include <nnt/nntest.h>

#include "testPsm_Common.h"
#include "testPsm_OvlnStub.h"

namespace nnt { namespace psm {

namespace {

struct PsmChargerStateChangeDataPolicyTestSetting
{
    //! USB 電源関連の状態
    ::nnt::psm::UsbPowerState usbPowerState;

    //! 給電能力
    ::nn::psm::ChargerType chargerType;
};

const PsmChargerStateChangeDataPolicyTestSetting PsmChargerStateChangeDataPolicyTestSettings[] =
{
    { ::nnt::psm::UsbPowerState::He15V,    ::nn::psm::ChargerType_EnoughPower },
    { ::nnt::psm::UsbPowerState::Sdp500Ma, ::nn::psm::ChargerType_LowPower    },
};

class PsmChargerStateChangeDataPolicyTest : public ::testing::TestWithParam<PsmChargerStateChangeDataPolicyTestSetting>
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

INSTANTIATE_TEST_CASE_P(PsmChargerStateChangeDataPolicyTestName, PsmChargerStateChangeDataPolicyTest, ::testing::ValuesIn(PsmChargerStateChangeDataPolicyTestSettings));

class PsmOverlayNotificationPolicyTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

} // namespace

// 1. FullAwake にします。
// 2. ChargerType を未接続にします。
// 3. ChargerType を十分な給電にします。
// 4. メッセージを確認します。

TEST_P(PsmChargerStateChangeDataPolicyTest, ChargerType)
{
    ChangeUsbPowerState(UsbPowerState::Inactive);

    ChangeUsbPowerState(GetParam().usbPowerState);

    ::nn::ovln::Message message = ::nnt::ovln::GetLastMessage();

    EXPECT_EQ(::nn::ovln::format::PowerManagementChargerStateChangeDataTag, message.tag);
    EXPECT_EQ(sizeof(::nn::ovln::format::BatteryStateChangeData), message.dataSize);

    ::nn::ovln::format::BatteryStateChangeData data;
    ::std::memcpy(&data, &(message.data), sizeof(::nn::ovln::format::BatteryStateChangeData));

    EXPECT_LE(1, data.batteryChargePercentage);
    EXPECT_GE(100, data.batteryChargePercentage);
    EXPECT_EQ(GetParam().chargerType , data.chargerType);

    ChangeUsbPowerState(UsbPowerState::Inactive);

    message = ::nnt::ovln::GetLastMessage();

    EXPECT_EQ(::nn::ovln::format::PowerManagementChargerStateChangeDataTag, message.tag);
    EXPECT_EQ(sizeof(::nn::ovln::format::BatteryStateChangeData), message.dataSize);

    ::std::memcpy(&data, &(message.data), sizeof(::nn::ovln::format::BatteryStateChangeData));

    EXPECT_LE(1, data.batteryChargePercentage);
    EXPECT_GE(100, data.batteryChargePercentage);
    EXPECT_EQ(::nn::psm::ChargerType_Unconnected , data.chargerType);
}

// 1. FullAwake にします。
// 2. 電池残量値を x% より大きくします。
// 3. 電池残量値を x% 以下にします。
// 4. メッセージを確認します。
// 2 ~ 4 の繰り返しです。

TEST_F(PsmOverlayNotificationPolicyTest, BatteryChargeLevel)
{
    ChangeBatteryChargePercentage(100.0);

    struct BatteryChargePercentagePair
    {
        double raw;
        int exposed;
    };

    const BatteryChargePercentagePair BatteryChargePercentageList[] =
    {
        { 17.0, 15 },
        { 7.0,  5  },
        { 0.1,  1  },
    };

    for ( const BatteryChargePercentagePair& batteryChargePercentagePair : BatteryChargePercentageList )
    {
        ChangeBatteryChargePercentage(batteryChargePercentagePair.raw);

        ::nn::ovln::Message message = ::nnt::ovln::GetLastMessage();

        EXPECT_EQ(::nn::ovln::format::PowerManagementLowBatteryDataTag, message.tag);
        EXPECT_EQ(sizeof(::nn::ovln::format::BatteryStateChangeData), message.dataSize);

        ::nn::ovln::format::BatteryStateChangeData data;
        ::std::memcpy(&data, &(message.data), sizeof(::nn::ovln::format::BatteryStateChangeData));

        EXPECT_EQ(batteryChargePercentagePair.exposed, data.batteryChargePercentage);
        EXPECT_EQ(::nn::psm::ChargerType_Unconnected , data.chargerType);
    }

    ChangeBatteryChargePercentage(100.0);
}

}} // namespace nnt::psm
