﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nnt/nntest.h>

#include <nn/psc/psc_Types.h>
#include <nn/psm/psm_SystemProcessApi.h>

#include "testPsm_ChargerDriverStub.h"
#include "testPsm_Common.h"

namespace nnt { namespace psm {

namespace {

struct PsmOtgPolicyTestSetting
{
    //! システム電源状態
    ::nn::psc::PmState pmState;

    //! 給電エミュレーションの有効化
    bool isEnoughPowerChargeEmulationEnabled;

    //! 出力の成功・失敗の期待値
    bool outputSuccess;
};

const PsmOtgPolicyTestSetting PsmOtgPolicyTestSettings[] =
{
    { ::nn::psc::PmState_MinimumAwake, false, true  },
    { ::nn::psc::PmState_MinimumAwake, true,  false },
    { ::nn::psc::PmState_SleepReady,   false, false },
};

class PsmOtgPolicyTest : public ::testing::TestWithParam<PsmOtgPolicyTestSetting>
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

INSTANTIATE_TEST_CASE_P(PsmOtgPolicyTestName, PsmOtgPolicyTest, ::testing::ValuesIn(PsmOtgPolicyTestSettings));

} // namespace

// 1. MinimumAwake (or FullAwake) にします。
// 2. 対向が何も刺さっていない状態にします。
// 3. 外部への出力を開始するリクエストを送ります。
// 4. 外部への出力完了通知を送ります。
// 5. 外部への出力を終了するリクエストを送ります。
//
// 2 ~ 5 の各状態で CHG_CFG, WatchDog, OutputCurrentLimit の期待値チェックをします。

// FUTURE WORK: チャージャーのエラーをテストに追加します。

TEST_P(PsmOtgPolicyTest, DisableOutputFirst)
{
    TransitPmState(::nn::psc::PmState_MinimumAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);
    ChangeEnoughPowerChargeEmulation(false);

    EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());

    // 非 OTG 時は 500 mA を保証しても良いと思いますが現在は保証されていません。
    EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());

    // ChangeUsbPowerState の完了割込みが返せないので pmState はテストしません。
    if ( GetParam().pmState != ::nn::psc::PmState_MinimumAwake )
    {
        return;
    }

    if ( GetParam().isEnoughPowerChargeEmulationEnabled )
    {
        ChangeEnoughPowerChargeEmulation(true);
    }

    ChangeUsbPowerState(UsbPowerState::StartOutput);

    if ( GetParam().outputSuccess )
    {
        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(1300, driver::detail::GetOutputCurrentLimitMilliAmpere());
        EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

        ChangeUsbPowerState(UsbPowerState::ActiveOutput);

        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(500, driver::detail::GetOutputCurrentLimitMilliAmpere());
        EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

        ChangeUsbPowerState(UsbPowerState::EndOutput);

        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());
    }
    else
    {
        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());
    }

    TransitPmState(::nn::psc::PmState_FullAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);
    ChangeEnoughPowerChargeEmulation(false);
}

TEST_P(PsmOtgPolicyTest, DisableOutputAfterStart)
{
    TransitPmState(::nn::psc::PmState_MinimumAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);
    ChangeEnoughPowerChargeEmulation(false);

    EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
    EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());

    ChangeUsbPowerState(UsbPowerState::StartOutput);

    EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
    EXPECT_EQ(1300, driver::detail::GetOutputCurrentLimitMilliAmpere());
    EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

    if ( GetParam().isEnoughPowerChargeEmulationEnabled )
    {
        ChangeEnoughPowerChargeEmulation(true);
    }

    TransitPmState(GetParam().pmState);

    if ( GetParam().outputSuccess )
    {
        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(1300, driver::detail::GetOutputCurrentLimitMilliAmpere());
        EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

        ChangeUsbPowerState(UsbPowerState::ActiveOutput);

        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(500, driver::detail::GetOutputCurrentLimitMilliAmpere());
        EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

        ChangeUsbPowerState(UsbPowerState::EndOutput);

        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());
    }
    else
    {
        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());
    }

    TransitPmState(::nn::psc::PmState_FullAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);
    ChangeEnoughPowerChargeEmulation(false);
}

TEST_P(PsmOtgPolicyTest, DisableOutputAfterActive)
{
    TransitPmState(::nn::psc::PmState_MinimumAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);
    ChangeEnoughPowerChargeEmulation(false);

    EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
    EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());

    ChangeUsbPowerState(UsbPowerState::StartOutput);

    EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
    EXPECT_EQ(1300, driver::detail::GetOutputCurrentLimitMilliAmpere());
    EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

    ChangeUsbPowerState(UsbPowerState::ActiveOutput);

    EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
    EXPECT_EQ(500, driver::detail::GetOutputCurrentLimitMilliAmpere());
    EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

    if ( GetParam().isEnoughPowerChargeEmulationEnabled )
    {
        ChangeEnoughPowerChargeEmulation(true);
    }

    TransitPmState(GetParam().pmState);

    if ( GetParam().outputSuccess )
    {
        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeOTG, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(500, driver::detail::GetOutputCurrentLimitMilliAmpere());
        EXPECT_EQ(true, driver::detail::IsWatchdogTimerEnabled());

        ChangeUsbPowerState(UsbPowerState::EndOutput);

        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());
    }
    else
    {
        EXPECT_EQ(driver::detail::ChargeConfiguration_ChargeBattery, driver::detail::GetChargeConfiguration());
        EXPECT_EQ(false, driver::detail::IsWatchdogTimerEnabled());
    }

    TransitPmState(::nn::psc::PmState_FullAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);
    ChangeEnoughPowerChargeEmulation(false);
}

}} // namespace nnt::psm
