﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/psm/driver/detail/psm_Constants.h>
#include <nn/psm/driver/psm.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_Manufacture.h>

#include <nnt/nntest.h>

#include "testPsm_Common.h"
#include "testPsm_GpioStub.h"

namespace nnt { namespace psm {

namespace {

struct PsmNcePolicyTestSetting
{
    //! 電池温度（電池電圧とセットで一つの因子）
    double batteryTemperatureCelsius;

    //! 電池電圧（電池温度とセットで一つの因子）
    int batteryVoltageMilliVolt;

    //! システム電源状態（電圧ベース電池残量、平均充電電流とセットで一つの因子）
    ::nn::psc::PmState systemPowerState;

    //! 電圧ベース電池残量（システム電源状態、平均充電電流とセットで一つの因子）
    double voltageFuelGaugePercentage;

    //! 平均充電電流（システム電源状態、電圧ベース電池残量とセットで一つの因子）
    int averageCurrentMilliAmpere;

    //! 充電可否
    bool isBatteryChargeEnabled;

    //! nCE 端子の状態の期待値。（High で充電禁止）
    ::nn::gpio::GpioValue nce;
};

const PsmNcePolicyTestSetting PsmNcePolicyTestSettings[] =
{
    // 電池温度、電池電圧の確認
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooLow  - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_High },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooLow,        4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdLow,           4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh,          4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooHigh - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooHigh,       4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_High },

    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooLow  - 0.1, 4050, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_High },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooLow,        4050, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdLow,           4050, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4050, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh,          4050, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_High },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooHigh - 0.1, 4050, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_High },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdTooHigh,       4050, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_High },

    // システム電源状態、電圧ベース電池残量、平均充電電流の確認
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 98.9, 128, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 99.0, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 99.0, 128, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_FullAwake,    98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_FullAwake,    98.9, 128, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_FullAwake,    99.0, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_FullAwake,    99.0, 128, true,  ::nn::gpio::GpioValue_High },

    // 充電可否
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, true,  ::nn::gpio::GpioValue_Low  },
    { ::nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdHigh    - 0.1, 4049, ::nn::psc::PmState_MinimumAwake, 98.9, 129, false, ::nn::gpio::GpioValue_High },
};

class PsmNcePolicyTest : public ::testing::TestWithParam<PsmNcePolicyTestSetting>
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

INSTANTIATE_TEST_CASE_P(PsmNcePolicyTestName, PsmNcePolicyTest, ::testing::ValuesIn(PsmNcePolicyTestSettings));

} // namespace

TEST_P(PsmNcePolicyTest, CheckChargeSettings)
{
    if ( GetParam().systemPowerState != ::nn::psc::PmState_FullAwake
        && GetParam().systemPowerState != ::nn::psc::PmState_MinimumAwake )
    {
        TransitPmState(::nn::psc::PmState_MinimumAwake);
    }

    ChangeBatteryTemperature(GetParam().batteryTemperatureCelsius);
    ChangeBatteryVoltageMilliVolt(GetParam().batteryVoltageMilliVolt);
    ChangeVoltageFuelGaugePercentage(GetParam().voltageFuelGaugePercentage);
    ChangeAverageCurrentMilliAmpere(GetParam().averageCurrentMilliAmpere);

    if ( GetParam().isBatteryChargeEnabled )
    {
        ::nn::psm::EnableBatteryCharging();
    }
    else
    {
        ::nn::psm::DisableBatteryCharging();
    }

    TransitPmState(GetParam().systemPowerState);

    EXPECT_TRUE(::nnt::gpio::CheckSessionContext(::nn::gpio::GpioPadName_BattChgEnableN, ::nn::gpio::Direction_Output, GetParam().nce));
}

}} // namespace nnt::psm
