﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <vector>

#include <nn/nn_Common.h>

#include <nn/psc.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>

#include <nnt/nntest.h>

#include "testPsm_Common.h"

namespace nnt { namespace psm {

namespace {

const std::vector<nn::psc::PmState> PmStateSequenceSleep =
{
    nn::psc::PmState_FullAwake,
    nn::psc::PmState_MinimumAwake,
    nn::psc::PmState_SleepReady,
    nn::psc::PmState_EssentialServicesSleepReady,
    nn::psc::PmState_EssentialServicesAwake,
    nn::psc::PmState_MinimumAwake,
    nn::psc::PmState_FullAwake,
};

const std::vector<nn::psc::PmState> PmStateSequenceShutdownFromFullAwake =
{
    nn::psc::PmState_FullAwake,
    nn::psc::PmState_ShutdownReady,
};

const std::vector<nn::psc::PmState> PmStateSequenceShutdownFromMinimumAwake =
{
    nn::psc::PmState_FullAwake,
    nn::psc::PmState_MinimumAwake,
    nn::psc::PmState_ShutdownReady,
};

const std::vector<nn::psc::PmState>* const PsmGetterPolicyTestSettings[] =
{
    &PmStateSequenceSleep,
    &PmStateSequenceShutdownFromFullAwake,
    &PmStateSequenceShutdownFromMinimumAwake,
};

class PsmGetterPolicyTest : public ::testing::TestWithParam<const std::vector<nn::psc::PmState>*>
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

INSTANTIATE_TEST_CASE_P(PsmGetterPolicyTestName, PsmGetterPolicyTest, ::testing::ValuesIn(PsmGetterPolicyTestSettings));

} // namespace

void ExecuteOnPmStateSequence(std::function<void(void)> functor, const std::vector<nn::psc::PmState>* pSequence) NN_NOEXCEPT
{
    for ( auto iterator = pSequence->cbegin(); iterator != pSequence->cend(); iterator++ )
    {
        TransitPmState(*iterator);
        functor();
    }

    // 次のテスト開始時に PSM が生きている状態にする
    TransitPmState(nn::psc::PmState_FullAwake);
}

// 各システム電源状態で以下の Getter 関数で値を取得できるかつ取得できる値が変化しないことを確認するテスト
// 各テストの 1 行目の Change 関数呼び出しは状態を決定するための呼び出しであり、その値に意味はない

TEST_P(PsmGetterPolicyTest, GetBatteryChargePercentage)
{
    ChangeBatteryChargePercentage(60);
    auto expected = nn::psm::GetBatteryChargePercentage();
    ExecuteOnPmStateSequence([&] () { EXPECT_EQ(expected, nn::psm::GetBatteryChargePercentage()); }, GetParam());
}

TEST_P(PsmGetterPolicyTest, GetChargerType)
{
    ChangeUsbPowerState(UsbPowerState::He15V);
    auto expected = nn::psm::GetChargerType();
    ExecuteOnPmStateSequence([&] () { EXPECT_EQ(expected, nn::psm::GetChargerType()); }, GetParam() );
}

TEST_P(PsmGetterPolicyTest, GetBatteryVoltageState)
{
    ChangeBatteryVoltageMilliVolt(4000);
    auto expected = nn::psm::GetBatteryVoltageState();
    ExecuteOnPmStateSequence([&] () { EXPECT_EQ(expected, nn::psm::GetBatteryVoltageState()); }, GetParam() );
}

TEST_P(PsmGetterPolicyTest, GetRawBatteryChargePercentage)
{
    ChangeBatteryChargePercentage(55);
    auto expected = nn::psm::GetRawBatteryChargePercentage();
    ExecuteOnPmStateSequence([&] () { EXPECT_EQ(expected, nn::psm::GetRawBatteryChargePercentage()); }, GetParam() );
}

TEST_P(PsmGetterPolicyTest, IsEnoughPowerSupplied)
{
    ChangeUsbPowerState(UsbPowerState::Lt13V);
    auto expected = nn::psm::IsEnoughPowerSupplied();
    ExecuteOnPmStateSequence([&] () { EXPECT_EQ(expected, nn::psm::IsEnoughPowerSupplied()); }, GetParam() );
}

}} // namespace nnt::psm
