﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>

#include <nn/gpio/gpio.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_FuelGaugeDriver.h"

#include "testPsm_GpioStub.h"

namespace {

#define NNT_DETAIL_PSM_ASSERT_MINMAX(value, min, max) \
    do \
    { \
        NN_ASSERT((value) >= (min));\
        NN_ASSERT((value) <= (max));\
    } \
    while ( NN_STATIC_CONDITION(false) )

const uint16_t StatusFieldTemperatureTooLow    = 0x00000001;
const uint16_t StatusFieldTemperatureTooHigh   = 0x00000002;
const uint16_t StatusFieldStateOfChargeTooLow  = 0x00000004;
const uint16_t StatusFieldStateOfChargeTooHigh = 0x00000008;
const uint16_t StatusFieldVoltageTooLow        = 0x00000010;
const uint16_t StatusFieldVoltageTooHigh       = 0x00000020;
const uint16_t StatusFieldBatteryInsertion     = 0x00000040;
const uint16_t StatusFieldBatteryRemoval       = 0x00000080;

struct FuelGaugeStatus
{
    bool interruptEnabled;

    int  lowTemperatureThreshold;
    int  highTemperatureThreshold;
    int  lowVoltageThreshold;
    int  highVoltageThreshold;
    int  lowChargeThreshold;
    int  highChargeThreshold;

    double temperatureCelsius;
    int    maxTemperatureCelsius;
    double chargePercentage;
    int    milliVolt;
    double agePercentage;
    int    averageCurrentMilliAmpere;
    double voltageFuelGaugePercentage;
    int    fullCapacity;
    int    filteredRemainingCapacity;

    bool needToRestoreParameters;

    uint16_t status;

    nn::psm::driver::detail::FuelGaugeDriver::Parameter parameter;

    void Clear() NN_NOEXCEPT
    {
        interruptEnabled = false;

        lowTemperatureThreshold = 0;
        highTemperatureThreshold = 0;
        lowVoltageThreshold = 0;
        highVoltageThreshold = 0;
        lowChargeThreshold = 0;
        highChargeThreshold = 0;

        temperatureCelsius = 35.0;
        maxTemperatureCelsius = 35;
        chargePercentage = 99.9;
        milliVolt = 3600;
        agePercentage = 50.0;
        averageCurrentMilliAmpere = 2048;
        voltageFuelGaugePercentage = 98.9;
        fullCapacity = 0x2476;
        filteredRemainingCapacity = 0x240F;

        needToRestoreParameters = false;

        status = 0;

        parameter.rcomp0 = 0;
        parameter.tempCo = 0;
        parameter.fullCap = 0;
        parameter.cycles = 0;
        parameter.fullCapNom = 0;
        parameter.iAvgEmpty = 0;
        parameter.qrTable00 = 0;
        parameter.qrTable10 = 0;
        parameter.qrTable20 = 0;
        parameter.qrTable30 = 0;
    }
};

FuelGaugeStatus g_FuelGaugeStatus;

} // namespace

namespace nn { namespace psm { namespace driver { namespace detail {

::nn::Result ClearInterruptImpl(uint16_t mask) NN_NOEXCEPT
{
    NN_ASSERT(
        (~(StatusFieldTemperatureTooLow
            | StatusFieldTemperatureTooHigh
            | StatusFieldVoltageTooLow
            | StatusFieldVoltageTooHigh
            | StatusFieldStateOfChargeTooLow
            | StatusFieldStateOfChargeTooHigh
            | StatusFieldBatteryInsertion
            | StatusFieldBatteryRemoval)
        & mask) == 0);

    g_FuelGaugeStatus.status &= ~mask;

    NN_RESULT_SUCCESS;
}

const int FuelGaugeDriver::TemperatureThresholdCelsiusMin = -128;
const int FuelGaugeDriver::TemperatureThresholdCelsiusMax = 127;
const int FuelGaugeDriver::VoltageThresholdMillivoltMax = 5100;
const int FuelGaugeDriver::CyclesMin = 96;
const int FuelGaugeDriver::CyclesClearThreshold = 296;
const int FuelGaugeDriver::InvalidLowChargeThresholdPercentage = 0x00;
const int FuelGaugeDriver::InvalidHighChargeThresholdPercentage = 0xFF;

void FuelGaugeDriver::Initialize() NN_NOEXCEPT
{
    g_FuelGaugeStatus.Clear();

    ::nn::gpio::Initialize();

    ::nn::gpio::OpenSession(&m_FuelGaugeInterruptSession, ::nn::gpio::GpioPadName_BattMgicIrq);
    ::nn::gpio::SetDirection(&m_FuelGaugeInterruptSession, ::nn::gpio::Direction_Input);
    ::nn::gpio::SetInterruptMode(&m_FuelGaugeInterruptSession, ::nn::gpio::InterruptMode_LowLevel);
    ::nn::gpio::SetInterruptEnable(&m_FuelGaugeInterruptSession, true);
}

void FuelGaugeDriver::Finalize() NN_NOEXCEPT
{
    ::nn::gpio::UnbindInterrupt(&m_FuelGaugeInterruptSession);
    ::nn::gpio::CloseSession(&m_FuelGaugeInterruptSession);

    ::nn::gpio::Finalize();
}

::nn::Result FuelGaugeDriver::EnableInterrupt() NN_NOEXCEPT
{
    g_FuelGaugeStatus.interruptEnabled = true;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::DisableInterrupt() NN_NOEXCEPT
{
    g_FuelGaugeStatus.interruptEnabled = false;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetLowTemperatureThreshold(int thresholdCelsius) NN_NOEXCEPT
{
    NNT_DETAIL_PSM_ASSERT_MINMAX(
        thresholdCelsius,
        TemperatureThresholdCelsiusMin,
        TemperatureThresholdCelsiusMax);

    g_FuelGaugeStatus.lowTemperatureThreshold = thresholdCelsius;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetHighTemperatureThreshold(int thresholdCelsius) NN_NOEXCEPT
{
    NNT_DETAIL_PSM_ASSERT_MINMAX(
        thresholdCelsius,
        TemperatureThresholdCelsiusMin,
        TemperatureThresholdCelsiusMax);

    g_FuelGaugeStatus.highTemperatureThreshold = thresholdCelsius;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetLowVoltageThreshold(int thresholdMillivolt) NN_NOEXCEPT
{
    NNT_DETAIL_PSM_ASSERT_MINMAX(thresholdMillivolt, 0, VoltageThresholdMillivoltMax);

    g_FuelGaugeStatus.lowVoltageThreshold = thresholdMillivolt;
    NN_UNUSED(thresholdMillivolt);
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetHighVoltageThreshold(int thresholdMillivolt) NN_NOEXCEPT
{
    NNT_DETAIL_PSM_ASSERT_MINMAX(thresholdMillivolt, 0, VoltageThresholdMillivoltMax);

    g_FuelGaugeStatus.highVoltageThreshold = thresholdMillivolt;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetLowChargeThreshold(int thresholdPercentage) NN_NOEXCEPT
{
    NNT_DETAIL_PSM_ASSERT_MINMAX(thresholdPercentage, 0, 100);

    g_FuelGaugeStatus.lowChargeThreshold = thresholdPercentage;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetHighChargeThreshold(int thresholdPercentage) NN_NOEXCEPT
{
    NNT_DETAIL_PSM_ASSERT_MINMAX(thresholdPercentage, 0, 100);

    g_FuelGaugeStatus.highChargeThreshold = thresholdPercentage;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::DisableLowChargeInterrupt() NN_NOEXCEPT
{
    g_FuelGaugeStatus.lowChargeThreshold = InvalidLowChargeThresholdPercentage;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::DisableHighChargeInterrupt() NN_NOEXCEPT
{
    g_FuelGaugeStatus.highChargeThreshold = InvalidHighChargeThresholdPercentage;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::BindInterrupt(::nn::os::SystemEventType* pOutInterruptEvent) NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::gpio::BindInterrupt(pOutInterruptEvent, &m_FuelGaugeInterruptSession));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearInterrupt(::nn::os::SystemEventType* pInterruptEvent) NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
        StatusFieldTemperatureTooLow
        | StatusFieldTemperatureTooHigh
        | StatusFieldVoltageTooLow
        | StatusFieldVoltageTooHigh
        | StatusFieldStateOfChargeTooLow
        | StatusFieldStateOfChargeTooHigh
        | StatusFieldBatteryInsertion
        | StatusFieldBatteryRemoval));

    // 連続で割り込みを受け付けないように最後に有効化
    ::nn::gpio::ClearInterruptStatus(&m_FuelGaugeInterruptSession);
    ::nn::os::ClearSystemEvent(pInterruptEvent);
    ::nn::gpio::SetInterruptEnable(&m_FuelGaugeInterruptSession, true);

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearTemperatureInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
        StatusFieldTemperatureTooLow
        | StatusFieldTemperatureTooHigh));

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearVoltageInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
        StatusFieldVoltageTooLow
        | StatusFieldVoltageTooHigh));

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearChargeInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
        StatusFieldStateOfChargeTooLow
        | StatusFieldStateOfChargeTooHigh));

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearBatteryInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
        StatusFieldBatteryInsertion
        | StatusFieldBatteryRemoval));

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetTemperatureCelsius(double* pTemperature) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTemperature);

    *pTemperature = g_FuelGaugeStatus.temperatureCelsius;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetMaxTemperatureCelsius(int* pOutMaxTemperatureCelsius) NN_NOEXCEPT
{
    *pOutMaxTemperatureCelsius = g_FuelGaugeStatus.maxTemperatureCelsius;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetStatus(Status* pStatus) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStatus);

    pStatus->Clear();

    pStatus->isLowTemperatureInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldTemperatureTooLow) != 0;
    pStatus->isHighTemperatureInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldTemperatureTooHigh) != 0;

    pStatus->isLowVoltageInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldVoltageTooLow) != 0;
    pStatus->isHighVoltageInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldVoltageTooHigh) != 0;

    pStatus->isLowChargeInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldStateOfChargeTooLow) != 0;
    pStatus->isHighChargeInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldStateOfChargeTooHigh) != 0;

    pStatus->isBatteryRemovalInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldBatteryRemoval) != 0;
    pStatus->isBatteryInsertionInterruptGenerated = (g_FuelGaugeStatus.status & StatusFieldBatteryInsertion) != 0;

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetBatteryChargePercentage(double* pOutValue) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = g_FuelGaugeStatus.chargePercentage;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetBatteryChargeMilliVoltage(int* pOutValue) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = g_FuelGaugeStatus.milliVolt;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetBatteryAgePercentage(double* pOutValue) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = g_FuelGaugeStatus.agePercentage;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SaveParameter(Parameter* pOutParameter) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutParameter);

    *pOutParameter = g_FuelGaugeStatus.parameter;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::RestoreParameter(const Parameter& parameter) NN_NOEXCEPT
{
    g_FuelGaugeStatus.parameter = parameter;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetCycles(int* pOutCycles) NN_NOEXCEPT
{
    *pOutCycles = g_FuelGaugeStatus.parameter.cycles;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearCycles() NN_NOEXCEPT
{
    // 何もしない。
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetAverageCurrentMilliAmpere(int* pOutValue) NN_NOEXCEPT
{
    *pOutValue = g_FuelGaugeStatus.averageCurrentMilliAmpere;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetVoltageFuelGaugePercentage(double* pOutValue) NN_NOEXCEPT
{
    *pOutValue = g_FuelGaugeStatus.voltageFuelGaugePercentage;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetFullCapacityMilliAmpereH(int* pOutMilliAmpereH) NN_NOEXCEPT
{
    *pOutMilliAmpereH = g_FuelGaugeStatus.fullCapacity;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetRemainingCapacityMilliAmpereH(int* pOutMilliAmpereH) NN_NOEXCEPT
{
    *pOutMilliAmpereH = g_FuelGaugeStatus.filteredRemainingCapacity;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetNeedToRestoreParameters(bool needToRestoreParameters) NN_NOEXCEPT
{
    g_FuelGaugeStatus.needToRestoreParameters = needToRestoreParameters;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetNeedToRestoreParameters(bool* pOutNeedToRestoreParameters) NN_NOEXCEPT
{
    *pOutNeedToRestoreParameters = g_FuelGaugeStatus.needToRestoreParameters;
    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail

namespace nnt { namespace psm { namespace driver { namespace detail {

void SetBatteryTemperature(double temperature) NN_NOEXCEPT
{
    g_FuelGaugeStatus.temperatureCelsius = temperature;
}

void SetBatteryChargePercentage(double batteryChargePercentage) NN_NOEXCEPT
{
    g_FuelGaugeStatus.chargePercentage = batteryChargePercentage;
}

void SetAverageCurrentMilliAmpere(int averageCurrentMilliAmpere) NN_NOEXCEPT
{
    g_FuelGaugeStatus.averageCurrentMilliAmpere = averageCurrentMilliAmpere;
}

void SetVoltageFuelGaugePercentage(double voltageFuelGaugePercentage) NN_NOEXCEPT
{
    g_FuelGaugeStatus.voltageFuelGaugePercentage = voltageFuelGaugePercentage;
}

void SignalBatteryTemperatureTooLow() NN_NOEXCEPT
{
    g_FuelGaugeStatus.status |= StatusFieldTemperatureTooLow;
    nnt::gpio::Signal(nn::gpio::GpioPadName_BattMgicIrq);
}

void SignalBatteryTemperatureTooHigh() NN_NOEXCEPT
{
    g_FuelGaugeStatus.status |= StatusFieldTemperatureTooHigh;
    nnt::gpio::Signal(nn::gpio::GpioPadName_BattMgicIrq);
}

void SetBatteryVoltageMilliVolt(int milliVolt) NN_NOEXCEPT
{
    g_FuelGaugeStatus.milliVolt = milliVolt;
}

void SignalBatteryVoltageTooLow() NN_NOEXCEPT
{
    g_FuelGaugeStatus.status |= StatusFieldVoltageTooLow;
    nnt::gpio::Signal(nn::gpio::GpioPadName_BattMgicIrq);
}

void SignalBatteryVoltageTooHigh() NN_NOEXCEPT
{
    g_FuelGaugeStatus.status |= StatusFieldVoltageTooHigh;
    nnt::gpio::Signal(nn::gpio::GpioPadName_BattMgicIrq);
}

}}}} // namespace nnt::psm::driver::detail
