﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/usb/usb_Pm.h>

#include <nnt/nntest.h>

// SIGLO-81559 の修正に伴い追加したテストです。
// 実機で EnoughPowerChargeEmulation が期待通りに（APM によって）操作されていることを保証します。
// 引数で --gtest_filter=*.*Disabled* もしくは --gtest_filter=*.*Enabled* を渡すことを前提としています。

namespace nnt { namespace psm {

int GetUpdRdoCurrentMilliAmpere(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    auto current = state.rdo.Get<::nn::usb::pd::Rdo::MaximumOperatingCurrent>();
    return static_cast<int>(current * ::nn::usb::pd::RdoMilliAmpereUnit);
}

int GetUpdVoltageMilliVolt(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    auto voltage = state.pdo.Get<::nn::usb::pd::Pdo::Voltage>();
    return static_cast<int>(voltage * ::nn::usb::pd::PdoMilliVoltUnit);
}

::nn::psm::ChargerType DeriveChargerType(const ::nn::usb::UsbPowerState& pmState) NN_NOEXCEPT
{
    if ( pmState.role != ::nn::usb::UsbPowerRole_Sink )
    {
        return ::nn::psm::ChargerType_Unconnected;
    }
    else if ( pmState.charger == ::nn::usb::UsbChargerType_Unknown )
    {
        return ::nn::psm::ChargerType_NotSupported;
    }
    else if ( pmState.charger == ::nn::usb::UsbChargerType_Pd )
    {
        const auto EnoughPowerMilliWatThreshold = 39000;

        auto voltage = GetUpdVoltageMilliVolt(pmState);
        auto current = GetUpdRdoCurrentMilliAmpere(pmState);

        if ( voltage * current / 1000 >= EnoughPowerMilliWatThreshold )
        {
            return ::nn::psm::ChargerType_EnoughPower;
        }
    }

    return ::nn::psm::ChargerType_LowPower;
}

}} // namespace ::nnt::psm

// EnoughPowerChargeEmulation 有効時の期待値
TEST(PsmEnoughPowerChargeEmulationTest, EmulationEnabledGetChargerType)
{
    ::nn::psm::Initialize();

    EXPECT_EQ(::nn::psm::GetChargerType(), ::nn::psm::ChargerType_EnoughPower);

    ::nn::psm::Finalize();
}

// EnoughPowerChargeEmulation 無効時の期待値
TEST(PsmEnoughPowerChargeEmulationTest, EmulationDisabledGetChargerType)
{
    ::nn::psm::Initialize();

    ::nn::usb::PmClient upmClient;
    ::nn::usb::UsbPowerState pmState;
    upmClient.Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(upmClient.GetPowerState(&pmState));
    upmClient.Finalize();

    EXPECT_EQ(::nn::psm::GetChargerType(), ::nnt::psm::DeriveChargerType(pmState));

    ::nn::psm::Finalize();
}

// EnoughPowerChargeEmulation 有効時の期待値
TEST(PsmEnoughPowerChargeEmulationTest, EmulationEnabledIsEnoughPowerSupplied)
{
    ::nn::psm::Initialize();

    EXPECT_TRUE(::nn::psm::IsEnoughPowerSupplied());

    ::nn::psm::Finalize();
}

// EnoughPowerChargeEmulation 無効時の期待値
TEST(PsmEnoughPowerChargeEmulationTest, EmulationDisabledIsEnoughPowerSupplied)
{
    ::nn::psm::Initialize();

    ::nn::usb::PmClient upmClient;
    ::nn::usb::UsbPowerState pmState;
    upmClient.Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(upmClient.GetPowerState(&pmState));
    upmClient.Finalize();

    EXPECT_EQ(::nn::psm::IsEnoughPowerSupplied(), (::nnt::psm::DeriveChargerType(pmState) == ::nn::psm::ChargerType_EnoughPower) ? true : false);

    ::nn::psm::Finalize();
}
