﻿param(
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]$SampleMode = [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]::InProcess,
    
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [string]$Platform,
    
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [string]$BuildType
)

[double]$diffToPass = 0.100

. "$([Nintendo.CPU.Profiler.Scripting.ScriptingCore]::ScriptDirectory)/testProfiler_Common.ps1"

write-output "Starting testProfiler_ThreadMigration"

[Nintendo.CPU.Profiler.Scripting.ScriptingGUI+ConsoleWindow]::Clear()
[bool]$isEnabled = [Nintendo.CPU.Profiler.Scripting.ScriptingGUI+ConsoleWindow]::IsEnabled()
[Nintendo.CPU.Profiler.Scripting.ScriptingGUI+ConsoleWindow]::Enable()

SyncWhenAvailable

write-output "Waiting for profile to transfer..."
profiler-waitfor ReadyToProfile
sleep 1
write-output "Received!"

if ($isEnabled -eq $false)
{
    [Nintendo.CPU.Profiler.Scripting.ScriptingGUI+ConsoleWindow]::Disable()
}

# Thread migration is now a checker.
# Wait for Checkers to finish before attempting to look at this more.
WaitForCheckersToFinish

function ThreadSampleCounts()
{
    $containedFailure = $false

    profiler-units Samples $false

    [string]$consoleOutput = [Nintendo.CPU.Profiler.Scripting.ScriptingGUI+ConsoleWindow]::Copy()
    [string[]]$lines = $consoleOutput -split [Environment]::NewLine
    [int[]]$counts = @()
    foreach ($line in $lines)
    {
        if ($line -match "^Thread \d: \d+$")
        {
            [string]$countStr = $line -match "\d+$" | foreach {$Matches[0]}
            [int]$count = [Convert]::ToInt32($countStr, 10)
            $counts += $count
        }
    }

    [double]$denominator = [double]($counts[0] + $counts[1] + $counts[2])
    [double]$numerator = [double]$counts[3]
    [double]$runtimeRatio = [Math]::Round($numerator / $denominator, 3)

    [string]$threadsCopy = profiler-copy Threads
    [string[]]$threads = $threadsCopy -split [Environment]::NewLine
    [int[]]$samples = 0,0,0,0
    foreach ($thread in $threads)
    {
        if ($thread -match "\s*Thread: \d$")
        {
            [string]$threadNumStr = $thread.Substring($thread.Length - 1)
            [int]$threadNum = [Convert]::ToInt32($threadNumStr, 10)
            [string]$selfStr = $thread -match "\s([\d,]+),\s" | foreach {$Matches[1]}
            [int]$self = [Convert]::ToInt32($selfStr.Replace(",",""), 10)
            $samples[$threadNum] += $self
        }
    }

    $denominator = [double]($samples[0] + $samples[1] + $samples[2])
    $numerator = [double]$samples[3]
    [double]$sampledRatio = [Math]::Round($numerator / $denominator, 3)

    write-output "Sampled Ratio = $sampledRatio"

    [double]$margin = $runtimeRatio * $global:diffToPass
    [double]$lowBound = $runtimeRatio - $margin
    [double]$highBound = $runtimeRatio + $margin

    write-output "Target = [$lowBound, $highBound]"

    if (!(($lowBound -le $sampledRatio) -and ($sampledRatio -le $highBound)))
    {
        write-error "Sampled Ratio out of bounds"
        $containedFailure = $true
    }

    return $containedFailure
}

function HasMigratingThreadWarning()
{
    $containedFailure = $false
    
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count
    
    if ($warningCount -ne 1)
    {
        [string]$error = "Incorrect number of warnings generated. Saw " + $warningCount.ToString() + ", expected 1."
        write-error "$error"
        $containedFailure = $true
    }
    
    return $containedFailure
}


if ($SampleMode -eq [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]::OutOfProcess)
{
    RunTest("ThreadSampleCounts")
}
if ($SampleMode -eq [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]::InProcess)
{
    RunTest("HasMigratingThreadWarning")
}

profiler-unsync

write-output "Unsynced from devkit"

TestsFinished
