﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <set>
#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/prepo/detail/service/core/prepo_UserManager.h>

namespace
{
    const nn::account::Uid User1 = {{0x00ull, 0x01ull}};
    const nn::account::Uid User2 = {{0x00ull, 0x02ull}};
    const nn::account::Uid User3 = {{0x00ull, 0x03ull}};
    const nn::account::Uid User4 = {{0x00ull, 0x04ull}};
    const nn::account::Uid User5 = {{0x00ull, 0x05ull}};
    const nn::account::Uid User6 = {{0x00ull, 0x06ull}};
    const nn::account::Uid User7 = {{0x00ull, 0x07ull}};
    const nn::account::Uid User8 = {{0x00ull, 0x08ull}};
    const nn::account::Uid User9 = {{0x00ull, 0x09ull}};
    const nn::account::Uid User10 = {{0x00ull, 0x0Aull}};
    const nn::account::Uid User11 = {{0x00ull, 0x0Bull}};

    using LessFuncForUid = bool (*)(const nn::account::Uid& lhs, const nn::account::Uid& rhs);

    bool Less(const nn::account::Uid& lhs, const nn::account::Uid& rhs)
    {
        if (lhs._data[0] < rhs._data[0])
        {
            return true;
        }
        if (lhs._data[0] == rhs._data[0] && lhs._data[1] < rhs._data[1])
        {
            return true;
        }
        return false;
    }

    std::set<nn::account::Uid, LessFuncForUid> g_Users(Less);

    void AddUser(const nn::account::Uid& user) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(g_Users.size() < nn::account::UserCountMax);
        g_Users.insert(user);
    }

    void RemoveUser(const nn::account::Uid& user) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(g_Users.find(user) != g_Users.end());
        g_Users.erase(user);
    }
}

namespace nn { namespace account {
    // テスト用に nn::account::ListAllUsers() を上書きする。
    Result GetUserExistence(bool* pOutExistence, const Uid& user) NN_NOEXCEPT
    {
        *pOutExistence = g_Users.find(user) != g_Users.end();
        NN_RESULT_SUCCESS;
    }
}}

TEST(UserManager, Basic)
{
    nn::prepo::detail::service::core::UserManager manager;

    // 初めはいずれのユーザも含まれない。
    EXPECT_FALSE(manager.Contains(User1));
    EXPECT_FALSE(manager.Contains(User2));
    EXPECT_FALSE(manager.Contains(User3));
    EXPECT_FALSE(manager.Contains(User4));
    EXPECT_FALSE(manager.Contains(User5));
    EXPECT_FALSE(manager.Contains(User6));
    EXPECT_FALSE(manager.Contains(User7));
    EXPECT_FALSE(manager.Contains(User8));
    EXPECT_FALSE(manager.Contains(User9));

    // account に User1 ~ 8 がいることにする。
    AddUser(User1);
    AddUser(User2);
    AddUser(User3);
    AddUser(User4);
    AddUser(User5);
    AddUser(User6);
    AddUser(User7);
    AddUser(User8);

    // 1個ずつの操作。
    manager.Add(User1);
    EXPECT_TRUE(manager.Contains(User1));
    manager.RemoveIfContains(User1);
    EXPECT_FALSE(manager.Contains(User1));

    // 追加されていないユーザーの削除も成功する。
    manager.RemoveIfContains(User1);
    EXPECT_FALSE(manager.Contains(User1));

    // 2 個連続の操作。
    manager.Add(User2);
    manager.Add(User3);
    EXPECT_TRUE(manager.Contains(User2));
    EXPECT_TRUE(manager.Contains(User3));
    manager.RemoveIfContains(User2);
    manager.RemoveIfContains(User3);
    EXPECT_FALSE(manager.Contains(User2));
    EXPECT_FALSE(manager.Contains(User3));

    // 同じユーザを繰り返し追加 & 削除する
    manager.Add(User4);
    manager.Add(User4);
    EXPECT_TRUE(manager.Contains(User4));
    manager.RemoveIfContains(User4);
    manager.RemoveIfContains(User4);
    EXPECT_FALSE(manager.Contains(User4));

    // 前方のスロットに空きがあっても、重複して追加されないことの確認。
    manager.Add(User1);
    manager.Add(User2);
    manager.RemoveIfContains(User1); // 先頭に空きスロットを作る。
    manager.Add(User2);
    manager.Add(User3);
    manager.Add(User4);
    manager.Add(User5);
    manager.Add(User6);
    manager.Add(User7);
    manager.Add(User8); // User2 が重複していれば、空きスロットが足りずにアボートする。

    // 最大数を超えるユーザーを追加した時点で、account にないユーザーが
    // manager から削除される。
    RemoveUser(User1);
    AddUser(User9);
    manager.Add(User9);
    EXPECT_TRUE(manager.Contains(User9));
    EXPECT_FALSE(manager.Contains(User1));

    // 2 個連続で操作する場合。
    RemoveUser(User2);
    RemoveUser(User3);
    AddUser(User10);
    AddUser(User11);
    manager.Add(User10);
    manager.Add(User11);
    EXPECT_TRUE(manager.Contains(User10));
    EXPECT_TRUE(manager.Contains(User11));
    EXPECT_FALSE(manager.Contains(User2));
    EXPECT_FALSE(manager.Contains(User3));
}
