﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <array>
#include <map>
#include <mutex>
#include <nn/os.h>

#include <nn/nn_Assert.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include "PsmStub.h"

namespace {
    nn::psm::ChargerType g_ChargerType = nn::psm::ChargerType_Unconnected;
    nn::psm::BatteryVoltageState g_BatteryVoltageState = nn::psm::BatteryVoltageState_Good;
    double g_RawBatteryChargePercentage = 50.0f;

    // テスト対象 (spsm) 内で psm の初期化・終了の多重呼び出しがあるので参照カウントが必要
    struct StaticMutex
    {
        nn::os::MutexType mutex;
        void lock() NN_NOEXCEPT { nn::os::LockMutex(&mutex); }
        void unlock() NN_NOEXCEPT { nn::os::UnlockMutex(&mutex); }
    };
    StaticMutex g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };
    StaticMutex g_SessionImplMapMutex = { NN_OS_MUTEX_INITIALIZER(false) };

    int g_InitializeCount = 0;

    class SessionImpl
    {
    public:
        SessionImpl() NN_NOEXCEPT : m_Opened(false),
            m_StateChangeEvent(nn::os::EventClearMode_ManualClear, true),
            m_ChargerTypeChangeEventEnabled(true),
            m_PowerSupplyChangeEventEnabled(true),
            m_BatteryVoltageStateChangeEventEnabled(true)
        {
        }

        void Open() NN_NOEXCEPT
        {
            m_Opened = true;
        }

        void Close() NN_NOEXCEPT
        {
            m_Opened = false;
        }

        bool IsOpened() NN_NOEXCEPT
        {
            return m_Opened;
        }

        void SetChargerTypeChangeEventEnabled(bool isEnabled) NN_NOEXCEPT
        {
            m_ChargerTypeChangeEventEnabled = isEnabled;
        }

        void SetPowerSupplyChangeEventEnabled(bool isEnabled) NN_NOEXCEPT
        {
            m_PowerSupplyChangeEventEnabled = isEnabled;
        }

        void SetBatteryVoltageStateChangeEventEnabled(bool isEnabled) NN_NOEXCEPT
        {
            m_BatteryVoltageStateChangeEventEnabled = isEnabled;
        }

        void SignalChargerTypeChangeEvent() NN_NOEXCEPT
        {
            if (m_ChargerTypeChangeEventEnabled)
            {
                m_StateChangeEvent.Signal();
            }
        }

        void SignalPowerSupplyChangeEvent() NN_NOEXCEPT
        {
            if (m_PowerSupplyChangeEventEnabled)
            {
                m_StateChangeEvent.Signal();
            }
        }

        void SignalBatteryVoltageStateChangeEvent() NN_NOEXCEPT
        {
            if (m_BatteryVoltageStateChangeEventEnabled)
            {
                m_StateChangeEvent.Signal();
            }
        }

        nn::os::SystemEventType* GetStateChangeEvent() NN_NOEXCEPT
        {
            return m_StateChangeEvent.GetBase();
        }

    private:
        bool m_Opened;
        bool m_ChargerTypeChangeEventEnabled;
        bool m_PowerSupplyChangeEventEnabled;
        bool m_BatteryVoltageStateChangeEventEnabled;
        nn::os::SystemEvent m_StateChangeEvent;
    };

    const int MaxSessionCount = 2;
    SessionImpl g_SessionImpls[MaxSessionCount];

    int g_OpenedSessionCount = 0;
    std::map<nn::psm::Session*, SessionImpl*> g_SessionImplMap;
}

namespace nn { namespace psm {

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);
    if ( g_InitializeCount == 0 )
    {
        g_BatteryVoltageState = BatteryVoltageState_Good;
    }
    g_InitializeCount++;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);
    g_InitializeCount--;
    if ( g_InitializeCount == 0 )
    {
        g_BatteryVoltageState = BatteryVoltageState_Good;
    }
}

void OpenSession(Session* pOutSession) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_SessionImplMapMutex);

    SessionImpl* pSessionImpl = nullptr;
    for (int i = 0; i < MaxSessionCount; i++)
    {
        if (!g_SessionImpls[i].IsOpened())
        {
            pSessionImpl = &g_SessionImpls[i];
        }
    }

    NN_ASSERT(pSessionImpl != nullptr, "Failed to OpenSession().\n");

    pSessionImpl->Open();
    g_SessionImplMap.insert(std::make_pair(pOutSession, pSessionImpl));
}

void CloseSession(Session* pSession) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_SessionImplMapMutex);

    g_SessionImplMap[pSession]->Close();
    g_SessionImplMap.erase(pSession);
}

void SetChargerTypeChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    g_SessionImplMap[pSession]->SetChargerTypeChangeEventEnabled(isEnabled);
}

void SetPowerSupplyChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    g_SessionImplMap[pSession]->SetPowerSupplyChangeEventEnabled(isEnabled);
}

void SetBatteryVoltageStateChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    g_SessionImplMap[pSession]->SetBatteryVoltageStateChangeEventEnabled(isEnabled);
}

void BindStateChangeEvent(nn::os::SystemEventType* pOutSystemEvent, Session* pSession) NN_NOEXCEPT
{
    nn::os::SystemEventType* pEvent = g_SessionImplMap[pSession]->GetStateChangeEvent();

    nn::os::AttachReadableHandleToSystemEvent(pOutSystemEvent,
        nn::os::GetReadableHandleOfSystemEvent(pEvent), false, nn::os::EventClearMode_ManualClear);
}

void UnbindStateChangeEvent(Session* pSession) NN_NOEXCEPT
{
    NN_UNUSED(pSession);
}

ChargerType GetChargerType() NN_NOEXCEPT
{
    return g_ChargerType;
}

BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT
{
    return g_BatteryVoltageState;
}

double GetRawBatteryChargePercentage() NN_NOEXCEPT
{
    return g_RawBatteryChargePercentage;
}


}} // namespace nn::psm

// 以下テスト環境専用

namespace nnt { namespace psm {

void SignalStateChangeEvent() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_SessionImplMapMutex);

    for (auto pair : g_SessionImplMap)
    {
        pair.second->SignalChargerTypeChangeEvent();
    }
}

void SignalPowerSupplyChangeEvent() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_SessionImplMapMutex);

    for (auto pair : g_SessionImplMap)
    {
        pair.second->SignalPowerSupplyChangeEvent();
    }
}

void SignalBatteryVoltageStateChangeEvent() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_SessionImplMapMutex);

    for (auto pair : g_SessionImplMap)
    {
        pair.second->SignalBatteryVoltageStateChangeEvent();
    }
}

void SetBatteryVoltageState(nn::psm::BatteryVoltageState batteryVoltageState) NN_NOEXCEPT
{
    g_BatteryVoltageState = batteryVoltageState;
}

void SetChargerType(nn::psm::ChargerType chargerType) NN_NOEXCEPT
{
    g_ChargerType = chargerType;
}

void SetRawBatteryChargePercentage(double rawBatteryCharge) NN_NOEXCEPT
{
    g_RawBatteryChargePercentage = rawBatteryCharge;
}

}} // namespace nnt::psm
