﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/psc.h>
#include <nn/psc/psc_PmControl.h>
#include <nn/nn_Log.h>
#include <nn/sm/sm_Result.h>
#include <nnt.h>

#include "PscWorker.h"

//
// Dependency graph for test modules
//
//          [14]-------------------
//          /                       \
//       [1]           [7]         [12] ----[16]
//      /  \          /  \         / | \
//    [2]  [5]      [8]   [9]     /  |  ----[15]
//   /  \     \    /      / \    /   |
// [3]  [4]    [10]      /   [11]   [13]
//        \      |      /            |
//         \     |     /             |
//          \    |    /              |
//           \   |   /               |
//            \  |  /                |
//             \ | /                 |
//              \ /                  |
//              [6]------------------
//
//

namespace nnt  {
namespace psc  {

#define countof(a) (sizeof(a)/sizeof(a[0]))

nn::psc::PmState g_States[20];

class TestModule : public Worker
{
private:
    nn::psc::PmModuleId         m_Id;
    nn::psc::PmModule           m_Module;
    nn::os::MultiWaitHolderType m_Holder;
    nn::os::MultiWaitHolderType m_TerminateHolder;
    nn::os::SystemEvent         m_TerminateEvent;
    bool                        m_Running;
    nn::os::MultiWaitType       m_MultiWait;
    nn::psc::PmModuleId*        m_ChildList;
    uint32_t                    m_ChildCount;
    nn::psc::PmModuleId*        m_ParentList;
    uint32_t                    m_ParentCount;

    void CheckParentState(nn::psc::PmState state)
    {
        for (auto i = 0; i < m_ParentCount; i++)
        {
            int id = m_ParentList[i];
            if (g_States[id] != state)
            {
                ADD_FAILURE();
            }
        }
    }

    void CheckChildState(nn::psc::PmState state)
    {
        for (auto i = 0; i < m_ChildCount; i++)
        {
            int id = m_ChildList[i];
            if (g_States[id] != state)
            {
                ADD_FAILURE();
            }
        }
    }

    void PmThread()
    {
        nn::Result         result;
        nn::psc::PmState   state;
        nn::psc::PmFlagSet flags;

        while (m_Running)
        {
            nn::os::MultiWaitHolderType* pHolder = nn::os::WaitAny(&m_MultiWait);

            if (pHolder == &m_TerminateHolder && !m_Running)
            {
                break;
            }

            (m_Module.GetEventPointer())->Clear();

            result = m_Module.GetRequest(&state, &flags);
            NNT_EXPECT_RESULT_SUCCESS(result);

            switch (state)
            {
            case nn::psc::PmState_FullAwake:
            case nn::psc::PmState_MinimumAwake:
            case nn::psc::PmState_SleepReady:
            case nn::psc::PmState_EssentialServicesSleepReady:
            case nn::psc::PmState_EssentialServicesAwake:
                // flag tells direction of the transition
                if (flags.Test<nn::psc::PmFlag::FlagBit0>())
                {
                    // to higher power state, check that all children are running
                    CheckChildState(state);
                }
                else
                {
                    // to lower power state, check that all parents are sleeping
                    CheckParentState(state);
                }
                break;

            default:
                ADD_FAILURE();
                break;
            }

            if (g_States[m_Id] == state)
            {
                // if module is already in 'requested' PM state
                // it should not have received a notification event
                ADD_FAILURE();
            }

            g_States[m_Id] = state;

            result = m_Module.Acknowledge(state, nn::ResultSuccess());
            NNT_EXPECT_RESULT_SUCCESS(result);
        }
    }

public:

    nn::Result Initialize()
    NN_NOEXCEPT
    {
        nn::Result result;

        m_TerminateEvent.Clear();

        result = m_Module.Initialize(
                            m_Id,
                            m_ChildList,
                            m_ChildCount,
                            nn::os::EventClearMode_ManualClear);

        if (result.IsFailure())
        {
            return result;
        }

        m_Running = true;

        nn::os::InitializeMultiWait(&m_MultiWait);
        nn::os::InitializeMultiWaitHolder(&m_TerminateHolder, m_TerminateEvent.GetBase());
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TerminateHolder);

        nn::os::InitializeMultiWaitHolder(&m_Holder, (m_Module.GetEventPointer())->GetBase());
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_Holder);

        result = Worker::Initialize(std::bind(&TestModule::PmThread, this), nn::os::DefaultThreadPriority);
        NNT_EXPECT_RESULT_SUCCESS(result);

        return result;
    }

    void Finalize()
    {
        if (m_Running)
        {
            m_Running = false;
            m_TerminateEvent.Signal();
            Worker::Finalize();
            m_Module.Finalize();
            nn::os::UnlinkMultiWaitHolder(&m_TerminateHolder);
            nn::os::UnlinkMultiWaitHolder(&m_Holder);
            nn::os::FinalizeMultiWait(&m_MultiWait);
        }
    }

    TestModule(
        nn::psc::PmModuleId id,
        nn::psc::PmModuleId* pChildren,
        int childCount,
        nn::psc::PmModuleId* pParent,
        int parentCount)
    NN_NOEXCEPT :
        m_Id(id),
        m_TerminateEvent(nn::os::EventClearMode_AutoClear, false),
        m_Running(false),
        m_ChildList(pChildren),
        m_ChildCount(childCount),
        m_ParentList(pParent),
        m_ParentCount(parentCount)
    {

    }

    ~TestModule()
    {
        Finalize();
    }

};

nn::psc::PmModuleId moduleId1    = (nn::psc::PmModuleId)(1);
nn::psc::PmModuleId moduleId2    = (nn::psc::PmModuleId)(2);
nn::psc::PmModuleId moduleId3    = (nn::psc::PmModuleId)(3);
nn::psc::PmModuleId moduleId4    = (nn::psc::PmModuleId)(4);
nn::psc::PmModuleId moduleId5    = (nn::psc::PmModuleId)(5);
nn::psc::PmModuleId moduleId6    = (nn::psc::PmModuleId)(6);
nn::psc::PmModuleId moduleId7    = (nn::psc::PmModuleId)(7);
nn::psc::PmModuleId moduleId8    = (nn::psc::PmModuleId)(8);
nn::psc::PmModuleId moduleId9    = (nn::psc::PmModuleId)(9);
nn::psc::PmModuleId moduleId10   = (nn::psc::PmModuleId)(10);
nn::psc::PmModuleId moduleId11   = (nn::psc::PmModuleId)(11);
nn::psc::PmModuleId moduleId12   = (nn::psc::PmModuleId)(12);
nn::psc::PmModuleId moduleId13   = (nn::psc::PmModuleId)(13);
nn::psc::PmModuleId moduleId14   = (nn::psc::PmModuleId)(14);
nn::psc::PmModuleId moduleId15   = (nn::psc::PmModuleId)(15);
nn::psc::PmModuleId moduleId16   = (nn::psc::PmModuleId)(16);
nn::psc::PmModuleId moduleId17   = (nn::psc::PmModuleId)(17);
nn::psc::PmModuleId moduleId18   = (nn::psc::PmModuleId)(18);
nn::psc::PmModuleId moduleId19   = (nn::psc::PmModuleId)(19);
nn::psc::PmModuleId moduleId20   = (nn::psc::PmModuleId)(20);
nn::psc::PmModuleId moduleId21   = (nn::psc::PmModuleId)(21);

// Child dependency from the graph - used for module registration and to verify wakeup order
nn::psc::PmModuleId children1[]  = {moduleId2, moduleId5};
nn::psc::PmModuleId children2[]  = {moduleId3, moduleId4};
nn::psc::PmModuleId children3[]  = {};
nn::psc::PmModuleId children4[]  = {moduleId6};
nn::psc::PmModuleId children5[]  = {moduleId10};
nn::psc::PmModuleId children6[]  = {};
nn::psc::PmModuleId children7[]  = {moduleId8, moduleId9};
nn::psc::PmModuleId children8[]  = {moduleId10};
nn::psc::PmModuleId children9[]  = {moduleId6, moduleId11};
nn::psc::PmModuleId children10[] = {moduleId6};
nn::psc::PmModuleId children11[] = {};
nn::psc::PmModuleId children12[] = {moduleId11, moduleId13, moduleId15, moduleId16};
nn::psc::PmModuleId children13[] = {moduleId6};
nn::psc::PmModuleId children14[] = {moduleId1, moduleId12};
nn::psc::PmModuleId children15[] = {};
nn::psc::PmModuleId children16[] = {};
nn::psc::PmModuleId children17[] = {};
nn::psc::PmModuleId children18[] = {};
nn::psc::PmModuleId children19[] = {};
nn::psc::PmModuleId children20[] = {};
nn::psc::PmModuleId children21[] = {};

// Parent dependency from the graph - used to verify sleep order
nn::psc::PmModuleId parents1[]   = {moduleId14};
nn::psc::PmModuleId parents2[]   = {moduleId1};
nn::psc::PmModuleId parents3[]   = {moduleId2};
nn::psc::PmModuleId parents4[]   = {moduleId2};
nn::psc::PmModuleId parents5[]   = {moduleId1};
nn::psc::PmModuleId parents6[]   = {moduleId4, moduleId9, moduleId10, moduleId13};
nn::psc::PmModuleId parents7[]   = {};
nn::psc::PmModuleId parents8[]   = {moduleId7};
nn::psc::PmModuleId parents9[]   = {moduleId7};
nn::psc::PmModuleId parents10[]  = {moduleId5, moduleId8};
nn::psc::PmModuleId parents11[]  = {moduleId9, moduleId12};
nn::psc::PmModuleId parents12[]  = {moduleId14};
nn::psc::PmModuleId parents13[]  = {moduleId12};
nn::psc::PmModuleId parents14[]  = {};
nn::psc::PmModuleId parents15[]  = {moduleId12};
nn::psc::PmModuleId parents16[]  = {moduleId12};

// Test module initialization
TestModule module1(moduleId1,   children1,  countof(children1),  parents1,  countof(parents1));
TestModule module2(moduleId2,   children2,  countof(children2),  parents2,  countof(parents2));
TestModule module3(moduleId3,   children3,  countof(children3),  parents3,  countof(parents3));
TestModule module4(moduleId4,   children4,  countof(children4),  parents4,  countof(parents4));
TestModule module5(moduleId5,   children5,  countof(children5),  parents5,  countof(parents5));
TestModule module6(moduleId6,   children6,  countof(children6),  parents6,  countof(parents6));
TestModule module7(moduleId7,   children7,  countof(children7),  parents7,  countof(parents7));
TestModule module8(moduleId8,   children8,  countof(children8),  parents8,  countof(parents8));
TestModule module9(moduleId9,   children9,  countof(children9),  parents9,  countof(parents9));
TestModule module10(moduleId10, children10, countof(children10), parents10, countof(parents10));
TestModule module11(moduleId11, children11, countof(children11), parents11, countof(parents11));
TestModule module12(moduleId12, children12, countof(children12), parents12, countof(parents12));
TestModule module13(moduleId13, children13, countof(children13), parents13, countof(parents13));
TestModule module14(moduleId14, children14, countof(children14), parents14, countof(parents14));
TestModule module15(moduleId15, children15, countof(children15), parents15, countof(parents15));
TestModule module16(moduleId16, children16, countof(children16), parents16, countof(parents16));
TestModule module17(moduleId17, nullptr, 0, nullptr, 0);
TestModule module18(moduleId18, nullptr, 0, nullptr, 0);
TestModule module19(moduleId19, nullptr, 0, nullptr, 0);
TestModule module20(moduleId20, nullptr, 0, nullptr, 0);
TestModule module21(moduleId21, nullptr, 0, nullptr, 0);

TEST(Psc, TransitionOrder)
{
    const int Iterations = 1000;

    struct
    {
        nn::psc::PmState           state;
        nn::psc::PmTransitionOrder order;
        bool                       flagSet;
    }
    requests[] =
    {
        { nn::psc::PmState_FullAwake,                   nn::psc::PmTransitionOrder_ToHigherPowerState, true  },
        { nn::psc::PmState_MinimumAwake,                nn::psc::PmTransitionOrder_ToLowerPowerState,  false },
        { nn::psc::PmState_SleepReady,                  nn::psc::PmTransitionOrder_ToLowerPowerState,  false },
        { nn::psc::PmState_EssentialServicesSleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState,  false },
        { nn::psc::PmState_EssentialServicesAwake,      nn::psc::PmTransitionOrder_ToHigherPowerState, true  },
        { nn::psc::PmState_MinimumAwake,                nn::psc::PmTransitionOrder_ToHigherPowerState, true  },
        { nn::psc::PmState_FullAwake,                   nn::psc::PmTransitionOrder_ToHigherPowerState, true  },
    };

    nn::Result           result;
    nn::psc::PmControl   control;
    nn::psc::PmFlagSet   flags;
    nn::os::SystemEvent* pControlEvent;

    if ((result = module1.Initialize()).IsFailure()  ||
        (result = module2.Initialize()).IsFailure()  ||
        (result = module3.Initialize()).IsFailure()  ||
        (result = module4.Initialize()).IsFailure()  ||
        (result = module5.Initialize()).IsFailure()  ||
        (result = module6.Initialize()).IsFailure()  ||
        (result = module7.Initialize()).IsFailure()  ||
        (result = module8.Initialize()).IsFailure()  ||
        (result = module9.Initialize()).IsFailure()  ||
        (result = module10.Initialize()).IsFailure() ||
        (result = module11.Initialize()).IsFailure() ||
        (result = module12.Initialize()).IsFailure() ||
        (result = module13.Initialize()).IsFailure() ||
        (result = module14.Initialize()).IsFailure() ||
        (result = module15.Initialize()).IsFailure() ||
        (result = module16.Initialize()).IsFailure())
    {
        NNT_EXPECT_RESULT_SUCCESS(result);
        return;
    }

    for (auto i = 0; i < countof(g_States); i++)
    {
        g_States[i] = nn::psc::PmState_Unknown;
    }

    result = control.Initialize(nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    pControlEvent = control.GetEventPointer();

    // This will dispatch FullAwake twice in a row,
    // the second dispatch should not be passed
    // to PM modules - there is a check in TestModule class.
    for (auto i = 0; i < Iterations; i++)
    {
        for (auto j = 0; j < sizeof(requests) / sizeof(requests[0]); j++)
        {
            flags.Set<nn::psc::PmFlag::FlagBit0>(requests[j].flagSet ? 1 : 0);

            result = control.DispatchRequest(requests[j].state, flags, requests[j].order);
            NNT_EXPECT_RESULT_SUCCESS(result);

            pControlEvent->Wait();
            pControlEvent->Clear();

            result = control.GetResult();
            NNT_EXPECT_RESULT_SUCCESS(result);
        }
    }

    // Print stats for most recent transition
    control.PrintModuleInformation();

    // Finalize will fail if children
    // are finalized before parents.
    module14.Finalize();
    module1.Finalize();
    module7.Finalize();
    module12.Finalize();
    module16.Finalize();
    module2.Finalize();
    module5.Finalize();
    module8.Finalize();
    module9.Finalize();
    module15.Finalize();
    module3.Finalize();
    module4.Finalize();
    module10.Finalize();
    module11.Finalize();
    module13.Finalize();
    module6.Finalize();

    control.Finalize();
}

TEST(Psc, SelfDependency)
{
    nn::Result result;
    nn::psc::PmModuleId children[] = {moduleId1};
    TestModule module(moduleId1, children,  countof(children), nullptr,  0);

    result = module.Initialize();

    if (!(result <= nn::psc::ResultSelfDependency()))
    {
        ADD_FAILURE();
    }

    module.Finalize();
}

TEST(Psc, Cancel)
{
    nn::psc::PmModule       module1;
    nn::psc::PmControl      control;
    nn::psc::PmControlState controlState;
    nn::psc::PmFlagSet      flags;
    nn::Result              result;

    result = module1.Initialize(moduleId1, nullptr, 0, nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = control.Initialize(nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = control.DispatchRequest(nn::psc::PmState_SleepReady, flags, nn::psc::PmTransitionOrder_ToLowerPowerState);
    NNT_EXPECT_RESULT_SUCCESS(result);

    for (int i = 0; i < 5; i++)
    {
        // OK to call PrintModuleInformation without cancelling request
        control.PrintModuleInformation();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    result = control.Cancel();
    NNT_EXPECT_RESULT_SUCCESS(result);

    (control.GetEventPointer())->Wait();
    (control.GetEventPointer())->Clear();

    // Print stats, moduleId1 should show invalid
    // timestamp (-1) since it did not complete transition
    control.PrintModuleInformation();

    result = control.GetState(&controlState);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (controlState != nn::psc::PmControlState_Error)
    {
        ADD_FAILURE();
    }

    result = control.GetResult();

    if (!(result <= nn::psc::ResultCancel()))
    {
        ADD_FAILURE();
    }

    control.PrintModuleInformation();
    control.Finalize();
    module1.Finalize();
}

TEST(Psc, CircularDependency)
{
    nn::psc::PmModuleId children1[] = {moduleId2};
    nn::psc::PmModuleId children2[] = {moduleId3};
    nn::psc::PmModuleId children3[] = {moduleId1};
    nn::psc::PmModule module1;
    nn::psc::PmModule module2;
    nn::psc::PmModule module3;
    nn::Result result;

    result = module1.Initialize(moduleId1, children1, countof(children1), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module2.Initialize(moduleId2, children2, countof(children2), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module3.Initialize(moduleId3, children3, countof(children3), nn::os::EventClearMode_ManualClear);

    if (!(result <= nn::psc::ResultCircularDependency()))
    {
        ADD_FAILURE();
    }

    module1.Finalize();
    module2.Finalize();
    module3.Finalize();
}

TEST(Psc, NonfinalizedModuleExit1)
{
    nn::psc::PmModule module1;
    nn::psc::PmModule module2;
    nn::psc::PmModule module3;
    nn::psc::PmModuleId children1[] = {moduleId2};
    nn::psc::PmModuleId children2[] = {moduleId3};
    nn::Result result;

    result = module1.Initialize(moduleId1, children1, countof(children1), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module2.Initialize(moduleId2, children2, countof(children2), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module3.Initialize(moduleId3, nullptr, 0, nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    module1.Finalize();
    module2.Finalize();
}

TEST(Psc, NonfinalizedModuleExit2)
{
    // repeat the same, registration from the NonfinalizedModuleExit1
    // should not be present, the following should succeed.

    nn::psc::PmModule module1;
    nn::psc::PmModule module2;
    nn::psc::PmModule module3;
    nn::psc::PmModuleId children1[] = {moduleId2};
    nn::psc::PmModuleId children2[] = {moduleId3};
    nn::Result result;

    result = module1.Initialize(moduleId1, children1, countof(children1), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module2.Initialize(moduleId2, children2, countof(children2), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module3.Initialize(moduleId3, nullptr, 0, nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    module1.Finalize();
    module2.Finalize();
}

TEST(Psc, NoninitializedModule)
{
    nn::psc::PmModule module1;
    nn::Result result;
    nn::psc::PmFlagSet flags;
    nn::psc::PmState   state;

    result = module1.GetRequest(&state, &flags);
    if (!(result <= nn::psc::ResultNotInitialized()))
    {
        ADD_FAILURE();
    }
}

TEST(Psc, DoubleModuleInit)
{
    nn::psc::PmModule module1;
    nn::Result result;

    result = module1.Initialize(moduleId1, nullptr, 0, nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module1.Initialize(moduleId1, nullptr, 0, nn::os::EventClearMode_ManualClear);
    if (!(result <= nn::psc::ResultAlreadyInitialized()))
    {
        ADD_FAILURE();
    }
}

TEST(Psc, NonfinalizedControlExit1)
{
    nn::Result result;
    nn::psc::PmControl control;

    result = control.Initialize(nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
}

TEST(Psc, NonfinalizedControlExit2)
{
    // repeat the same, registration from NonfinalizedControlExit1
    // should not be present, the following should succeed
    nn::Result result;
    nn::psc::PmControl control;

    result = control.Initialize(nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
}

TEST(Psc, DoubleControlInit)
{
    nn::Result result;
    nn::psc::PmControl control1;
    nn::psc::PmControl control2;

    result = control1.Initialize(nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    // same control object can't be initialized twice
    result = control1.Initialize(nn::os::EventClearMode_ManualClear);
    if (!(result <= nn::psc::ResultAlreadyInitialized()))
    {
        ADD_FAILURE();
    }

    // verify only one control service is allowed in the system
    result = control2.Initialize(nn::os::EventClearMode_ManualClear);
    if (!(result <= nn::sm::ResultMaxSessions()))
    {
        ADD_FAILURE();
    }

    // finalize first controller, re-init the second one
    control1.Finalize();

    // Server has only one session for control. After finalize returns it may
    // take some time for the server to release that session.
    // If another initialize is called, an 'out of session' error may be returned.
    // Give server a chance to clean up after connection is closed.
    // In a typical scenario, rapid initialize/finalize will never be done, though.

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));

    result = control2.Initialize(nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
}

TEST(Psc, MaxDependencyDepth)
{
    nn::psc::PmModule module1;
    nn::psc::PmModule module2;
    nn::psc::PmModule module3;
    nn::psc::PmModule module4;
    nn::psc::PmModule module5;
    nn::psc::PmModule module6;
    nn::psc::PmModule module7;
    nn::psc::PmModule module8;
    nn::psc::PmModule module9;
    nn::psc::PmModule module10;
    nn::psc::PmModule module11;
    nn::psc::PmModule module12;
    nn::psc::PmModule module13;
    nn::psc::PmModule module14;
    nn::psc::PmModule module15;
    nn::psc::PmModule module16;
    nn::psc::PmModule module17;
    nn::psc::PmModule module18;
    nn::psc::PmModule module19;
    nn::psc::PmModule module20;
    nn::psc::PmModule module21;

    nn::psc::PmModuleId children1[]  = {moduleId2};
    nn::psc::PmModuleId children2[]  = {moduleId3};
    nn::psc::PmModuleId children3[]  = {moduleId4};
    nn::psc::PmModuleId children4[]  = {moduleId5};
    nn::psc::PmModuleId children5[]  = {moduleId6};
    nn::psc::PmModuleId children6[]  = {moduleId7};
    nn::psc::PmModuleId children7[]  = {moduleId8};
    nn::psc::PmModuleId children8[]  = {moduleId9};
    nn::psc::PmModuleId children9[]  = {moduleId10};
    nn::psc::PmModuleId children10[] = {moduleId11};
    nn::psc::PmModuleId children11[] = {moduleId12};
    nn::psc::PmModuleId children12[] = {moduleId13};
    nn::psc::PmModuleId children13[] = {moduleId14};
    nn::psc::PmModuleId children14[] = {moduleId15};
    nn::psc::PmModuleId children15[] = {moduleId16};
    nn::psc::PmModuleId children16[] = {moduleId17};
    nn::psc::PmModuleId children17[] = {moduleId18};
    nn::psc::PmModuleId children18[] = {moduleId19};
    nn::psc::PmModuleId children19[] = {moduleId20};
    nn::psc::PmModuleId children20[] = {moduleId21};
    nn::Result result;

    result = module1.Initialize(moduleId1, children1, countof(children1), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module2.Initialize(moduleId2, children2, countof(children2), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module3.Initialize(moduleId3, children3, countof(children3), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module4.Initialize(moduleId4, children4, countof(children4), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module5.Initialize(moduleId5, children5, countof(children5), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module6.Initialize(moduleId6, children6, countof(children6), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module7.Initialize(moduleId7, children7, countof(children7), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module8.Initialize(moduleId8, children8, countof(children8), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module9.Initialize(moduleId9, children9, countof(children9), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module10.Initialize(moduleId10, children10, countof(children10), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module11.Initialize(moduleId11, children11, countof(children11), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module12.Initialize(moduleId12, children12, countof(children12), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module13.Initialize(moduleId13, children13, countof(children13), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module14.Initialize(moduleId14, children14, countof(children14), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module15.Initialize(moduleId15, children15, countof(children15), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module16.Initialize(moduleId16, children16, countof(children16), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module17.Initialize(moduleId17, children17, countof(children17), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module18.Initialize(moduleId18, children18, countof(children18), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);
    result = module19.Initialize(moduleId19, children19, countof(children19), nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = module20.Initialize(moduleId20, children20, countof(children20), nn::os::EventClearMode_ManualClear);
    if (!(result <= nn::psc::ResultMaxDependencyLevels()))
    {
        ADD_FAILURE();
    }

    module1.Finalize();
    module2.Finalize();
    module3.Finalize();
    module4.Finalize();
    module5.Finalize();
    module6.Finalize();
    module7.Finalize();
    module8.Finalize();
    module9.Finalize();
    module10.Finalize();
    module11.Finalize();
    module12.Finalize();
    module13.Finalize();
    module14.Finalize();
    module15.Finalize();
    module16.Finalize();
    module17.Finalize();
    module18.Finalize();
    module19.Finalize();
} // NOLINT(impl/function_size)

TEST(Psc, UninitializedModule)
{
    nn::psc::PmControl      control;
    nn::psc::PmControlState controlState;
    nn::psc::PmFlagSet      flags;
    nn::os::SystemEvent*    pEvent;
    nn::Result result;

    flags.Set<nn::psc::PmFlag::FlagBit0>(1);

    result = control.Initialize(nn::os::EventClearMode_ManualClear);
    NNT_EXPECT_RESULT_SUCCESS(result);

    g_States[moduleId6] = nn::psc::PmState_Unknown;
    g_States[moduleId13] = nn::psc::PmState_Unknown;

    result = control.GetState(&controlState);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (controlState != nn::psc::PmControlState_Idle)
    {
        ADD_FAILURE();
    }

    // module13 depends on module6
    result = module13.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    pEvent = control.GetEventPointer();
    pEvent->Clear();

    result = control.DispatchRequest(nn::psc::PmState_FullAwake, flags, nn::psc::PmTransitionOrder_ToHigherPowerState);
    NNT_EXPECT_RESULT_SUCCESS(result);

    // transition should block since dependent moduleId6 is not initialized
    if (pEvent->TimedWait(nn::TimeSpan::FromMilliSeconds(100)))
    {
        ADD_FAILURE();
    }

    result = control.GetState(&controlState);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (controlState != nn::psc::PmControlState_Running)
    {
        ADD_FAILURE();
    }

    // initialize dependent moduleId6
    result = module6.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    // initializing dependent module should have unblocked this transition
    if (!(pEvent->TimedWait(nn::TimeSpan::FromMilliSeconds(100))))
    {
        ADD_FAILURE();
    }

    result = control.GetState(&controlState);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (controlState != nn::psc::PmControlState_Idle)
    {
        ADD_FAILURE();
    }

    result = control.GetResult();
    NNT_EXPECT_RESULT_SUCCESS(result);

    control.Finalize();
    module13.Finalize();
    module6.Finalize();
}

TEST(Psc, ModuleInitUnit)
{
    nn::Result         result;
    nn::psc::PmModule  module;

    for (int i = 0; i < 500; i++)
    {
        result = module.Initialize((nn::psc::PmModuleId)100, nullptr, 0, nn::os::EventClearMode_ManualClear);
        NNT_EXPECT_RESULT_SUCCESS(result);

        result = module.Finalize();
        NNT_EXPECT_RESULT_SUCCESS(result);
    }
}

TEST(Psc, ControlInitUnit)
{
    nn::Result result;
    nn::psc::PmControl control;

    for (int i = 0; i < 500; i++)
    {
        result = control.Initialize(nn::os::EventClearMode_ManualClear);
        NNT_EXPECT_RESULT_SUCCESS(result);

        result = control.Finalize();
        NNT_EXPECT_RESULT_SUCCESS(result);

        // Server has only one session for control. After finalize returns it may
        // take some time for the server to release that session.
        // If another initialize is called, an 'out of session' error may be returned.
        // Give server a chance to clean up after connection is closed.
        // In a typical scenario, rapid initialize/finalize will never be done, though.

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
    }
}

// Needs to be the last one as it intentionally leaks memory,
// relies on process exit for cleanup
TEST(Psc, MaxModulesPerProcess)
{
    nn::psc::PmModule* module;
    nn::Result         result;
    int                i;

    for (i = 0; i < nn::psc::MaximumModulesPerProcess; i++)
    {
        module = new nn::psc::PmModule;

        result = module->Initialize((nn::psc::PmModuleId)(i), nullptr, 0, nn::os::EventClearMode_ManualClear);
        NNT_EXPECT_RESULT_SUCCESS(result);

    }

    module = new nn::psc::PmModule;

    result = module->Initialize((nn::psc::PmModuleId)(i), nullptr, 0, nn::os::EventClearMode_ManualClear);

    if (!(result <= nn::sf::ResultMemoryAllocationFailed()))
    {
        ADD_FAILURE();
    }
}

}}
