﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/os.h>
#include <nn/psc.h>
#include <nn/psc/psc_PmControl.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/sm/sm_Result.h>
#include <nnt.h>

namespace nnt  {
namespace psc  {

namespace {
    const char *Usage =
        "Usage: PscRequestSender.nsp [-i <PowerState>] [-t <PowerState>| -w <MilliSeconds>]...\n"
        "    -i    Initial power state (default: FA). Specify in mnemonics described below\n"
        "    -t    Next target power state. Specify in mnemonics described below.\n"
        "    -w    Add interval before next transition [ms]\n"
        "\n"
        "    [Power state mnemonics]\n"
        "        FA = FullAwake\n"
        "        MA = MinimumAwake\n"
        "        SR = SleepReady\n"
        "        ESSR = EssentialServicesSleepReady\n"
        "        ESA = EssentialServicesAwake\n"
        "\n"
        "Example: PscRequestSender.nsp -t MA -w 1500 -t SR -w 500 -t MA\n"
        "    1: Transit from FullAwake (initial power state) to MinimumAwake.\n"
        "    2: Wait 1500 milli seconds.\n"
        "    3: Transit from MinimumAwake to SleepReady.\n"
        "    4: Wait 500 milli seconds.\n"
        "    5: Transit from SleepReady to MinimumAwake.\n";

    // プログラムが実行する処理を Instruction という単位で扱う。
    // 現在、各 Instruction では PowerState の遷移もしくはウェイトを行う。
    const int MaxInstructionCount = 32;

    enum InstructionType
    {
        InstructionType_Transition,
        InstructionType_Wait,
    };

    struct Instruction
    {
        InstructionType type;
        nn::psc::PmState state;
        int time;
    };

    struct ProgramOption
    {
        nn::psc::PmState initialPmState;
        int instructionCount;
        Instruction instructions[MaxInstructionCount];
    };

    nn::psc::PmState GetPmState(const char* str)
    {
        NN_ASSERT_NOT_NULL(str);

        if (strcmp(str, "FullAwake") == 0 || strcmp(str, "FA") == 0)
        {
            return nn::psc::PmState_FullAwake;
        }
        else if (strcmp(str, "MinimumAwake") == 0 || strcmp(str, "MA") == 0)
        {
            return nn::psc::PmState_MinimumAwake;
        }
        else if (strcmp(str, "SleepReady") == 0 || strcmp(str, "SR") == 0)
        {
            return nn::psc::PmState_SleepReady;
        }
        else if (strcmp(str, "EssentialServicesAwake") == 0 || strcmp(str, "ESA") == 0)
        {
            return nn::psc::PmState_EssentialServicesAwake;
        }
        else if (strcmp(str, "EssentialServicesSleepReady") == 0 || strcmp(str, "ESSR") == 0)
        {
            return nn::psc::PmState_EssentialServicesSleepReady;
        }

        return nn::psc::PmState_Unknown;
    }

    bool GetPmTransitionOrder(nn::psc::PmTransitionOrder* pOutOrder, nn::psc::PmState current, nn::psc::PmState next)
    {
        NN_ASSERT_NOT_NULL(pOutOrder);

        if (current > next)
        {
            *pOutOrder = nn::psc::PmTransitionOrder_ToHigherPowerState;
            return true;
        }

        if (current < next)
        {
            *pOutOrder = nn::psc::PmTransitionOrder_ToLowerPowerState;
            return true;
        }

        return false;
    }
}

bool ParseProgramOption(ProgramOption* pOutOption, int argc, char** argv)
{
    NN_ASSERT_NOT_NULL(pOutOption);

    if (argc < 2)
    {
        NN_LOG("%s", Usage);
        return false;
    }

    enum ParseState
    {
        ParseState_Default,
        ParseState_ExpectInitialPowerState,
        ParseState_ExpectPowerState,
        ParseState_ExpectWaitTime
    };

    ParseState parseState = ParseState_Default;
    int instructionIndex = 0;

    pOutOption->initialPmState = nn::psc::PmState_FullAwake;

    for (int i = 1; i < argc; i++)
    {
        if (instructionIndex >= MaxInstructionCount)
        {
            NN_LOG("Error: too many program options.\n");
            return false;
        }

        switch (parseState)
        {
        case ParseState_Default:
            if (strcmp(argv[i], "-i") == 0)
            {
                parseState = ParseState_ExpectInitialPowerState;
            }
            else if (strcmp(argv[i], "-t") == 0)
            {
                parseState = ParseState_ExpectPowerState;
                pOutOption->instructions[instructionIndex].type = InstructionType_Transition;
            }
            else if (strcmp(argv[i], "-w") == 0)
            {
                parseState = ParseState_ExpectWaitTime;
                pOutOption->instructions[instructionIndex].type = InstructionType_Wait;
            }
            else
            {
                NN_LOG("Error: failed to parse argument %d (%s).\n", i, argv[i]);
                return false;
            }
            break;

        case ParseState_ExpectInitialPowerState:
            if ((pOutOption->initialPmState = GetPmState(argv[i])) == nn::psc::PmState_Unknown)
            {
                NN_LOG("Error: failed to parse argument %d (%s).\n", i, argv[i]);
                return false;
            }
            parseState = ParseState_Default;
            break;

        case ParseState_ExpectPowerState:
            if ((pOutOption->instructions[instructionIndex].state = GetPmState(argv[i])) == nn::psc::PmState_Unknown)
            {
                NN_LOG("Error: failed to parse argument %d (%s).\n", i, argv[i]);
                return false;
            }
            instructionIndex++;
            parseState = ParseState_Default;
            break;

        case ParseState_ExpectWaitTime:
            if ((pOutOption->instructions[instructionIndex].time = atoi(argv[i])) <= 0)
            {
                NN_LOG("Error: failed to parse argument %d (%s).\n", i, argv[i]);
                return false;
            }
            instructionIndex++;
            parseState = ParseState_Default;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

    }

    pOutOption->instructionCount = instructionIndex;

    if (parseState != ParseState_Default)
    {
        NN_LOG("Error: the end of argument is illegal.\n");
        return false;
    }

    return true;
}

extern "C"
void nndiagStartup()
{
}

extern "C" void nnMain()
{
    nn::Result result;
    nn::psc::PmControl control;
    ProgramOption option;

    // コマンドライン引数をパースして、option に格納
    int    argc = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();
    if (!ParseProgramOption(&option, argc, argv))
    {
        return;
    }

    result = control.Initialize(nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // option に含まれる instruction を順番に処理する
    nn::psc::PmState currentPmState = option.initialPmState;
    for (int i = 0; i < option.instructionCount; i++)
    {
        const nn::psc::PmState nextPmState = option.instructions[i].state;

        switch (option.instructions[i].type)
        {
        case InstructionType_Transition:
            nn::psc::PmTransitionOrder order;
            if (!GetPmTransitionOrder(&order, currentPmState, nextPmState))
            {
                NN_LOG("Error: GetPmTransitionOrder() failed.\n");
                break;
            }

            result = control.DispatchRequest(nextPmState, nn::psc::MakeNoPmFlags(), order);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            if (!control.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(10)))
            {
                NN_LOG("Dispatch Request Timed Out!\n");
                control.Cancel();
                control.PrintModuleInformation();
            }
            else
            {
                control.GetEventPointer()->Clear();
            }

            currentPmState = nextPmState;
            break;

        case InstructionType_Wait:
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(option.instructions[i].time));
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // result from dispatch operation
    result = control.GetResult();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    control.Finalize();
}

} // ~nnt
} // ~psc
