﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/dd.h>

#include <nn/pinmux/pinmux.h>
#include <nn/pinmux/pinmux_ApiForHardwareTest.h>

#include <nne/pinmux/pinmux.h>

#include <nnt/nntest.h>

#define ADDITIONAL_SLEEP_FOR_AVOIDING_TIMEOUT

namespace
{

// UartDHiz の条件
const nn::Bit32 UartDHiz = 1 << nne::pinmux::tegra::PinmuxPadBitOffset_Tristate  |
                           1 << nne::pinmux::tegra::PinmuxPadBitOffset_EInput |
                           0 << nne::pinmux::tegra::PinmuxPadBitOffset_Pupd;
const nn::Bit32 UartDHizMask = nne::pinmux::tegra::PinmuxPadMask_Tristate  |
                               nne::pinmux::tegra::PinmuxPadMask_EInput    |
                               nne::pinmux::tegra::PinmuxPadMask_Pupd;

void CheckRegister(nn::Bit32 expectValue, nn::Bit32 mask, nne::pinmux::tegra::tx1::PinmuxPadIndex index)
{
    auto reg = nne::pinmux::tegra::ReadRegPinmuxPad(index);
    EXPECT_EQ(expectValue, mask & reg);
}


}

//-----------------------------------------------------------------------------
// Pinmux ライブラリの 基盤検査向け API のテスト
// UARTD を disable にしてしまうため、自動テストには加えない
TEST(PinmuxForHardwareTestTest, UartDSwitch)
{
#if (defined ADDITIONAL_SLEEP_FOR_AVOIDING_TIMEOUT)
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000LL));
#endif

    // テスト用にテストの開始時の UARTD を取得しておく
    nne::pinmux::tegra::Initialize();
    auto uart4TxReg = nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Tx);
    auto uart4RxReg = nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rx);
    auto uart4RtsReg = nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rts);
    auto uart4CtsReg = nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Cts);

    nn::pinmux::PinmuxSession uartDSession;

    // Pinmux ドライバの Initialize が必要
    nn::pinmux::Initialize();

    // 基盤検査のためにセッションをオープンする
    nn::pinmux::OpenSessionForHardwareTest(&uartDSession, nn::pinmux::AssignablePinGroupNameForHardwareTest_UartD);

    // 起動時に UART として enable になっているピン設定を HiZ に変更する
    NN_LOG("UARTD : Disable (HiZ) \n");
    nn::pinmux::SetPinAssignmentForHardwareTest(&uartDSession, nn::pinmux::PinAssignmentForHardwareTest_UartDHiZ);

    // レジスタの値が思った通りになっているか直読みする
    CheckRegister(UartDHiz, UartDHizMask, nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Tx);
    CheckRegister(UartDHiz, UartDHizMask, nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rx);
    CheckRegister(UartDHiz, UartDHizMask, nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rts);
    CheckRegister(UartDHiz, UartDHizMask, nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Cts);

    // レジスタを元に戻す
    NN_LOG("UARTD : Enable\n");
    nn::pinmux::SetPinAssignmentForHardwareTest(&uartDSession, nn::pinmux::PinAssignmentForHardwareTest_UartDEnable);

    // 値が一致しているかチェック
    EXPECT_EQ(uart4TxReg, nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Tx));
    EXPECT_EQ(uart4RxReg, nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rx));
    EXPECT_EQ(uart4RtsReg, nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rts));
    EXPECT_EQ(uart4CtsReg, nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Cts));

    // 終了時にはクローズ→ファイナライズする (クローズしてもピンの設定はそのまま)
    nn::pinmux::CloseSessionForHardwareTest(&uartDSession);

    nn::pinmux::Finalize();
}

