﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>

#include <nn/fgm/fgm.h>

// initialize our heap
const size_t HeapSize = 8 * 1024 * 1024;
const size_t BlockSize = 8 * 1024 * 1024;

// The default nninitStartup() function automatically allocates the maximum usable memory heap when the program is started
// use our own for memory management
extern "C" void nninitStartup()
{
    // allocate the memory heap
    NN_LOG("SetMemoryHeapSize initialization\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(HeapSize));

    // use a block of the allocated memory heap
    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, BlockSize));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), BlockSize);
}

struct ConfigInfo
{
    nn::fgm::Setting cpu;
    nn::fgm::Setting gpu;
    nn::fgm::Setting emc;
};

const ConfigInfo configInfoList[] =
{
    // cpu, gpu, emc
    { 1020000000, 768000000, 1600000000 },
    { 1020000000, 307200000, 1331200000 },
    { 1020000000, 384000000, 1331200000 },
};

const int NUM_CONFIGS = sizeof(configInfoList) / sizeof(configInfoList[0]);

bool VerifynnResult(nn::Result result)
{
    if (nn::fgm::ResultNotOperational::Includes(result))
        return true;

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return false;
}

//
// Updates the clock rate
//
void CreateRequestSession(nn::fgm::Setting cpu, nn::fgm::Setting gpu, nn::fgm::Setting emc)
{
    nn::fgm::Request requestCpu;
    nn::fgm::Request requestGpu;
    nn::fgm::Request requestEmc;
    nn::fgm::Setting actualCpu = 0;
    nn::fgm::Setting actualGpu = 0;
    nn::fgm::Setting actualEmc = 0;

    bool rejected = false;

    NN_LOG("Setting clock rate to %u Hz/%u Hz/%u Hz\n", cpu, gpu, emc);

    // Initialize the request with the highest priority
    rejected |= VerifynnResult(requestCpu.Initialize(nn::fgm::Module_Cpu, 0));
    rejected |= VerifynnResult(requestGpu.Initialize(nn::fgm::Module_Gpu, 0));
    rejected |= VerifynnResult(requestEmc.Initialize(nn::fgm::Module_Emc, 0));

    // Make the request
    rejected |= VerifynnResult(requestCpu.SetAndWait(cpu, cpu));
    rejected |= VerifynnResult(requestGpu.SetAndWait(gpu, gpu));
    rejected |= VerifynnResult(requestEmc.SetAndWait(emc, emc));

    // Make sure the request happened
    rejected |= VerifynnResult(requestCpu.Get(&actualCpu));
    rejected |= VerifynnResult(requestGpu.Get(&actualGpu));
    rejected |= VerifynnResult(requestEmc.Get(&actualEmc));

    NN_LOG("Actual: %u Hz/%u Hz/%u Hz\n", actualCpu, actualGpu, actualEmc);

    if (!rejected)
    {
        NN_ASSERT(actualCpu == cpu);
        NN_ASSERT(actualGpu == gpu || actualGpu == 0); // In certain scenarios, the returned value for the gpu will be 0 (see SIGLONTD-9821)
        NN_ASSERT(actualEmc == emc);
    }

    VerifynnResult(requestCpu.Finalize());
    VerifynnResult(requestGpu.Finalize());
    VerifynnResult(requestEmc.Finalize());
}

extern "C" void nnMain()
{
    while (true)
    {
        int i = rand() % NUM_CONFIGS;

        CreateRequestSession(configInfoList[i].cpu, configInfoList[i].gpu, configInfoList[i].emc);

        int sleepTime = (rand() % 1501 + 500); // Random number between 0.5 and 4, biased toward 1
        if (sleepTime > 1000)
            sleepTime = sleepTime * sleepTime / 1000;

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepTime));
    }
}
