﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Assert.h>

#include <nn/os.h>

#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>

namespace {

int g_StubInitialized = 0;

bool g_AcOk = false;

nn::os::SystemEventType g_AcOkRising;

struct StaticMutex
{
    ::nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&mutex);
    }
};

StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

} // namespace;

namespace nn { namespace bpc {

void InitializeBoardPowerControl() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    if ( g_StubInitialized == 0 )
    {
        nn::os::CreateSystemEvent(&g_AcOkRising, nn::os::EventClearMode_ManualClear, true);
    }

    g_StubInitialized++;
}

void FinalizeBoardPowerControl() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized > 0);

    g_StubInitialized--;

    if ( g_StubInitialized == 0 )
    {
        nn::os::DestroySystemEvent(&g_AcOkRising);
    }
}

nn::Result GetAcOk(bool* pOutAcOk) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized > 0);

    NN_ASSERT_NOT_NULL(pOutAcOk);

    *pOutAcOk = g_AcOk;

    NN_RESULT_SUCCESS;
}

nn::Result GetBoardPowerControlEvent(nn::os::SystemEventType* pOutEvent, BoardPowerControlEventTarget target) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized > 0);

    NN_ASSERT_NOT_NULL(pOutEvent);

    switch ( target )
    {
    case nn::bpc::BoardPowerControlEventTarget_AcOkRising:
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, nn::os::GetReadableHandleOfSystemEvent(&g_AcOkRising), false, nn::os::EventClearMode_ManualClear);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

}} // namespace nne::bpc

namespace nnt { namespace bpc {

void SetAcOk(bool acOk) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized > 0);

    bool rising = (!g_AcOk && acOk);

    g_AcOk = acOk;

    if ( rising )
    {
        nn::os::SignalSystemEvent(&g_AcOkRising);
    }
}

}} // namespace nnt::bpc
