﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/account.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/os.h>
#include <nn/pdm.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Fs.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_PlayEventFactory.h>
#include <nn/pdm/detail/pdm_SaveDataCommitThread.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/pdm/pdm_QueryApiForDebug.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/pdm/pdm_QueryLastPlayTimeApi.h>
#include <nn/time.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "testPdm_Common.h"

using namespace nn;
using namespace nn::pdm;

namespace t = nnt::pdm;

#define PEB pdm::detail::PlayEventBuffer::GetInstance()

namespace {

    void ClearAllEvents() NN_NOEXCEPT
    {
        PEB.Add(pdm::detail::MakePowerStateChangeEvent(PowerStateChangeEventType::On)); // AccountPlayEvent の Runtime Condition クリア
        pdm::detail::AccountPlayEventProvider::GetInstance().Discard();
        PEB.Clear();
    }

    int AddEvents(std::initializer_list<PlayEvent> list) NN_NOEXCEPT
    {
        int count = 0;
        for( const auto& p : list )
        {
            PEB.Add(p);
            count++;
        }
        pdm::detail::AccountPlayEventProvider::GetInstance().Flush();
        PEB.Flush();
        return count;
    }

    const uint32_t  TestVersion = 0;
    const nn::ncm::ApplicationId HomeMenuId = nn::ncm::ApplicationId{ 0x0100000000001000ull };
    const nn::ncm::ApplicationId PselId = nn::ncm::ApplicationId{ 0x0100000000001007ull };
    const nn::ncm::ApplicationId OverlayAppletId = nn::ncm::ApplicationId{ 0x010000000000100Cull };
    const nn::ncm::ApplicationId TestApplicationId1 = nn::ncm::ApplicationId{ 0x1111111111111111ull };
    const nn::ncm::ApplicationId TestApplicationId2 = nn::ncm::ApplicationId{ 0x2222222222222222ull };
    const nn::ncm::ApplicationId TestApplicationId3 = nn::ncm::ApplicationId{ 0x3333333333333333ull };
    const nn::ncm::ApplicationId TestApplicationIdList[] = {
        TestApplicationId1,
        TestApplicationId2,
        TestApplicationId3,
    };
    const nn::ncm::ProgramId InvalidProgramId = nn::ncm::ProgramId{ 0ull };
    const nn::ncm::ProgramId TestProgramId1 = nn::ncm::ProgramId{ 0xAAAAAAAAAAAAAAAAull };
    account::Uid TestUsers[] = {account::InvalidUid, account::InvalidUid};

    class PdmQueryTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            ClearAllEvents();
        }
        virtual void TearDown()
        {

        }

        static void SetUpTestCase()
        {
            account::InitializeForAdministrator();
            pdm::InitializeForQuery();

            int userCount;
            account::Uid uids[account::UserCountMax];
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&userCount, uids, NN_ARRAY_SIZE(uids)));
            for( int i = 0; i < userCount; i++ )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::DeleteUser(uids[i]));
            }

            for (int i = 0; i < NN_ARRAY_SIZE(TestUsers); ++i)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::BeginUserRegistration(&TestUsers[i]));
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::CompleteUserRegistrationForcibly(TestUsers[i]));
            }
            pdm::detail::AccountPlayEventProvider::GetInstance().Initialize(TestUsers, NN_ARRAY_SIZE(TestUsers), pdm::detail::MigrationState::Done);
        }

        static void TearDownTestCase()
        {
            for (int i = 0; i < NN_ARRAY_SIZE(TestUsers); ++i)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::DeleteUser(TestUsers[i]));
            }
            pdm::detail::AccountPlayEventProvider::GetInstance().UpdateUserRegistry(nullptr, 0);
            pdm::FinalizeForQuery();
        }
    };

    // 起動 → オープン → クローズ → 終了のイベントを追加して InOpen アカウントイベントと InFocus アカウントイベントを追加する。
    void AddAccountPlayEvent(const ncm::ApplicationId& applicationId, const nn::account::Uid& uid) NN_NOEXCEPT
    {
        AddEvents({
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, applicationId, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, applicationId, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, uid, 0, 0, 0),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, uid, 100, 100, 100),
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, applicationId, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 100, 100, 100),
        });
    }
}

TEST_F(PdmQueryTest, Application_0)
{
    // 起動終了の繰り返し。

    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 100, 0, 100),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId2, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId2, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 400, 0, 400),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 500, 0, 500),
    });

    const int EventQueryNum = 50;
    pdm::ApplicationEvent appEvent[EventQueryNum];

    {
        auto count = pdm::QueryApplicationEvent(appEvent, EventQueryNum, 0);
        ASSERT_EQ(6, count);

        EXPECT_EQ(TestApplicationId1.value, appEvent[0].applicationId.value);
        EXPECT_EQ(TestApplicationId1.value, appEvent[1].applicationId.value);
        EXPECT_EQ(TestApplicationId2.value, appEvent[2].applicationId.value);
        EXPECT_EQ(TestApplicationId2.value, appEvent[3].applicationId.value);
        EXPECT_EQ(TestApplicationId1.value, appEvent[4].applicationId.value);
        EXPECT_EQ(TestApplicationId1.value, appEvent[5].applicationId.value);

        for( int i = 0; i < 6; i++ )
        {
            EXPECT_EQ(appEvent[i].eventTime.eventIndex * 100, appEvent[i].eventTime.userClockTime);
        }
    }

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];
    pdm::ApplicationPlayStatistics applicationPlayStatistics[StatisticsQueryNum];

    {
        auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum);
        ASSERT_EQ(2, count);

        NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 2, 200, 0, 0, 0, 5, 500, 0);
        NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[1], TestApplicationId2, 1, 100, 2, 200, 0, 3, 300, 0);
    }

    {
        auto count = pdm::QueryApplicationPlayStatisticsForSystem(applicationPlayStatistics, TestApplicationIdList, NN_ARRAY_SIZE(TestApplicationIdList));

        EXPECT_EQ(3, count);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[0], TestApplicationId1, 2, 200);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[1], TestApplicationId2, 1, 100);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[2], TestApplicationId3, 0, 0);
    }
}

TEST_F(PdmQueryTest, Application_1)
{
    // 途中でライブラリアプレットを起動（アプリケーションのイベントからは見えない）
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 100, 0, 100),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestProgramId1, TestVersion, nn::applet::AppletId_LibraryAppletError, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestProgramId1, TestVersion, nn::applet::AppletId_LibraryAppletError, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 400, 0, 400),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 500, 0, 500),
    });

    const int EventQueryNum = 50;
    pdm::ApplicationEvent appEvent[EventQueryNum];

    auto count = pdm::QueryApplicationEvent(appEvent, EventQueryNum, 0);
    ASSERT_EQ(4, count);

    EXPECT_EQ(0, appEvent[0].eventTime.eventIndex);
    EXPECT_EQ(1, appEvent[1].eventTime.eventIndex);
    EXPECT_EQ(4, appEvent[2].eventTime.eventIndex);
    EXPECT_EQ(5, appEvent[3].eventTime.eventIndex);

    for( int i = 0; i < count; i++ )
    {
        EXPECT_EQ(TestApplicationId1.value, appEvent[i].applicationId.value);
        EXPECT_EQ(appEvent[i].eventTime.eventIndex * 100, appEvent[i].eventTime.userClockTime);
    }

    // offset を指定してのクエリ。
    count = pdm::QueryApplicationEvent(appEvent, EventQueryNum, 2);
    ASSERT_EQ(2, count);
    EXPECT_EQ(4, appEvent[0].eventTime.eventIndex);
    EXPECT_EQ(5, appEvent[1].eventTime.eventIndex);

    for( int i = 0; i < count; i++ )
    {
        EXPECT_EQ(TestApplicationId1.value, appEvent[i].applicationId.value);
        EXPECT_EQ(appEvent[i].eventTime.eventIndex * 100, appEvent[i].eventTime.userClockTime);
    }

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];
    pdm::ApplicationPlayStatistics applicationPlayStatistics[StatisticsQueryNum];

    count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 200, 0, 0, 0, 5, 500, 0);

    {
        auto appCount = pdm::QueryApplicationPlayStatisticsForSystem(applicationPlayStatistics, TestApplicationIdList, NN_ARRAY_SIZE(TestApplicationIdList));
        ASSERT_EQ(3, appCount);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[0], TestApplicationId1, 1, 200);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[1], TestApplicationId2, 0, 0);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[2], TestApplicationId3, 0, 0);
    }
}

TEST_F(PdmQueryTest, Account_0)
{
    // 起動 → オープン → クローズ → 終了
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 0),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 100, 1, 100, 0, 1, 200, 0);

    count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[1]);
    EXPECT_EQ(0, count);

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_TRUE(user0Statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*user0Statistics), TestApplicationId1, 1, 100, 1, 100, 0, 1, 200, 0);

    auto user1Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[1]);
    ASSERT_FALSE(user1Statistics);
}

TEST_F(PdmQueryTest, Account_1)
{
    // 起動 → オープン → 中断 → 再開 → クローズ → 終了
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 125, 0, 125),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 175, 0, 175),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 50, 0, 100, 0, 2, 200, 0);
}

TEST_F(PdmQueryTest, Account_2)
{
    // 起動 → 中断 → 再開 → オープン → クローズ → 終了
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 100, 0, 100),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 150, 0, 150),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 200, 0, 200),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 250, 0, 250),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 50, 1, 200, 0, 1, 250, 0);
}

TEST_F(PdmQueryTest, Account_3)
{
    // 起動 → オープン → クローズ → オープン → クローズ → 終了
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 150, 0, 150),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 200, 0, 200),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 250, 0, 250),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 2, 100, 1, 100, 0, 3, 250, 0);
}

TEST_F(PdmQueryTest, Account_4)
{
    // 起動(1) → オープン → クローズ → 終了(1) → 起動(2) → オープン → クローズ → 終了(2) → 起動(1) → オープン → クローズ → 終了(1)
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 150, 0, 150),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),

        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId2, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId2, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 400, 0, 400),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 450, 0, 450),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId2, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 500, 0, 500),

        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 600, 0, 600),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 600, 0, 600),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 700, 0, 700),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 750, 0, 750),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 800, 0, 800),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(2, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 2, 100, 1, 100, 0, 5, 750, 0);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[1], TestApplicationId2, 1, 50, 3, 400, 0, 3, 450, 0);
}

TEST_F(PdmQueryTest, Account_5)
{
    // 起動 → オープン(1) → オープン(2) → クローズ(1) → クローズ(2) → 終了
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[1], 125, 0, 125),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 150, 0, 150),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[1], 175, 0, 175),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 50, 1, 100, 0, 1, 150, 0);

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_TRUE(user0Statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*user0Statistics), TestApplicationId1, 1, 50, 1, 100, 0, 1, 150, 0);

    count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[1]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 50, 1, 125, 0, 1, 175, 0);

    auto user1Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[1]);
    ASSERT_TRUE(user1Statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*user1Statistics), TestApplicationId1, 1, 50, 1, 125, 0, 1, 175, 0);
}

TEST_F(PdmQueryTest, Account_6)
{
    // 起動 → オープン(1) → オープン(2) → クローズ(2) → クローズ(1)  → 終了
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[1], 125, 0, 125),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[1], 150, 0, 150),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 175, 0, 175),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 75, 1, 100, 0, 1, 175, 0);

    count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[1]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 25, 1, 125, 0, 1, 150, 0);
}

TEST_F(PdmQueryTest, Account_7)
{
    // 起動 → オープン → 起動 → オープン → クローズ → 終了
    // プレイ回数の記録単位は、アカウントデータベース移行に伴い、記録単位がOpen期間を１回とするため、Closeを伴わない、最初の Open はプレイ回数には反映されません。
    // また同様にプレイ時間も InFocus & Open期間中となるため、最初の Open からの期間は反映されません。
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        pdm::detail::MakePowerStateChangeEvent(PowerStateChangeEventType::On),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 300, 0, 300),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 400, 0, 400),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 500, 0, 500),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 100, 1, 300, 0, 1, 400, 0);
}

TEST_F(PdmQueryTest, Account_8)
{
    // 起動 → オープン → 中断 → 終了 → クローズ
    // オープン → 中断までが加算される。
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 300),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 300, 0, 300),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 100, 0, 100, 0, 0, 200, 0);
}

TEST_F(PdmQueryTest, Account_9)
{
    // 起動 → オープン → 終了 → クローズ
    // 終了とクローズは、記録するモジュールが異なるので前後する可能性がある。クローズが終了の後に回った場合にも動作することを確認する。
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 200, 0, 200),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 201, 0, 201),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 100, 0, 100, 0, 0, 200, 0);

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_TRUE(user0Statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*user0Statistics), TestApplicationId1, 1, 100, 0, 100, 0, 0, 200, 0);
}

TEST_F(PdmQueryTest, Account_10)
{
    // 起動 → オープン → クローズ。通常はありえないシーケンス。InOpen のみが記録される。
    // InOpen のみが記録されている場合、PlayStatistics は無効にする。
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 200, 0, 200),
    });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(count, 0);

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_FALSE(user0Statistics);
}

TEST_F(PdmQueryTest, Account_11)
{
    // 起動 → オープン → LA起動(AllForeground) → LA終了 → クローズ → 終了
    // LA(AllForeground)の時間もアカウントのプレイ情報に含まれる。
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 110, 0, 110),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::Launch, TestApplicationId1, applet::LibraryAppletMode_AllForeground, applet::AppletId_LibraryAppletError, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 110, 0, 110),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::InFocus, TestApplicationId1, applet::LibraryAppletMode_AllForeground, applet::AppletId_LibraryAppletError, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 110, 0, 110),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::OutOfFocus, TestApplicationId1, applet::LibraryAppletMode_AllForeground, applet::AppletId_LibraryAppletError, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 120, 0, 120),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::Exit, TestApplicationId1, applet::LibraryAppletMode_AllForeground, applet::AppletId_LibraryAppletError, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 120, 0, 120),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 120, 0, 120),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 0),
        });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 100, 0, 100, 0, 3, 200, 0);

    count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[1]);
    EXPECT_EQ(0, count);

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_TRUE(user0Statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*user0Statistics), TestApplicationId1, 1, 100, 0, 100, 0, 3, 200, 0);

    auto user1Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[1]);
    ASSERT_FALSE(user1Statistics);
}

TEST_F(PdmQueryTest, Account_12)
{
    // 起動 → オープン → LA起動(PartialForeground) → LA終了 → クローズ → 終了
    // LA(PartialForeground)の時間はアカウントのプレイ情報に含まれない。
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 100, 0, 100),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::Launch, TestApplicationId1, applet::LibraryAppletMode_PartialForeground, applet::AppletId_LibraryAppletSwkbd, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 110, 0, 110),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::InFocus, TestApplicationId1, applet::LibraryAppletMode_PartialForeground, applet::AppletId_LibraryAppletSwkbd, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 110, 0, 110),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::OutOfFocus, TestApplicationId1, applet::LibraryAppletMode_PartialForeground, applet::AppletId_LibraryAppletSwkbd, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 120, 0, 120),
        t::MakeLibraryAppletEventInMinutes(pdm::AppletEventType::Exit, TestApplicationId1, applet::LibraryAppletMode_PartialForeground, applet::AppletId_LibraryAppletSwkbd, ncm::StorageId::BuiltInSystem, ns::PlayLogPolicy::LogOnly, 120, 0, 120),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 200, 0, 200),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 300, 0, 0),
        });

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 100, 2, 100, 0, 2, 200, 0);

    count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[1]);
    EXPECT_EQ(0, count);

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_TRUE(user0Statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*user0Statistics), TestApplicationId1, 1, 100, 2, 100, 0, 2, 200, 0);

    auto user1Statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[1]);
    ASSERT_FALSE(user1Statistics);
}

TEST_F(PdmQueryTest, SIGLO_83033_Before)
{
    // アプリの InFocus がアカウントのオープンの「前」にくる
    const auto user = TestUsers[0];

    AddEvents({
        // アプリ起動
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, HomeMenuId, TestVersion, nn::applet::AppletId_SystemAppletMenu, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        // psel を起動してユーザーアカウントを選択・オープン
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, user, 0, 0, 0),
        // psel を起動してユーザーアカウントを選択・クローズ
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, user, 1, 1, 1),
        // アプリ終了
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, HomeMenuId, TestVersion, nn::applet::AppletId_SystemAppletMenu, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
    });

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, user);
    ASSERT_TRUE(user0Statistics);
    EXPECT_EQ(1u, user0Statistics->totalPlayCount);
    EXPECT_EQ(1u, user0Statistics->totalPlayTime);
}

TEST_F(PdmQueryTest, SIGLO_83033_After)
{
    // アプリの InFocus がアカウントのオープンの「後」にくる
    const auto user = TestUsers[0];

    AddEvents({
        // アプリ起動
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, HomeMenuId, TestVersion, nn::applet::AppletId_SystemAppletMenu, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        // psel を起動してユーザーアカウントを選択・オープン
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, user, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        // psel を起動してユーザーアカウントを選択・クローズ
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, PselId, TestVersion, nn::applet::AppletId_LibraryAppletPlayerSelect, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, user, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
        // アプリ終了
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, HomeMenuId, TestVersion, nn::applet::AppletId_SystemAppletMenu, nn::ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::LogOnly, 1, 1, 1),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 1, 1, 1),
    });

    auto user0Statistics = pdm::QueryPlayStatistics(TestApplicationId1, user);
    ASSERT_TRUE(user0Statistics);
    EXPECT_EQ(1u, user0Statistics->totalPlayCount);
    EXPECT_EQ(1u, user0Statistics->totalPlayTime);
}

TEST_F(PdmQueryTest, AppletSequenceSample)
{
    // アプリ起動 → アカウントオープン → 途中でエラーアプレット起動 → アプリに戻って終了（明示的なアカウントクローズはなし）
    const uint32_t ExpectTotalEvents = static_cast<uint32_t>(AddEvents({
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 0, 10),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 0, 10),
        t::MakeAppletEventInMinutes(AppletEventType::OutOfFocus, InvalidProgramId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 10, 0, 10),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 0, 10),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 10, 0, 10),
        t::MakeAppletEventInMinutes(AppletEventType::Launch, InvalidProgramId, TestVersion, applet::AppletId_LibraryAppletError, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, InvalidProgramId, TestVersion, applet::AppletId_LibraryAppletError, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeAppletEventInMinutes(AppletEventType::OutOfFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, InvalidProgramId, TestVersion, applet::AppletId_LibraryAppletError, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeAppletEventInMinutes(AppletEventType::Background, InvalidProgramId, TestVersion, applet::AppletId_LibraryAppletError, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeAppletEventInMinutes(AppletEventType::Exit, InvalidProgramId, TestVersion, applet::AppletId_LibraryAppletError, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 30, 0, 30),
        t::MakeAppletEventInMinutes(AppletEventType::Background, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 30, 0, 30),
        t::MakeAppletEventInMinutes(AppletEventType::Exit, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 30, 0, 30),
    }));

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 20, 0, 10, 0, ExpectTotalEvents - 1, 30, 0);

    count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 20, 0, 10, 0, 2, 30, 0);

    auto statistics = pdm::QueryPlayStatistics(TestApplicationId1);
    ASSERT_TRUE(statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*statistics), TestApplicationId1, 1, 20, 0, 10, 0, ExpectTotalEvents - 1, 30, 0);

    statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_TRUE(statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*statistics), TestApplicationId1, 1, 20, 0, 10, 0, 2, 30, 0);

    {
        pdm::ApplicationPlayStatistics applicationPlayStatistics[5];
        auto appCount = pdm::QueryApplicationPlayStatisticsForSystem(applicationPlayStatistics, TestApplicationIdList, NN_ARRAY_SIZE(TestApplicationIdList));
        EXPECT_EQ(3, appCount);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[0], TestApplicationId1, 1, 20);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[1], TestApplicationId2, 0, 0);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[2], TestApplicationId3, 0, 0);
    }

} // NOLINT(impl/function_size)

TEST_F(PdmQueryTest, ForceShutdownSequence)
{
    // 強制電源断を跨いでプレイ時間が加算されないことを確認。
    // アプリ起動 → アカウントオープン → HOME メニュー遷移 → アプリに戻る → (強制電源断) → 起動 → 再度アプリ起動 → HOME メニューに遷移 → アプリ終了
    const uint32_t ExpectTotalEvents = static_cast<uint32_t>(AddEvents({
        // アプリ起動
        t::MakeAppletEventInMinutes(AppletEventType::OutOfFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 10, 0, 10),
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 0, 10),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 0, 10),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 10, 0, 10),
        // HOMEに遷移
        t::MakeAppletEventInMinutes(AppletEventType::Background, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 20, 0, 20),
        // アプリに戻る
        t::MakeAppletEventInMinutes(AppletEventType::OutOfFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 30, 0, 30),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 30, 0, 30),
        // (強制電源断 +) 起動
        pdm::detail::MakePowerStateChangeEvent(PowerStateChangeEventType::On),  // 電源再投入に伴う電源投入イベント。
        t::MakeAppletEventInMinutes(AppletEventType::Launch, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 110, 0, 110),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 110, 0, 110),
        t::MakeAppletEventInMinutes(AppletEventType::OutOfFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 110, 0, 110),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 110, 0, 110),
        t::MakeAppletEventInMinutes(AppletEventType::Launch, OverlayAppletId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 110, 0, 110),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, OverlayAppletId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 110, 0, 110),
        // アプリ起動
        t::MakeAppletEventInMinutes(AppletEventType::OutOfFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 120, 0, 120),
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 120, 0, 120),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 120, 0, 120),
        // HOMEに遷移
        t::MakeAppletEventInMinutes(AppletEventType::Background, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 130, 0, 130),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, HomeMenuId, TestVersion, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem, nn::ns::PlayLogPolicy::All, 130, 0, 130),
        // アプリ終了
        t::MakeAppletEventInMinutes(AppletEventType::Background, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 130, 0, 130),
        t::MakeAppletEventInMinutes(AppletEventType::Exit, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 130, 0, 130),
    }));

    const int StatisticsQueryNum = 1;
    pdm::PlayStatistics playStatistics[1];

    auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum);
    ASSERT_EQ(1, count);
    NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 2, 20, 1, 10, 0, ExpectTotalEvents - 1, 130, 0);

    auto statistics = pdm::QueryPlayStatistics(TestApplicationId1);
    ASSERT_TRUE(statistics);
    NNT_PDM_EXPECT_PLAYSTAT_EQ((*statistics), TestApplicationId1, 2, 20, 1, 10, 0, ExpectTotalEvents - 1, 130, 0);

    // アカウントDB経由の場合は以下仕様となるため、本テストのイベントシーケンスでは「プレイ回数は0」になります。
    //  - GhostPlay (Closeを伴わない InFocus期間) はプレイ回数に含まれない。
    statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
    ASSERT_FALSE(statistics);

    // QueryPlayStatistics( AppId, Uid )をQueryPlayStatistics(List<PlayStatistics>, count, Uid) で代用した実装。
    // アカウントDB経由の場合は以下仕様となるため、本テストのイベントシーケンスでは「プレイ回数は0」になります。
    //  - GhostPlay (Closeを伴わない InFocus期間) はプレイ回数に含まれない。
    statistics = util::nullopt;
    pdm::PlayStatistics pTestBuffer[16];
    const auto foundCount = pdm::QueryPlayStatistics(pTestBuffer, NN_ARRAY_SIZE(pTestBuffer), TestUsers[0]);
    for (int i = 0; i < foundCount; ++i)
    {
        const auto& entry = pTestBuffer[i];
        if (entry.applicationId == TestApplicationId1)
        {
            statistics = entry;
            break;
        }
    }
    ASSERT_FALSE(statistics);

} // NOLINT(impl/function_size)

TEST_F(PdmQueryTest, QueryPlayStatistics_PowerOffWhileApplicationIsAlive)
{
    // SIGLO-81667
    // アプリが生きている間に電源オフされた場合、電源オフまでの時間をプレイ時間に追加する。
    // 2種類のアプリでイベント登録しているのは間違って別のアプリの情報を補正していないかの確認のため。
    AddEvents({
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakePowerStateChangeEventInMinutes(PowerStateChangeEventType::Off, 10, 10, 10),
        t::MakePowerStateChangeEventInMinutes(PowerStateChangeEventType::On, 10, 10, 10),
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId2, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 20, 20, 20),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId2, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 20, 20, 20),
        t::MakePowerStateChangeEventInMinutes(PowerStateChangeEventType::Off, 40, 40, 40),
    });

    {
        pdm::PlayStatistics playStatistics[2];

        auto count = pdm::QueryPlayStatistics(playStatistics, 2);
        ASSERT_EQ(2, count);
        NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 10, 0, 0, 0, 2, 10, 10);
        NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[1], TestApplicationId2, 1, 20, 4, 20, 20, 6, 40, 40);
    }

    {
        auto playStatistics = pdm::QueryPlayStatistics(TestApplicationId1);
        ASSERT_TRUE(playStatistics);
        NNT_PDM_EXPECT_PLAYSTAT_EQ((*playStatistics), TestApplicationId1, 1, 10, 0, 0, 0, 2, 10, 10);

        auto playStatistics2 = pdm::QueryPlayStatistics(TestApplicationId2);
        ASSERT_TRUE(playStatistics2);
        NNT_PDM_EXPECT_PLAYSTAT_EQ((*playStatistics2), TestApplicationId2, 1, 20, 4, 20, 20, 6, 40, 40);
    }

    {
        pdm::ApplicationPlayStatistics applicationPlayStatistics[2];

        auto count = pdm::QueryApplicationPlayStatisticsForSystem(applicationPlayStatistics, TestApplicationIdList, 2);
        ASSERT_EQ(2, count);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[0], TestApplicationId1, 1, 10);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[1], TestApplicationId2, 1, 20);
    }
}

TEST_F(PdmQueryTest, QueryPlayStatistics_PowerOffWhileApplicationIsAlive_ForceShutdown)
{
    // SIGLO-81667
    // アプリが生きている間に電源オフされた場合、電源オフまでの時間をプレイ時間に追加する、
    // という仕様が、アプリ起動中に強制電源断（12秒押しで電源オフが記録できない） → 次回起動時にアプリを起動せずに正常電源断、
    // というパターンで誤って電源オフ期間をプレイ時間に含める補正をしてしまわないことを確認。この場合は終了時刻が取れないので時間を補正しない。
    // プレイ回数は Launch 時点で増える。
    AddEvents({
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(AppletEventType::InFocus, TestApplicationId1, TestVersion, applet::AppletId_Application, ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        // 強制電源断のため電源オフの記録無しで電源オンが入る。
        t::MakePowerStateChangeEventInMinutes(PowerStateChangeEventType::On, 10, 10, 10),
        t::MakePowerStateChangeEventInMinutes(PowerStateChangeEventType::Off, 40, 40, 40),
    });

    {
        pdm::PlayStatistics playStatistics[1];

        auto count = pdm::QueryPlayStatistics(playStatistics, 1);
        ASSERT_EQ(1, count);
        NNT_PDM_EXPECT_PLAYSTAT_EQ(playStatistics[0], TestApplicationId1, 1, 0, 0, 0, 0, 1, 0, 0);
    }

    {
        auto playStatistics = pdm::QueryPlayStatistics(TestApplicationId1);
        ASSERT_TRUE(playStatistics);
        NNT_PDM_EXPECT_PLAYSTAT_EQ((*playStatistics), TestApplicationId1, 1, 0, 0, 0, 0, 1, 0, 0);
    }

    {
        pdm::ApplicationPlayStatistics applicationPlayStatistics[1];

        auto count = pdm::QueryApplicationPlayStatisticsForSystem(applicationPlayStatistics, TestApplicationIdList, 1);
        ASSERT_EQ(1, count);
        NNT_PDM_EXPECT_APP_PLAYSTAT_EQ(applicationPlayStatistics[0], TestApplicationId1, 1, 0);
    }
}

TEST_F(PdmQueryTest, QueryLastPlayTimeBasic)
{
    pdm::detail::SetLastSteadyClockResetPointForDebug(0, nn::TimeSpan::FromMinutes(30));
    AddEvents({
        pdm::detail::MakeSteadyClockResetEvent(),
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, TestVersion, applet::AppletId_Application,
                                    ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 10, 10),
        pdm::detail::MakePowerStateChangeEvent(PowerStateChangeEventType::On), // Launch 状態をリセットするために電源オンイベント。
        t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId2, TestVersion, applet::AppletId_Application,
                                    ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 20, 20, 20),
    });

    pdm::LastPlayTime lastPlayTimes[2];
    nn::ApplicationId applicationIds[] = {
        nn::ApplicationId{ TestApplicationId1.value },
        nn::ApplicationId{ TestApplicationId2.value },
    };

    {
        auto count = pdm::QueryLastPlayTime(lastPlayTimes, applicationIds, 2);

        ASSERT_EQ(2, count);
        NNT_PDM_EXEPCT_LASTPLAYTIME_EQ(lastPlayTimes[0], TestApplicationId1, 10, 10, true, 20);
        NNT_PDM_EXEPCT_LASTPLAYTIME_EQ(lastPlayTimes[1], TestApplicationId2, 20, 20, true, 10);
    }

    {
        // 途中で SteadyClockReset イベントが起きたことにして Query し直す。
        pdm::detail::SetLastSteadyClockResetPointForDebug(2, nn::TimeSpan::FromMinutes(30));
        auto count = pdm::QueryLastPlayTime(lastPlayTimes, applicationIds, 2);

        ASSERT_EQ(2, count);
        NNT_PDM_EXEPCT_LASTPLAYTIME_EQ(lastPlayTimes[0], TestApplicationId1, 10, 10, false, -1);
        NNT_PDM_EXEPCT_LASTPLAYTIME_EQ(lastPlayTimes[1], TestApplicationId2, 20, 20, true, 10);
    }

    {
        // 見つかるまでイベントを遡ることの確認。Add し続けるのに時間がかかるので切っておく。
#if 0
        ClearAllEvents();
        pdm::detail::SetLastSteadyClockResetPointForDebug(0, nn::TimeSpan::FromMinutes(10000));
        PEB.Add(pdm::detail::MakeSteadyClockResetEvent());
        PEB.Add(t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, TestVersion, applet::AppletId_Application,
            ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 10, 1));
        PEB.Add(t::MakeAppletEventInMinutes(AppletEventType::Exit, TestApplicationId1, TestVersion, applet::AppletId_Application,
            ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 10, 10, 1));
        for( int i = 2; i < 5000; i++ )
        {
            PEB.Add(pdm::detail::MakePowerStateChangeEvent(PowerStateChangeEventType::Awake)); // 適当なイベント。何でも良い。
        }
        PEB.Add(t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId2, TestVersion, applet::AppletId_Application,
            ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 20, 20, 9999));
        PEB.Flush();
        auto count = pdm::QueryLastPlayTime(lastPlayTimes, applicationIds, 2);

        ASSERT_EQ(2, count);
        NNT_PDM_EXEPCT_LASTPLAYTIME_EQ(lastPlayTimes[0], TestApplicationId1, 10, 10, true, 9999);
        NNT_PDM_EXEPCT_LASTPLAYTIME_EQ(lastPlayTimes[1], TestApplicationId2, 20, 20, true, 1);
#endif
    }
} // NOLINT(impl/function_size)

TEST_F(PdmQueryTest, QueryPlayEvent)
{
    const auto AddedPlayEventCount = 100;
    pdm::PlayEvent playEvents[AddedPlayEventCount];
    for( int i = 0; i < AddedPlayEventCount; i+=2 )
    {
        playEvents[i] = t::MakeAppletEventInMinutes(AppletEventType::Launch, TestApplicationId1, static_cast<uint32_t>(i), applet::AppletId_Application,
            ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, i, i, i);
        PEB.Add(playEvents[i]);
        playEvents[i + 1] = pdm::detail::MakePowerStateChangeEvent(PowerStateChangeEventType::On), // Launch 状態をリセットするために電源オンイベント。
        PEB.Add(playEvents[i + 1]);
    }

    int startIndex;
    int lastIndex;
    EXPECT_EQ(AddedPlayEventCount, GetAvailablePlayEventRange(&startIndex, &lastIndex));
    EXPECT_EQ(0, startIndex);
    EXPECT_EQ(99, lastIndex);

    const auto QueryPlayEventCountMax = AddedPlayEventCount + 1;
    pdm::PlayEvent queriedPlayEvent[QueryPlayEventCountMax];

    {
        // クエリー数 > イベント数
        auto count = QueryPlayEvent(queriedPlayEvent, QueryPlayEventCountMax, 0);
        EXPECT_EQ(AddedPlayEventCount, count);
        for( int i = 0; i < count; i++ )
        {
            EXPECT_EQ(playEvents[i], queriedPlayEvent[i]) << "Added PlayEvent and Querired PlayEvent differ at index " << i;
        }
    }

    {
        // クエリー数 < イベント数
        auto count = QueryPlayEvent(queriedPlayEvent, 10, 0);
        EXPECT_EQ(10, count);
        for( int i = 0; i < count; i++ )
        {
            EXPECT_EQ(playEvents[i], queriedPlayEvent[i]) << "Added PlayEvent and Querired PlayEvent differ at index " << i;
        }
    }

    {
        // オフセット指定
        auto offset = 50;
        auto count = QueryPlayEvent(queriedPlayEvent, 10, offset);
        EXPECT_EQ(10, count);
        for( int i = 0; i < count; i++ )
        {
            EXPECT_EQ(playEvents[i + offset], queriedPlayEvent[i]) << "Added PlayEvent and Querired PlayEvent differ at index " << i;
        }
    }

    {
        // オフセット指定（範囲外）
        auto offset = AddedPlayEventCount;
        auto count = QueryPlayEvent(queriedPlayEvent, 10, offset);
        EXPECT_EQ(0, count);
    }
}

TEST_F(PdmQueryTest, QueryAccountEvent)
{
    const auto AddedPlayEventCount = 30;
    pdm::PlayEvent playEvents[AddedPlayEventCount];

    const auto QueryAccountEventCountMax = AddedPlayEventCount;
    pdm::AccountEvent queriedAccountEvent[QueryAccountEventCountMax];

    // 全てアカウントイベント。
    {
        for( int i = 0; i < AddedPlayEventCount; i++ )
        {
            playEvents[i] = t::MakeUserAccountEventInMinutes((i % 2 == 0) ? UserAccountEventType::Open : UserAccountEventType::Close, TestUsers[0], i, i * 2, i * 3);
            PEB.Add(playEvents[i]);
        }

        auto count = QueryAccountEvent(queriedAccountEvent, QueryAccountEventCountMax, 0);
        EXPECT_EQ(AddedPlayEventCount, count);
        for( int i = 0; i < count; i++ )
        {
            EXPECT_EQ(pdm::detail::GetUid(playEvents[i]),  queriedAccountEvent[i].uid);
            EXPECT_EQ(playEvents[i].userAccountEventData.eventType, queriedAccountEvent[i].eventType);
            EXPECT_EQ(playEvents[i].userTime, queriedAccountEvent[i].eventTime.userClockTime);
            EXPECT_EQ(playEvents[i].networkTime, queriedAccountEvent[i].eventTime.networkClockTime);
            EXPECT_EQ(playEvents[i].steadyTime, queriedAccountEvent[i].eventTime.steadyClockTimeValueSinceBase);
            EXPECT_EQ(i, queriedAccountEvent[i].eventTime.eventIndex);
        }

        ClearAllEvents();
    }

    // 半分その他のイベントが混ざっている。
    {
        for( int i = 0; i < AddedPlayEventCount; i++ )
        {
            if( i % 2 == 0 )
            {
                playEvents[i] = t::MakeUserAccountEventInMinutes(UserAccountEventType::Open, TestUsers[0], i, i * 2, i * 3);
            }
            else
            {
                playEvents[i] = pdm::detail::MakePowerStateChangeEvent(PowerStateChangeEventType::On);
            }
            PEB.Add(playEvents[i]);
        }

        auto count = QueryAccountEvent(queriedAccountEvent, QueryAccountEventCountMax, 0);
        EXPECT_EQ(AddedPlayEventCount / 2, count);
        for( int i = 0; i < count; i++ )
        {
            EXPECT_EQ(pdm::detail::GetUid(playEvents[i * 2]), queriedAccountEvent[i].uid);
            EXPECT_EQ(playEvents[i * 2].userAccountEventData.eventType, queriedAccountEvent[i].eventType);
            EXPECT_EQ(playEvents[i * 2].userTime, queriedAccountEvent[i].eventTime.userClockTime);
            EXPECT_EQ(playEvents[i * 2].networkTime, queriedAccountEvent[i].eventTime.networkClockTime);
            EXPECT_EQ(playEvents[i * 2].steadyTime, queriedAccountEvent[i].eventTime.steadyClockTimeValueSinceBase);
            EXPECT_EQ(i * 2, queriedAccountEvent[i].eventTime.eventIndex);
        }

        ClearAllEvents();
    }
}

TEST_F(PdmQueryTest, AccountAdjust)
{
    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    pdm::detail::SetLastSteadyClockResetPointForDebug(0, nn::TimeSpan::FromMinutes(60));
    auto currentTimePoint = pdm::detail::GetSteadyClockTimeForPlayEvent();

    {
        // 起動 → InFocus → オープン → クエリ
        // 初プレイ時の記録が全く残っていない場合のクエリ。
        AddEvents({
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 30, 30, (currentTimePoint / 60) - 10),
        });

        {
            // 単体取得
            auto statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
            ASSERT_TRUE(statistics);
            EXPECT_EQ(1, statistics->totalPlayCount);
            EXPECT_EQ(10, statistics->totalPlayTime);

            EXPECT_EQ(0, statistics->firstEventTime.eventIndex);
            EXPECT_EQ(30, statistics->firstEventTime.userClockTime);
            EXPECT_EQ(30, statistics->firstEventTime.networkClockTime);

            // InFocus中の最後に遊んだ時刻は現在時刻
            EXPECT_EQ(0, statistics->latestEventTime.eventIndex);
            time::PosixTime currentTime;
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&currentTime));
            EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), statistics->latestEventTime.userClockTime, 1);
            if( time::StandardNetworkSystemClock::GetCurrentTime(&currentTime).IsSuccess() )
            {
                EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), statistics->latestEventTime.networkClockTime, 1);
            }
        }

        {
            // リスト取得
            auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
            ASSERT_EQ(count, 1);
            EXPECT_EQ(1, playStatistics[0].totalPlayCount);
            EXPECT_EQ(10, playStatistics[0].totalPlayTime);

            EXPECT_EQ(0, playStatistics[0].firstEventTime.eventIndex);
            EXPECT_EQ(30, playStatistics[0].firstEventTime.userClockTime);
            EXPECT_EQ(30, playStatistics[0].firstEventTime.networkClockTime);

            // InFocus中の最後に遊んだ時刻は現在時刻
            EXPECT_EQ(0, playStatistics[0].latestEventTime.eventIndex);
            time::PosixTime currentTime;
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&currentTime));
            EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), playStatistics[0].latestEventTime.userClockTime, 1u);
            if( time::StandardNetworkSystemClock::GetCurrentTime(&currentTime).IsSuccess() )
            {
                EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), playStatistics[0].latestEventTime.networkClockTime, 1u);
            }
        }

        ClearAllEvents();
    }

    {
        // 一つ記録を残してからの → オープン → クエリ
        // プレイ時間とプレイ回数、最後に遊んだ時刻が補正される。
        AddEvents({
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 5, 5, 5),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, TestUsers[0], 10, 10, 10),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 30, 30, (currentTimePoint / 60) - 10),
        });

        {
            // 単体取得
            auto statistics = pdm::QueryPlayStatistics(TestApplicationId1, TestUsers[0]);
            ASSERT_TRUE(statistics);
            EXPECT_EQ(2, statistics->totalPlayCount);
            EXPECT_EQ(15, statistics->totalPlayTime);

            EXPECT_EQ(1, statistics->firstEventTime.eventIndex);
            EXPECT_EQ(5, statistics->firstEventTime.userClockTime);
            EXPECT_EQ(5, statistics->firstEventTime.networkClockTime);

            EXPECT_EQ(0, statistics->latestEventTime.eventIndex);
            time::PosixTime currentTime;
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&currentTime));
            EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), statistics->latestEventTime.userClockTime, 1);
            if( time::StandardNetworkSystemClock::GetCurrentTime(&currentTime).IsSuccess() )
            {
                EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), statistics->latestEventTime.networkClockTime, 1);
            }
        }

        {
            // リスト取得
            auto count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, TestUsers[0]);
            ASSERT_EQ(count, 1);
            EXPECT_EQ(2, playStatistics[0].totalPlayCount);
            EXPECT_EQ(15, playStatistics[0].totalPlayTime);

            EXPECT_EQ(1, playStatistics[0].firstEventTime.eventIndex);
            EXPECT_EQ(5, playStatistics[0].firstEventTime.userClockTime);
            EXPECT_EQ(5, playStatistics[0].firstEventTime.networkClockTime);

            EXPECT_EQ(0, playStatistics[0].latestEventTime.eventIndex);
            time::PosixTime currentTime;
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&currentTime));
            EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), playStatistics[0].latestEventTime.userClockTime, 1u);
            if( time::StandardNetworkSystemClock::GetCurrentTime(&currentTime).IsSuccess() )
            {
                EXPECT_NEAR(static_cast<uint32_t>((currentTime.value - time::InputPosixTimeMin.value) / 60), playStatistics[0].latestEventTime.networkClockTime, 1u);
            }
        }

        ClearAllEvents();
    }
} // NOLINT(impl/function_size)

TEST_F(PdmQueryTest, QueryRecentlyPlayedApplication)
{
    AddAccountPlayEvent(TestApplicationId1, TestUsers[0]);
    AddAccountPlayEvent(TestApplicationId2, TestUsers[0]);
    AddAccountPlayEvent(TestApplicationId3, TestUsers[0]);

    AddAccountPlayEvent(TestApplicationId3, TestUsers[1]);
    AddAccountPlayEvent(TestApplicationId2, TestUsers[1]);
    AddAccountPlayEvent(TestApplicationId1, TestUsers[1]);

    ncm::ApplicationId applicationId[5];

    {
        int count = pdm::QueryRecentlyPlayedApplication(applicationId, NN_ARRAY_SIZE(applicationId), TestUsers[0]);
        ASSERT_EQ(3, count);
        ASSERT_EQ(TestApplicationId3, applicationId[0]);
        ASSERT_EQ(TestApplicationId2, applicationId[1]);
        ASSERT_EQ(TestApplicationId1, applicationId[2]);
    }

    {
        int count = pdm::QueryRecentlyPlayedApplication(applicationId, NN_ARRAY_SIZE(applicationId), TestUsers[1]);
        ASSERT_EQ(3, count);
        ASSERT_EQ(TestApplicationId1, applicationId[0]);
        ASSERT_EQ(TestApplicationId2, applicationId[1]);
        ASSERT_EQ(TestApplicationId3, applicationId[2]);
    }

    // 起動 → アプレット InFocus → アカウント Open。この段階ではアカウントイベントは追加されない。
    AddEvents({
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, TestUsers[0], 0, 0, 0),
    });

    {
        // Open 中のアプリの情報が反映されて TestAPplicationId1 が最も最近遊んだアプリケーションになっていることを確認。
        int count = pdm::QueryRecentlyPlayedApplication(applicationId, NN_ARRAY_SIZE(applicationId), TestUsers[0]);
        ASSERT_EQ(3, count);
        ASSERT_EQ(TestApplicationId1, applicationId[0]);
        ASSERT_EQ(TestApplicationId3, applicationId[1]);
        ASSERT_EQ(TestApplicationId2, applicationId[2]);
    }
}

TEST_F(PdmQueryTest, GetRecentlyPlayedApplicationUpdateEvent)
{
    // 更新イベント待ち → アカウントイベント追加を繰り返す。

    os::Event addReadyEvent(os::EventClearMode_ManualClear);
    os::Event queryReadyEvent(os::EventClearMode_ManualClear);
    struct ThreadArg
    {
        os::Event* pAddReadyEvent;
        os::Event* pQueryReadyEvent;
    } threadArg{ &addReadyEvent, &queryReadyEvent };
    os::ThreadType thread;
    static NN_OS_ALIGNAS_THREAD_STACK Bit8 threadStack[8 * 1024];
    // AddAccountPlayEvent でアカウントオープンイベントが追加されて更新イベントが発行された後、
    // アカウントクローズイベント追加によって RecordEntry<UserAccount> が Invalidate されてから（Open 中という扱いでなくなってから）
    // 対応した InOpen イベントが作成されるまでの間にクエリに割り込まれると失敗するので、queryReadyEvent を使ってイベント登録が全て完了してからクエリが走るようにする。
    os::CreateThread(&thread, [](void* arg) {
        auto pAddReadyEvent = reinterpret_cast<ThreadArg*>(arg)->pAddReadyEvent;
        auto pQueryReadyEvent = reinterpret_cast<ThreadArg*>(arg)->pQueryReadyEvent;
        for( int i = 0; i < NN_ARRAY_SIZE(TestApplicationIdList); i++ )
        {
            ASSERT_TRUE(pAddReadyEvent->TimedWait(TimeSpan::FromSeconds(10)));
            ASSERT_FALSE(pQueryReadyEvent->TryWait());
            pAddReadyEvent->Clear();
            AddAccountPlayEvent(TestApplicationIdList[i], TestUsers[0]);
            pQueryReadyEvent->Signal();
        }
    },
    &threadArg, threadStack, sizeof(threadStack), os::HighestThreadPriority);
    os::StartThread(&thread);
    NN_UTIL_SCOPE_EXIT{ os::DestroyThread(&thread); };

    auto pUpdateEvent = pdm::GetRecentlyPlayedApplicationUpdateEvent();
    pUpdateEvent->Clear();

    for( int i = 0; i < NN_ARRAY_SIZE(TestApplicationIdList); i++ )
    {
        ASSERT_FALSE(pUpdateEvent->TryWait());
        ASSERT_FALSE(queryReadyEvent.TryWait());
        addReadyEvent.Signal();
        ASSERT_TRUE(pUpdateEvent->TimedWait(nn::TimeSpan::FromSeconds(10)));
        pUpdateEvent->Clear();
        ASSERT_TRUE(queryReadyEvent.TimedWait(nn::TimeSpan::FromSeconds(10)));
        queryReadyEvent.Clear();

        ncm::ApplicationId applicationId[NN_ARRAY_SIZE(TestApplicationIdList)];
        int count = pdm::QueryRecentlyPlayedApplication(applicationId, NN_ARRAY_SIZE(applicationId), TestUsers[0]);
        ASSERT_EQ(i + 1, count);
        for( int j = 0; j <= i; j++ )
        {
            ASSERT_EQ(TestApplicationIdList[i - j], applicationId[j]);
        }
    }
}
