﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nn/nn_TimeSpan.h>
#include <cstdlib>
#include "../Common/ModuleInfo.h"
#include "../Common/ModuleNames.h"

using namespace nnt::pcv;

extern int g_StressRuns;

nn::os::Tick    g_TotalTick,
                g_BeginTick,
                g_EndTick;
float           g_SecDuration;

const nn::pcv::Module stressTestModules[] =
    {
        nn::pcv::Module_Cpu,
        nn::pcv::Module_Gpu,
    };

class StressTest : public ::testing::TestWithParam<nn::pcv::Module>{};

INSTANTIATE_TEST_CASE_P(stressTestModules, StressTest, testing::ValuesIn(stressTestModules));

const int numRandIterations = 100;

TEST_P(StressTest, StressModule)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nnt::pcv::ModuleInfo moduleInfo = ModuleInfoList[GetParam()];
    nn::pcv::Module module = moduleInfo.name;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    bool allSuccess = true;

    NN_LOG("Stress Runs: %d\n", g_StressRuns);
    NN_LOG("Testing %s", GetModuleName(module));
    g_BeginTick = nn::os::GetSystemTick();

    for (int i = 0; i < g_StressRuns; ++i)
    {
        if (i % (g_StressRuns / 10) == 0)
            NN_LOG(".");
        for (int j = 0; j < numRates; ++j)
        {
            // NN_LOG("Setting %s to %.2f MHz\n", nnt::pcv::GetModuleName(module), rates[j] / (1000.0 * 1000.0));
            nn::Result setClockResult = nn::pcv::SetClockRate(module, rates[j]);
            allSuccess =  allSuccess && setClockResult.IsSuccess();
        }
        ASSERT_TRUE(allSuccess);

        for (int j = numRates - 1; j >= 0; --j)
        {
            // NN_LOG("Setting %s to %.2f MHz\n", nnt::pcv::GetModuleName(module), rates[j] / (1000.0 * 1000.0));
            nn::Result setClockResult = nn::pcv::SetClockRate(module, rates[j]);
            allSuccess =  allSuccess && setClockResult.IsSuccess();
        }
        ASSERT_TRUE(allSuccess);

        int randClockRate = 0;

        int tick = (int) nn::os::GetSystemTick().GetInt64Value();
        srand((unsigned int) tick);
        for (int j = 0; j < numRandIterations; ++j)
        {
            randClockRate = rand() % (rates[numRates - 1] - rates[0]) + rates[0];
            // NN_LOG("Cpu rate: %d\n", randClockRate);
            nn::Result setClockResult = nn::pcv::SetClockRate(module, randClockRate);
            allSuccess =  allSuccess && setClockResult.IsSuccess();

        }
        ASSERT_TRUE(allSuccess);
    }


    g_EndTick = nn::os::GetSystemTick();
    g_TotalTick = g_EndTick - g_BeginTick;
    g_SecDuration = nn::os::ConvertToTimeSpan(g_TotalTick).GetMilliSeconds() / 1000.0;
    NN_LOG("\nDone.\n");
    NN_LOG("Duration: %.3f sec.\n", g_SecDuration);
    int totalIterations = numRandIterations + (2 * numRates);
    NN_LOG("Average setClockRate per second: %f\n",
        (g_StressRuns * totalIterations) / g_SecDuration);
}


TEST_F(StressTest, Random_Cpu_Gpu)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz cpu_rates[nn::pcv::MaxNumClockRates];
    memset(cpu_rates, 0, sizeof(cpu_rates) / sizeof(cpu_rates[0]));
    nn::pcv::ClockHz gpu_rates[nn::pcv::MaxNumClockRates];
    memset(gpu_rates, 0, sizeof(gpu_rates) / sizeof(gpu_rates[0]));

    int cpu_numRates = 0;
    int gpu_numRates = 0;

    int cpuRate, gpuRate;

    int tick = (int) nn::os::GetSystemTick().GetInt64Value();
    srand((unsigned int) tick);

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        cpu_rates,
        &cpu_numRates,
        nn::pcv::Module_Cpu,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        gpu_rates,
        &gpu_numRates,
        nn::pcv::Module_Gpu,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    bool allSuccess = true;

    NN_LOG("Stress Runs: %d\n", g_StressRuns);
    NN_LOG("Testing");
    g_BeginTick = nn::os::GetSystemTick();
    for (int i = 0; i < g_StressRuns; ++i)
    {
        if (i % (g_StressRuns / 10) == 0)
            NN_LOG(".");

        for (int j = 0; j < numRandIterations; ++j)
        {
            cpuRate = rand() % (cpu_rates[cpu_numRates - 1] - cpu_rates[0]) + cpu_rates[0];
            // NN_LOG("Cpu rate: %d\n", cpuRate);
            nn::Result setClockResult = nn::pcv::SetClockRate(nn::pcv::Module_Cpu, cpuRate);
            allSuccess = allSuccess && setClockResult.IsSuccess();

            gpuRate = rand() % (gpu_rates[gpu_numRates - 1] - gpu_rates[0]) + gpu_rates[0];
            // NN_LOG("Gpu rate: %d\n", gpuRate);
            setClockResult = nn::pcv::SetClockRate(nn::pcv::Module_Gpu, gpuRate);
            allSuccess = allSuccess && setClockResult.IsSuccess();
        }
    }
    ASSERT_TRUE(allSuccess);
    g_EndTick = nn::os::GetSystemTick();
    g_TotalTick = g_EndTick - g_BeginTick;
    g_SecDuration = nn::os::ConvertToTimeSpan(g_TotalTick).GetMilliSeconds() / 1000.0;
    NN_LOG("\nDone.\n");
    NN_LOG("Duration: %.3f sec.\n", g_SecDuration);
    NN_LOG("Average setClockRate per second: %f\n", (g_StressRuns * (2 * numRandIterations)) / g_SecDuration);

}
