﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcm/pcm.h>
#include <nn/pcv/pcv.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>

#include "../Common/ModuleNames.h"

namespace nnt { namespace pcv {

struct ModuleGatingDefaultInfo
{
    nn::pcv::Module module;
    bool clockEnabledDefault;
    bool powerEnabledDefault;
};

// PowerGating, ClockGating の対象となるモジュールを列挙する。
const nnt::pcv::ModuleGatingDefaultInfo GatingModules[] =
{
    { nn::pcv::Module_Nvenc, false, false },
    { nn::pcv::Module_Nvdec, false, false },
    { nn::pcv::Module_Nvjpg, false, false },
};

class GatingTest : public ::testing::TestWithParam<nnt::pcv::ModuleGatingDefaultInfo>{};

INSTANTIATE_TEST_CASE_P(GatingModules, GatingTest, ::testing::ValuesIn(GatingModules));

void PrintVsysAp(int count) NN_NOEXCEPT
{
    nn::pcm::Initialize();

    for ( int i = 0; i < count; i++ )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        int mW = nn::pcm::ReadCurrentPower(nn::pcm::MeasuringPoint_VsysAp);
        NN_LOG("VsysAp is %d mW.\n", mW);
    }

    nn::pcm::Finalize();
}

// PowerGating が想定通りであることの確認。
TEST_P(GatingTest, CheckPowerGatingDefault)
{
    nn::pcv::ModuleState state;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetState(&state, GetParam().module));
    NN_LOG("%s power enabled: %s.\n", nnt::pcv::GetModuleName(GetParam().module), state.powerEnabled ? "true" : "false");
    EXPECT_EQ(GetParam().powerEnabledDefault, state.powerEnabled);
}

// Gating による消費電力差の雑感を把握するためのテスト。
// VsysAp の消費電力を表示するだけで具体的なチェックは何もしない。
TEST(GatingTestForAll, DisplayPowerConsumption)
{
    // 初期状態が想定の通りか確認する。
    for ( const auto& info : GatingModules )
    {
        nn::pcv::ModuleState state;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetState(&state, info.module));
        if ( info.clockEnabledDefault != state.clockEnabled
            || info.powerEnabledDefault != state.powerEnabled )
        {
            // 想定と異なる場合このテストは実行しない。
            return;
        }
    }

    // 初期状態の消費電力を 5 秒表示する。
    PrintVsysAp(5);

    NN_LOG("Enable all power.\n");

    // PowerGating を無効にする。
    for ( const auto& info : GatingModules )
    {
        if ( info.powerEnabledDefault != true )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetPowerEnabled(info.module, true));
        }
    }

    // 消費電力を 5 秒表示する。
    PrintVsysAp(5);

    NN_LOG("Enable all clock.\n");

    // ClockGating を無効にする。
    for ( const auto& info : GatingModules )
    {
        if ( info.clockEnabledDefault != true )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockEnabled(info.module, true));
        }
    }

    // 消費電力を 5 秒表示する。
    PrintVsysAp(5);

    // 変更(true に)したものだけ元に戻す。
    for ( const auto& info : GatingModules )
    {
        if ( info.clockEnabledDefault != true )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockEnabled(info.module, false));
        }

        if ( info.powerEnabledDefault != true )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetPowerEnabled(info.module, false));
        }
    }
}

}} // namespace nnt::pcv
