﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <vector>

#include <nn/nn_Common.h>

#include <nn/pcv/pcv.h>
#include <nn/spl/spl_Api.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nnt/nntest.h>

#include "IsValidPowerDomain.h"

namespace nnt { namespace pcv {

namespace {

const nn::pcv::PowerDomain ValidGpuPowerDomain = nn::pcv::PowerDomain_Max77621_Gpu;

const std::vector<nn::pcv::PowerDomain> ValidPowerDomains =
{
    nn::pcv::PowerDomain_Max77620_Sd0,
    nn::pcv::PowerDomain_Max77620_Sd1,
    nn::pcv::PowerDomain_Max77620_Sd2,
    nn::pcv::PowerDomain_Max77620_Sd3,

    nn::pcv::PowerDomain_Max77620_Ldo0,
    nn::pcv::PowerDomain_Max77620_Ldo1,
    nn::pcv::PowerDomain_Max77620_Ldo2,
    nn::pcv::PowerDomain_Max77620_Ldo3,
    nn::pcv::PowerDomain_Max77620_Ldo4,
    nn::pcv::PowerDomain_Max77620_Ldo5,
    nn::pcv::PowerDomain_Max77620_Ldo6,
    nn::pcv::PowerDomain_Max77620_Ldo7,
    nn::pcv::PowerDomain_Max77620_Ldo8,

    nn::pcv::PowerDomain_Max77621_Cpu,
    nn::pcv::PowerDomain_Max77621_Gpu,
};

const nn::pcv::PowerDomain ValidGpuPowerDomainForMariko = nn::pcv::PowerDomain_Max77812_Gpu;

const std::vector<nn::pcv::PowerDomain> ValidPowerDomainsForMariko31 =
{
    nn::pcv::PowerDomain_Max77620_Sd0,
    nn::pcv::PowerDomain_Max77620_Sd1,
    nn::pcv::PowerDomain_Max77620_Sd2,
    nn::pcv::PowerDomain_Max77620_Sd3,

    nn::pcv::PowerDomain_Max77620_Ldo0,
    nn::pcv::PowerDomain_Max77620_Ldo1,
    nn::pcv::PowerDomain_Max77620_Ldo2,
    nn::pcv::PowerDomain_Max77620_Ldo3,
    nn::pcv::PowerDomain_Max77620_Ldo4,
    nn::pcv::PowerDomain_Max77620_Ldo5,
    nn::pcv::PowerDomain_Max77620_Ldo6,
    nn::pcv::PowerDomain_Max77620_Ldo7,
    nn::pcv::PowerDomain_Max77620_Ldo8,

    nn::pcv::PowerDomain_Max77812_Cpu,
    nn::pcv::PowerDomain_Max77812_Gpu,
};

const std::vector<nn::pcv::PowerDomain> ValidPowerDomainsForMariko211 =
{
    nn::pcv::PowerDomain_Max77620_Sd0,
    nn::pcv::PowerDomain_Max77620_Sd1,
    nn::pcv::PowerDomain_Max77620_Sd2,
    nn::pcv::PowerDomain_Max77620_Sd3,

    nn::pcv::PowerDomain_Max77620_Ldo0,
    nn::pcv::PowerDomain_Max77620_Ldo1,
    nn::pcv::PowerDomain_Max77620_Ldo2,
    nn::pcv::PowerDomain_Max77620_Ldo3,
    nn::pcv::PowerDomain_Max77620_Ldo4,
    nn::pcv::PowerDomain_Max77620_Ldo5,
    nn::pcv::PowerDomain_Max77620_Ldo6,
    nn::pcv::PowerDomain_Max77620_Ldo7,
    nn::pcv::PowerDomain_Max77620_Ldo8,

    nn::pcv::PowerDomain_Max77812_Cpu,
    nn::pcv::PowerDomain_Max77812_Gpu,
    nn::pcv::PowerDomain_Max77812_Dram,
};

enum class PlatformType
{
    Unknown,
    Icosa,
    Copper,
    Hoag,
    IcosaMariko,
};

PlatformType GetPlatformType() NN_NOEXCEPT
{
    const char* const name = "platformconfig";
    const char* const key = "platformtype";

    size_t size = ::nn::settings::fwdbg::GetSettingsItemValueSize(name, key);
    EXPECT_NE(0, size);

    char buffer[size];
    EXPECT_EQ(size, ::nn::settings::fwdbg::GetSettingsItemValue(buffer, size, name, key));

    if ( std::strcmp("Icosa", buffer) == 0 )
    {
        return PlatformType::Icosa;
    }
    else if ( std::strcmp("Copper", buffer) == 0 )
    {
        return PlatformType::Copper;
    }
    else if ( std::strcmp("Hoag", buffer) == 0 )
    {
        return PlatformType::Hoag;
    }
    else if ( std::strcmp("IcosaMariko", buffer) == 0 )
    {
        return PlatformType::IcosaMariko;
    }

    return PlatformType::Unknown;
}

bool IsMax77812PhaseConfiguration211() NN_NOEXCEPT
{
    nn::spl::Initialize();
    auto result = (nn::spl::Regulator_Max77812PhaseConfiguration211 == nn::spl::GetRegulator());
    nn::spl::Finalize();

    return result;
}

} // namespace

bool IsValidPowerDomain(nn::pcv::PowerDomain powerDomain) NN_NOEXCEPT
{
    const std::vector<nn::pcv::PowerDomain>* pValidPowerDomains;

    auto platformType = GetPlatformType();
    EXPECT_NE(PlatformType::Unknown, platformType);
    switch ( platformType )
    {
    case PlatformType::Icosa:
    case PlatformType::Copper:
        pValidPowerDomains = &ValidPowerDomains;
        break;
    case PlatformType::Hoag:
    case PlatformType::IcosaMariko:
        pValidPowerDomains = IsMax77812PhaseConfiguration211() ? &ValidPowerDomainsForMariko211 : &ValidPowerDomainsForMariko31;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    auto pFound = std::find(pValidPowerDomains->cbegin(), pValidPowerDomains->cend(), powerDomain);
    return pFound != pValidPowerDomains->cend();
}

nn::pcv::PowerDomain GetProperGpuPowerDomain() NN_NOEXCEPT
{
    nn::pcv::PowerDomain gpuPowerDomain;

    auto platformType = GetPlatformType();
    EXPECT_NE(PlatformType::Unknown, platformType);
    switch ( platformType )
    {
    case PlatformType::Icosa:
    case PlatformType::Copper:
        gpuPowerDomain = ValidGpuPowerDomain;
        break;
    case PlatformType::Hoag:
    case PlatformType::IcosaMariko:
        gpuPowerDomain = ValidGpuPowerDomainForMariko;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return gpuPowerDomain;
}

}} // namespace nnt::pcv
