﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/os.h>

#include <nn/init/init_Malloc.h>
#include <nn/pcv/pcv.h>

#include <nnt/nntest.h>

// Contains necessary initialization code, such as setting heap size and initializing PCV

namespace {

const size_t TotalHeapSize = 32 * 1024 * 1024;

// After testing, always set the clocks back to safe values
void SetSafeClocks() NN_NOEXCEPT
{
    nn::pcv::SetClockRate(nn::pcv::Module_Cpu, 1224 * 1000 * 1000);
}

class PcvEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        NN_LOG("Initializing Pcv...\n");
        nn::pcv::Initialize();

        NN_LOG("Initialized\n");
    }
    virtual void TearDown() override
    {
        NN_LOG("Finalizing Pcv...\n");
        SetSafeClocks();
        nn::pcv::Finalize();
    }
};

} // namespace

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(TotalHeapSize);
    if ( !result.IsSuccess() )
    {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", TotalHeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, TotalHeapSize / 2);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), TotalHeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new PcvEnvironment);
}
