﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include "../Common/ModuleNames.h"
#include "../Common/ModuleInfo.h"

using namespace nnt::pcv;

namespace {
    const nnt::pcv::ModuleInfo clockModules[] =
    {
        { nn::pcv::Module_I2s1,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2s2,              NonDiscrete,      Floor },
        { nn::pcv::Module_Pwm,               NonDiscrete,      Floor },
        { nn::pcv::Module_I2c1,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2c2,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2c3,              NonDiscrete,      Floor },
        { nn::pcv::Module_Sdmmc1,            NonDiscrete,      Floor },
        { nn::pcv::Module_Sdmmc2,            NonDiscrete,      Floor },
        { nn::pcv::Module_Sdmmc4,            NonDiscrete,      Floor },
        { nn::pcv::Module_Csite,             NonDiscrete,      Floor },
        { nn::pcv::Module_Hda2codec2x,       NonDiscrete,      Floor },
        { nn::pcv::Module_Actmon,            NonDiscrete,      Floor },
        { nn::pcv::Module_I2cSlow,           NonDiscrete,      Floor },
        { nn::pcv::Module_Hda,               NonDiscrete,      Floor },

        // Cannot change clock rate of UartA during this test
        // because UartA is used for system (kernel) log.
        // { nn::pcv::Module_UartA,             NonDiscrete,      Floor },

        { nn::pcv::Module_UartB,             NonDiscrete,      Floor },
        { nn::pcv::Module_UartC,             NonDiscrete,      Floor },
        { nn::pcv::Module_UartD,             NonDiscrete,      Floor },
        { nn::pcv::Module_Entropy,           NonDiscrete,      Floor },
        { nn::pcv::Module_SocTherm,          NonDiscrete,      Floor },
        { nn::pcv::Module_Qspi,              NonDiscrete,      Floor },
        { nn::pcv::Module_AudioUart,         NonDiscrete,      Floor },

        { nn::pcv::Module_Maud,              NonDiscrete,      Floor   },
        { nn::pcv::Module_UartFstMipiCal,    NonDiscrete,      Floor   },
        { nn::pcv::Module_Extperiph1,        NonDiscrete,      Floor   },
        { nn::pcv::Module_Ahub,              NonDiscrete,      Floor   },

    };

    const nnt::pcv::ModuleInfo usbClockModules[] =
    {
        { nn::pcv::Module_XusbCoreHostSrc,   NonDiscrete,      Floor },
        { nn::pcv::Module_XusbFalconSrc,     NonDiscrete,      Floor },
        { nn::pcv::Module_XusbFsSrc,         NonDiscrete,      Floor },
        { nn::pcv::Module_XusbCoreDevSrc,    NonDiscrete,      Floor },
        { nn::pcv::Module_XusbSsSrc,         NonDiscrete,      Floor },
    };

    class FloorSetClockTest : public ::testing::TestWithParam<nnt::pcv::ModuleInfo>{};

    INSTANTIATE_TEST_CASE_P(SetClockRateModules, FloorSetClockTest, testing::ValuesIn(clockModules));
    INSTANTIATE_TEST_CASE_P(SetClockRateModulesForUsb, FloorSetClockTest, testing::ValuesIn(usbClockModules));
}

// Test SetClockRates() for performance IPs (CPU, GPU, EMC, ADSP, System Bus, VIC,
// NVDEC, NVENC, NVJPG, TSECB)
// Query Possible Rates function should return discrete list of frequencies
TEST_P(FloorSetClockTest, FloorRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Range, listType);

    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), (rates[1] - 1000) / (1000.0 * 1000.0));
    nn::Result setClockResult = nn::pcv::SetClockRate(module, rates[1] - 1000);
    EXPECT_TRUE(setClockResult.IsSuccess());

    nn::pcv::ModuleState state = {0};
    nn::Result getStateResult = nn::pcv::GetState(&state, module);
    ASSERT_TRUE(getStateResult.IsSuccess());
    EXPECT_GE(rates[1] - 1000, state.clockFrequency);
    NN_LOG("Clock frequency: %d Hz \n", state.clockFrequency);
}

TEST_P(FloorSetClockTest, BelowMin)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    // Set to 10 hz
    NN_LOG("Setting %s to %d KHz\n", GetModuleName(module), 100);
    nn::Result setLowResult = nn::pcv::SetClockRate(module, 100000);
    EXPECT_TRUE(setLowResult.IsSuccess());

    nn::pcv::ModuleState state = {0};
    nn::Result getStateResult = nn::pcv::GetState(&state, module);
    ASSERT_TRUE(getStateResult.IsSuccess());
}

TEST_P(FloorSetClockTest, AboveMax)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Range, listType);

    // Check for higher frequency case for error
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), rates[1] / (1000.0 * 1000.0) * 2);
    nn::Result setHighResult = nn::pcv::SetClockRate(module, rates[1] * 2);
    EXPECT_TRUE(setHighResult.IsSuccess());

    nn::pcv::ModuleState state = {0};
    nn::Result getStateResult = nn::pcv::GetState(&state, module);
    ASSERT_TRUE(getStateResult.IsSuccess());
    EXPECT_LT(0, state.clockFrequency); //relaxed test to account for clock granularity
}

TEST_P(FloorSetClockTest, setToZeroRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(GetParam().name), 0);
    nn::Result setZeroLimit = nn::pcv::SetClockRate(GetParam().name, 0);
    ASSERT_FALSE(setZeroLimit.IsSuccess());
}
