﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include "../Common/ModuleNames.h"
#include "../Common/ModuleInfo.h"

using namespace nnt::pcv;

namespace {
    const nnt::pcv::ModuleInfo setClockExactModules[] =
    {
        { nn::pcv::Module_Disp2,             NonDiscrete,      Exact   },
        { nn::pcv::Module_Sor1,              NonDiscrete,      Exact   },
    };

    class ExactSetClockTest : public ::testing::TestWithParam<nnt::pcv::ModuleInfo>{};

    INSTANTIATE_TEST_CASE_P(SetClockRateModules, ExactSetClockTest, testing::ValuesIn(setClockExactModules));

}

// Test SetClockRates() for modules that only support set clock rates to exact rates
// Query Possible Rates function should return non discrete list of frequencies
TEST_P(ExactSetClockTest, ExactRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Range, listType);
    ASSERT_EQ(2, numRates);

    // Set to max
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), rates[1] / (1000.0 * 1000.0));
    nn::Result setExactRateResult = nn::pcv::SetClockRate(module, rates[1]);
    EXPECT_TRUE(setExactRateResult.IsSuccess());

    nn::pcv::ModuleState state = {0};
    EXPECT_TRUE(nn::pcv::SetClockEnabled(module, true).IsSuccess());
    nn::Result getStateResult = nn::pcv::GetState(&state, module);
    EXPECT_TRUE(nn::pcv::SetClockEnabled(module, false).IsSuccess());
    EXPECT_TRUE(getStateResult.IsSuccess());
    EXPECT_LT(0, state.clockFrequency); //relaxed test to account for clock granularity
}


// TODO: figure out how to find Min Rate for modules in this case
TEST_P(ExactSetClockTest, BelowMin)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    // Set to 10 hz
    NN_LOG("Setting %s to %d KHz\n", GetModuleName(module), 100);
    nn::Result setLowResult = nn::pcv::SetClockRate(module, 100000);
    EXPECT_TRUE(setLowResult.IsSuccess());

}

TEST_P(ExactSetClockTest, AboveMax)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Range, listType);

    // Check for higher frequency case for error case
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), rates[1] / (1000.0 * 1000.0) * 2);
    nn::Result setHighResult = nn::pcv::SetClockRate(module, rates[1] * 2);
    EXPECT_TRUE(setHighResult.IsSuccess());

    nn::pcv::ModuleState state = {0};
    EXPECT_TRUE(nn::pcv::SetClockEnabled(module, true).IsSuccess());
    nn::Result getStateResult = nn::pcv::GetState(&state, module);
    EXPECT_TRUE(nn::pcv::SetClockEnabled(module, false).IsSuccess());
    EXPECT_TRUE(getStateResult.IsSuccess());
    EXPECT_LT(0, state.clockFrequency); //relaxed test to account for clock granularity
}

TEST_P(ExactSetClockTest, setToZeroRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(GetParam().name), 0);
    nn::Result result = nn::pcv::SetClockRate(GetParam().name, 0);
    ASSERT_FALSE(result.IsSuccess());
}
