﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include "../Common/ModuleNames.h"
#include "../Common/ModuleInfo.h"

using namespace nnt::pcv;

namespace {
    const nnt::pcv::ModuleInfo setClockModules[] =
    {
        { nn::pcv::Module_Cpu,               Discrete,         Ceiling },
        // CI テストの際は fromhost 起動を用いるため、GPU の電源を入れるためのプロセスが立ち上がっておらず、テスト不可。
        // { nn::pcv::Module_Gpu,               Discrete,         Ceiling },
        { nn::pcv::Module_AudioDsp,          Discrete,         Ceiling },
        { nn::pcv::Module_Emc,               Discrete,         Ceiling },
        { nn::pcv::Module_Vic,               Discrete,         Ceiling },
        { nn::pcv::Module_Nvenc,             Discrete,         Ceiling },
        { nn::pcv::Module_Nvjpg,             Discrete,         Ceiling },
        { nn::pcv::Module_Nvdec,             Discrete,         Ceiling },
        { nn::pcv::Module_Tsecb,             Discrete,         Ceiling },
        { nn::pcv::Module_SysBus,            Discrete,         Ceiling },
        { nn::pcv::Module_Tsec,              Discrete,         Ceiling },
        { nn::pcv::Module_Ape,               Discrete,         Ceiling },
    };

    const nnt::pcv::ModuleInfo sharedPLLC2Modules[] =
    {
        { nn::pcv::Module_Vic,               Discrete,         Ceiling },
        { nn::pcv::Module_Nvjpg,             Discrete,         Ceiling },
        { nn::pcv::Module_Tsecb,             Discrete,         Ceiling },
    };

    const nnt::pcv::ModuleInfo sharedPLLC3Modules[] =
    {
        { nn::pcv::Module_Nvenc,             Discrete,         Ceiling },
        { nn::pcv::Module_Nvdec,             Discrete,         Ceiling },
    };

    void ClockGateSharedModules(const nnt::pcv::ModuleInfo sharedModules[], int length)
    {
         for (int i = 0; i < length; i++)
        {
            nn::pcv::ModuleState sharedModuleState = {0};
            nn::pcv::Module sharedModule = sharedModules[i].name;
            nn::Result getSharedModuleResult = nn::pcv::GetState(&sharedModuleState, sharedModule);
            EXPECT_TRUE(getSharedModuleResult.IsSuccess());

            // Disable all IPS who share clock
            if (sharedModuleState.clockEnabled)
            {
                nn::Result enableClkResult = nn::pcv::SetClockEnabled(sharedModule, false);
                EXPECT_TRUE(enableClkResult.IsSuccess());
            }

            nn::pcv::ModuleState validState = {0};
            nn::Result validateStateResult = nn::pcv::GetState(&validState, sharedModule);
            EXPECT_TRUE(validateStateResult.IsSuccess());
            EXPECT_EQ(false, validState.clockEnabled);
        }
    }

    void RestoreModule(nn::pcv::Module module, nn::pcv::ModuleState *initialState)
    {
        nn::pcv::ModuleState finalState = {0};
        nn::Result finalStateResult = nn::pcv::GetState(&finalState, module);
        EXPECT_TRUE(finalStateResult.IsSuccess());

        nn::Result restoreInitResult;
        if (finalState.powerEnabled != initialState->powerEnabled)
        {
            restoreInitResult = nn::pcv::SetPowerEnabled(module, initialState->powerEnabled);
            EXPECT_TRUE(restoreInitResult.IsSuccess());

            // SIGLO-43269: finalState must be updated after restoring powerEnabled.
            nn::Result finalStateResult = nn::pcv::GetState(&finalState, module);
            EXPECT_TRUE(finalStateResult.IsSuccess());
        }

        // SIGLO-43269: clockEnabled must be checked and restored if powerEnabled is false.
        //if (initialState->powerEnabled)
        {
            //restoreInitResult = nn::pcv::SetReset(module, initialState->resetAsserted);
            //EXPECT_TRUE(restoreInitResult.IsSuccess());

            // Prevent from enabling when already enabled
            if (finalState.clockEnabled != initialState->clockEnabled)
            {
                restoreInitResult = nn::pcv::SetClockEnabled(module, initialState->clockEnabled);
                EXPECT_TRUE(restoreInitResult.IsSuccess());
            }

            if (initialState->clockEnabled)
            {
                restoreInitResult = nn::pcv::SetClockRate(module, initialState->clockFrequency);
                EXPECT_TRUE(restoreInitResult.IsSuccess());
            }
        }

        // Check to see module reverted back to default settings
        nn::pcv::ModuleState endState = {0};
        restoreInitResult = nn::pcv::GetState(&endState, module);
        EXPECT_TRUE(restoreInitResult.IsSuccess());
        EXPECT_EQ(initialState->powerEnabled, endState.powerEnabled);
        EXPECT_EQ(initialState->clockEnabled, endState.clockEnabled);

        // If reset was enabled then H/W could of set IP initial frequency outside of its list of frequencies range
        if (!initialState->resetAsserted)
            EXPECT_EQ(initialState->clockFrequency, endState.clockFrequency);
    }

    class CeilingSetClockTest : public ::testing::TestWithParam<nnt::pcv::ModuleInfo>{};

    INSTANTIATE_TEST_CASE_P(SetClockRateCeilingModules, CeilingSetClockTest, testing::ValuesIn(setClockModules));
}

TEST_P(CeilingSetClockTest, ExactRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;
    nn::pcv::Module module = GetParam().name;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    // Get default clock rate
    nn::pcv::ModuleState defaultState = {0};
    nn::Result getStateResult = nn::pcv::GetState(&defaultState, module);
    ASSERT_TRUE(getStateResult.IsSuccess());

    // Special case for modules who share clocks
    // Clock gates other IPs who share current IP's clock
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        int length = sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC2Modules, length);
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        int length = sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC3Modules, length);
    }

    // IF list of frequency rates is discrete
    ASSERT_EQ(GetParam().setClockType, Ceiling);
    for (int i=0; i < numRates; ++i)
    {
        NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), rates[i] / (1000.0 * 1000.0));
        nn::Result setClockResult = nn::pcv::SetClockRate(module, rates[i]);
        EXPECT_TRUE(setClockResult.IsSuccess());

        nn::pcv::ModuleState state = {0};
        nn::Result getStateResult = nn::pcv::GetState(&state, module);
        ASSERT_TRUE(getStateResult.IsSuccess());
        EXPECT_EQ(rates[i], state.clockFrequency);

    }

    // Restore default clock rate
    RestoreModule(module, &defaultState);

    // Special case for modules who share clocks
    // Restore initial state for other IPs who share clocks with current IP (Same initial state as current IP)
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        for (int i = 0; i < sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC2Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        for (int i = 0; i < sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC3Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
}

TEST_P(CeilingSetClockTest, CelingRates)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());

    // Get default clock rate
    nn::pcv::ModuleState defaultState = {0};
    nn::Result getStateResult = nn::pcv::GetState(&defaultState, module);
    ASSERT_TRUE(getStateResult.IsSuccess());

    // Special case for modules who share clocks
    // Clock gates other IPs who share current IP's clock
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        int length = sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC2Modules, length);
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        int length = sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC3Modules, length);
    }

    // If list of frequency rates is discrete
    ASSERT_EQ(GetParam().setClockType, Ceiling);

    for (int i=0; i < numRates - 1; ++i)
    {
        nn::pcv::ClockHz ceilRate = rates[i] + ((rates[i + 1] - rates[i]) / 2);
        NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), ceilRate / (1000.0 * 1000.0));
        nn::Result setClockResult = nn::pcv::SetClockRate(module, ceilRate);
        ASSERT_TRUE(setClockResult.IsSuccess());

        nn::pcv::ModuleState validState = {0};
        nn::Result validateStateResult = nn::pcv::GetState(&validState, module);
        ASSERT_TRUE(validateStateResult.IsSuccess());
        EXPECT_EQ(rates[i + 1], validState.clockFrequency);
    }

    // Restore default clock rate
    RestoreModule(module, &defaultState);

    // Special case for modules who share clocks
    // Restore initial state for other IPs who share clocks with current IP (Same initial state as current IP)
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        for (int i = 0; i < sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC2Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        for (int i = 0; i < sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC3Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
}

TEST_P(CeilingSetClockTest, BelowMin)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());

    // Get default clock rate
    nn::pcv::ModuleState defaultState = {0};
    nn::Result getStateResult = nn::pcv::GetState(&defaultState, module);
    ASSERT_TRUE(getStateResult.IsSuccess());

    // Special case for modules who share clocks
    // Clock gates other IPs who share current IP's clock
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        int length = sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC2Modules, length);
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        int length = sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC3Modules, length);
    }

    ASSERT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    // Check for lower frequency case for round to min
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), rates[0] / (1000.0 * 1000.0) / 2);
    nn::Result setLowResult = nn::pcv::SetClockRate(module, rates[0] / 2);
    EXPECT_TRUE(setLowResult.IsSuccess());

    // Restore default clock rate
    RestoreModule(module, &defaultState);

    // Special case for modules who share clocks
    // Restore initial state for other IPs who share clocks with current IP (Same initial state as current IP)
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        for (int i = 0; i < sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC2Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        for (int i = 0; i < sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC3Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
}

TEST_P(CeilingSetClockTest, AboveMax)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());

    // Get default clock rate
    nn::pcv::ModuleState defaultState = {0};
    nn::Result getStateResult = nn::pcv::GetState(&defaultState, module);
    ASSERT_TRUE(getStateResult.IsSuccess());

    // Special case for modules who share clocks
    // Clock gates other IPs who share current IP's clock
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        int length = sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC2Modules, length);
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        int length = sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo);
        ClockGateSharedModules(sharedPLLC3Modules, length);
    }

    ASSERT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    // Check for higher frequency case for floor to max
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(module), rates[numRates - 1] / (1000.0 * 1000.0) * 2);
    nn::Result setHighResult = nn::pcv::SetClockRate(module, rates[numRates - 1] * 2);
    EXPECT_TRUE(setHighResult.IsSuccess());

    // Restore default clock rate
    RestoreModule(module, &defaultState);

    // Special case for modules who share clocks
    // Restore initial state for other IPs who share clocks with current IP (Same initial state as current IP)
    if (module == nn::pcv::Module_Vic || module == nn::pcv::Module_Nvjpg ||
        module == nn::pcv::Module_Tsecb)
    {
        for (int i = 0; i < sizeof(sharedPLLC2Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC2Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
    else if (module == nn::pcv::Module_Nvenc || module == nn::pcv::Module_Nvdec)
    {
        for (int i = 0; i < sizeof(sharedPLLC3Modules) / sizeof(nnt::pcv::ModuleInfo); i++)
        {
            nn::pcv::Module sharedModule = sharedPLLC3Modules[i].name;
            if (module != sharedModule)
            {
                RestoreModule(sharedModule, &defaultState);
            }
        }
    }
}

TEST_P(CeilingSetClockTest, setToZeroRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());
    NN_LOG("Setting %s to %.2f MHz\n", GetModuleName(GetParam().name), 0);
    nn::Result setZeroResult = nn::pcv::SetClockRate(GetParam().name, 0);
    ASSERT_FALSE(setZeroResult.IsSuccess());
}


TEST_F(CeilingSetClockTest, InvalidModule)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::Result invalidModuleResult = nn::pcv::SetClockRate(nn::pcv::Module_NumModule, 50 * 1000 * 1000);
    ASSERT_FALSE(invalidModuleResult.IsSuccess());
}
