﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include "../Common/ModuleInfo.h"
#include "../Common/ModuleNames.h"

using namespace nnt::pcv;

class GetStateTest : public ::testing::TestWithParam<nnt::pcv::ModuleInfo>{};
const nnt::pcv::ModuleInfo getStateModules[] =
    {
        { nn::pcv::Module_Cpu,               Discrete,         Ceiling },
        { nn::pcv::Module_Gpu,               Discrete,         Ceiling },
        { nn::pcv::Module_I2s1,              NonDiscrete,      Floor   },
        { nn::pcv::Module_I2s2,              NonDiscrete,      Floor   },
        { nn::pcv::Module_Pwm,               NonDiscrete,      Floor   },
        { nn::pcv::Module_I2c1,              NonDiscrete,      Floor   },
        { nn::pcv::Module_I2c2,              NonDiscrete,      Floor   },
        { nn::pcv::Module_I2c3,              NonDiscrete,      Floor   },
        { nn::pcv::Module_Disp1,             NonDiscrete,      Exact   },
        { nn::pcv::Module_Sdmmc1,            NonDiscrete,      Floor   },
        { nn::pcv::Module_Sdmmc2,            NonDiscrete,      Floor   },
        { nn::pcv::Module_Sdmmc4,            NonDiscrete,      Floor   },
        { nn::pcv::Module_Csite,             NonDiscrete,      Floor   },
        { nn::pcv::Module_Tsec,              NonDiscrete,      Floor   },
        { nn::pcv::Module_Mselect,           Discrete,         Floor   },
        { nn::pcv::Module_Hda2codec2x,       NonDiscrete,      Floor   },
        { nn::pcv::Module_Actmon,            NonDiscrete,      Floor   },
        { nn::pcv::Module_Sor1,              NonDiscrete,      Floor   },
        { nn::pcv::Module_Hda,               NonDiscrete,      Floor   },
        { nn::pcv::Module_XusbCoreHostSrc,   NonDiscrete,      Floor   },
        { nn::pcv::Module_XusbFalconSrc,     NonDiscrete,      Floor   },
        { nn::pcv::Module_XusbFsSrc,         NonDiscrete,      Floor   },
        { nn::pcv::Module_XusbCoreDevSrc,    NonDiscrete,      Floor   },
        { nn::pcv::Module_XusbSsSrc,         NonDiscrete,      Floor   },
        { nn::pcv::Module_UartB,             NonDiscrete,      Floor   },
        { nn::pcv::Module_UartC,             NonDiscrete,      Floor   },
        { nn::pcv::Module_UartD,             NonDiscrete,      Floor   },
        { nn::pcv::Module_Hostx,             Discrete,         Floor   },
        { nn::pcv::Module_SocTherm,          NonDiscrete,      Floor   },
        { nn::pcv::Module_Vic,               Discrete,         Ceiling },
        { nn::pcv::Module_Nvenc,             Discrete,         Ceiling },
        { nn::pcv::Module_Nvjpg,             Discrete,         Ceiling },
        { nn::pcv::Module_Nvdec,             Discrete,         Ceiling },
        { nn::pcv::Module_Tsecb,             Discrete,         Ceiling },
        { nn::pcv::Module_Ape,               NonDiscrete,      Exact   },
        { nn::pcv::Module_AudioDsp,          Discrete,         Ceiling },
        { nn::pcv::Module_AudioUart,         NonDiscrete,      Floor   },
        { nn::pcv::Module_Emc,               Discrete,         Ceiling },
        { nn::pcv::Module_Plle,              NonDiscrete,      Floor   },
        { nn::pcv::Module_PlleHwSeq,         NonDiscrete,      Floor   },

        { nn::pcv::Module_Dsi,               NonDiscrete,      Exact   },
        { nn::pcv::Module_Maud,              NonDiscrete,      Exact   },
        { nn::pcv::Module_Dpaux1,            NonDiscrete,      Exact   },
        { nn::pcv::Module_MipiCal,           NonDiscrete,      Exact   },
        { nn::pcv::Module_UartFstMipiCal,    NonDiscrete,      Exact   },
        { nn::pcv::Module_Osc,               NonDiscrete,      Exact   },
        { nn::pcv::Module_SysBus,            NonDiscrete,      Exact   },
        { nn::pcv::Module_SorSafe,           NonDiscrete,      Exact   },
        { nn::pcv::Module_XusbSs,            NonDiscrete,      Exact   },
        { nn::pcv::Module_XusbHost,          NonDiscrete,      Exact   },
        { nn::pcv::Module_XusbDevice,        NonDiscrete,      Exact   },
        { nn::pcv::Module_Extperiph1,        NonDiscrete,      Exact   },
        { nn::pcv::Module_Ahub,              NonDiscrete,      Exact   },
        { nn::pcv::Module_Hda2hdmicodec,     NonDiscrete,      Exact   },
        { nn::pcv::Module_Gpuaux,            Discrete,         Exact   },

        { nn::pcv::Module_UsbD,              Discrete,         Ceiling  },
        { nn::pcv::Module_Usb2,              Discrete,         Ceiling  },
        { nn::pcv::Module_Pcie,              Discrete,         Ceiling  },
        { nn::pcv::Module_Afi,               Discrete,         Ceiling  },
        { nn::pcv::Module_PciExClk,          Discrete,         Ceiling  },
        { nn::pcv::Module_PExUsbPhy,         Discrete,         Ceiling  },
        { nn::pcv::Module_XUsbPadCtl,        Discrete,         Ceiling  },
    };

INSTANTIATE_TEST_CASE_P(GetStateModules, GetStateTest, testing::ValuesIn(getStateModules));

TEST_P(GetStateTest, Module)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ModuleState state = {0};
    nn::Result getStateResult = nn::pcv::GetState(&state, GetParam().name);
    EXPECT_TRUE(getStateResult.IsSuccess());
    if (getStateResult.IsFailure())
        NN_LOG("Error : %d:%d\n", getStateResult.GetModule(), getStateResult.GetDescription());

    // Print state TODO:: Move printing to Client sample test process
    NN_LOG("Module: %s\n", GetModuleName(GetParam().name));
    NN_LOG("\tReset:         %s\n", state.resetAsserted ? "TRUE" : "FALSE");
    NN_LOG("\tPower enabled: %s\n", state.powerEnabled ? "TRUE" : "FALSE");
    NN_LOG("\tClock enabled: %s\n", state.clockEnabled ? "TRUE" : "FALSE");
    NN_LOG("\tClock frequency: %.2f MHz\n\n", state.clockFrequency / (1000.0 * 1000.0));
}

// Invalid module passed in
TEST_F(GetStateTest, InvalidModule)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    // ModuleState struct is initialized to all 0
    nn::pcv::ModuleState modState = {0};

    nn::pcv::Module errorMod = nn::pcv::Module_NumModule;

    nn::Result invalidModuleResult = nn::pcv::GetState(&modState, errorMod);
    EXPECT_FALSE(invalidModuleResult.IsSuccess());
}

TEST_F(GetStateTest, DISABLED_NULLState)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ModuleState *pModState = NULL;
    nn::Result r = nn::pcv::GetState(pModState, nn::pcv::Module_Cpu);
    ASSERT_FALSE(r.IsSuccess());
}
