﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include "../Common/ModuleNames.h"
#include "../Common/ModuleInfo.h"

using namespace nnt::pcv;

class GetDiscreteRatesTest : public ::testing::TestWithParam<nnt::pcv::ModuleInfo>{};
const nnt::pcv::ModuleInfo discreteRatesModules[] =
    {
        { nn::pcv::Module_Cpu,               Discrete,         Ceiling },
        { nn::pcv::Module_Gpu,               Discrete,         Ceiling },
        { nn::pcv::Module_Mselect,           Discrete,         Floor   },
        { nn::pcv::Module_Vic,               Discrete,         Ceiling },
        { nn::pcv::Module_Hostx,             Discrete,         Floor   },
        { nn::pcv::Module_Nvenc,             Discrete,         Ceiling },
        { nn::pcv::Module_Nvjpg,             Discrete,         Ceiling },
        { nn::pcv::Module_Nvdec,             Discrete,         Ceiling },
        { nn::pcv::Module_Tsecb,             Discrete,         Ceiling },
        { nn::pcv::Module_Ape,               Discrete,         Exact   },
        { nn::pcv::Module_AudioDsp,          Discrete,         Ceiling },
        { nn::pcv::Module_Emc,               Discrete,         Ceiling },
        { nn::pcv::Module_Plle,              Discrete,         Floor   },
        { nn::pcv::Module_PlleHwSeq,         Discrete,         Floor   },
        { nn::pcv::Module_Dpaux1,            Discrete,         Exact   },
        { nn::pcv::Module_MipiCal,           Discrete,         Exact   },
        { nn::pcv::Module_Osc,               Discrete,         Exact   },
        { nn::pcv::Module_SysBus,            Discrete,         Exact   },
        { nn::pcv::Module_SorSafe,           Discrete,         Exact   },
        { nn::pcv::Module_Gpuaux,            Discrete,         Exact   },
        { nn::pcv::Module_Tsec,              Discrete,         Ceiling },

        { nn::pcv::Module_UsbD,              Discrete,         Ceiling  },
        { nn::pcv::Module_Usb2,              Discrete,         Ceiling  },
        { nn::pcv::Module_Pcie,              Discrete,         Ceiling  },
        { nn::pcv::Module_Afi,               Discrete,         Ceiling  },
        { nn::pcv::Module_PciExClk,          Discrete,         Ceiling  },
        { nn::pcv::Module_PExUsbPhy,         Discrete,         Ceiling  },
        { nn::pcv::Module_XUsbPadCtl,        Discrete,         Ceiling  },
    };

INSTANTIATE_TEST_CASE_P(GetDiscreteRates, GetDiscreteRatesTest, testing::ValuesIn(discreteRatesModules));

TEST_P(GetDiscreteRatesTest, GetPossibleClockRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());
    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    // Expectations:
    // GetPossibleClockRates should succeed
    // There should be at least one valid rate returned (numRates > 0)
    // No more than g_MaxRates should be returned
    // Every rate returned should be > 0
    EXPECT_TRUE(queryResult.IsSuccess()) << "Failed to GetPossibleClockRates with module: " << GetModuleName(module);
    EXPECT_LT(0, numRates);
    EXPECT_GE(nn::pcv::MaxNumClockRates, numRates);
    EXPECT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    NN_LOG("Module: %s\n", GetModuleName(module));

    ASSERT_EQ(Discrete, GetParam().ratesListType);
    for (int i=0; i < numRates; ++i)
    {
        EXPECT_LT(0, rates[i]);
        NN_LOG("\t%.2f MHz\n", rates[i] / (1000.0 * 1000.0));
    }

}

// Invalid module passed to GetPossibleClockRates
// Expect result to fail.
TEST_F(GetDiscreteRatesTest, InvalidModule)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    // Invalid module
    nn::pcv::Module invalidMod = nn::pcv::Module_NumModule;

    int numRates = 0;
    nn::Result queryResult = nn::pcv::GetPossibleClockRates(&listType, rates, &numRates, invalidMod, nn::pcv::MaxNumClockRates);
    ASSERT_FALSE(queryResult.IsSuccess());
}

// Pass in max count = 0
// Expect an empty list
TEST_F(GetDiscreteRatesTest, MaxCount0)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];

    // Test on CPU
    nn::pcv::Module mod = nn::pcv::Module_Cpu;

    int numRates = 0;
    nn::Result queryResult = nn::pcv::GetPossibleClockRates(&listType, rates, &numRates, mod, 0);
    ASSERT_TRUE(queryResult.IsSuccess());
    EXPECT_EQ(0, numRates);
}

// Null tests
// Expect ??, currently kernel will break process and throw exception

TEST_F(GetDiscreteRatesTest, DISABLED_NullOutType)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());
}

TEST_F(GetDiscreteRatesTest, DISABLED_NullOutRates)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());
    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz *rates = NULL;

    //Test on CPU
    nn::pcv::Module mod = nn::pcv::Module_Cpu;

    int numRates = 0;
    nn::Result queryResult = nn::pcv::GetPossibleClockRates(&listType, rates, &numRates, mod, nn::pcv::MaxNumClockRates);
    ASSERT_FALSE(queryResult.IsSuccess());
}

TEST_F(GetDiscreteRatesTest, DISABLED_NullOutCount)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());
    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];

    //Test on CPU
    nn::pcv::Module mod = nn::pcv::Module_Cpu;
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int *numRates = NULL;
    nn::Result queryResult = nn::pcv::GetPossibleClockRates(&listType, rates, numRates, mod, nn::pcv::MaxNumClockRates);
    ASSERT_FALSE(queryResult.IsSuccess());
}

