﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/pctl/detail/service/pctl_PlayState.h>
#include <nn/pctl/detail/service/watcher/pctl_EventLog.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>

using namespace nn::pctl::detail::service;

PlayState g_PlayState;

class PlayStateTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    }
    static void TearDownTestCase()
    {
        nn::time::Finalize();
    }

protected:
    virtual void SetUp() NN_OVERRIDE
    {
        // テストケース開始時に毎回初期化を行う
        g_PlayState.InitializeData();
    }
    virtual void TearDown() NN_OVERRIDE
    {
    }
};

////////////////////////////////////////////////////////////////////////////////

uint8_t g_EventDataVersion = watcher::EventDataVersionCurrent;

void SetEventDataVersion(uint8_t version) NN_NOEXCEPT
{
    g_EventDataVersion = version;
}

inline int64_t CalcElapsedTime(int64_t lastElapsedTime, int64_t lastEventTimestamp, int64_t currentEventTimestamp)
{
    return lastElapsedTime + (currentEventTimestamp - lastEventTimestamp);
}

// 簡単のため timestamp は int64_t で受け取る
watcher::EventData MakeEventData(watcher::EventType eventType, int64_t timestamp,
    watcher::EventData::Payload payload, const nn::time::LocationName& locationName) NN_NOEXCEPT
{
    watcher::EventData e = {
        eventType, false, 0, { timestamp }, locationName, 0, g_EventDataVersion, {0}, payload
    };
    return e;
}

// 簡単のため timestamp は int64_t で受け取る
// ※ タイムゾーンはUTC固定
watcher::EventData MakeEventData(watcher::EventType eventType, int64_t timestamp,
    watcher::EventData::Payload payload) NN_NOEXCEPT
{
    return MakeEventData(eventType, timestamp, payload,
        nn::time::LocationName{ "UTC" });
}

watcher::EventData MakeDeviceActivateEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidDeviceActivate,
        timestamp, p);
}

watcher::EventData MakeDeviceLaunchEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidDeviceLaunch,
        timestamp, p);
}

watcher::EventData MakeIdleEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::Idle,
        timestamp, p);
}

watcher::EventData MakeApplicationLaunchEvent(int64_t timestamp,
    nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    p.applicationLaunch.applicationId = applicationId;
    return MakeEventData(watcher::EventType::DidApplicationLaunch,
        timestamp, p);
}

watcher::EventData MakeApplicationTerminateEvent(int64_t timestamp,
    nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    p.applicationTerminate.applicationId = applicationId;
    return MakeEventData(watcher::EventType::DidApplicationTerminate,
        timestamp, p);
}

watcher::EventData MakeApplicationSuspendEvent(int64_t timestamp,
    nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    p.suspendedApplication.applicationId = applicationId;
    return MakeEventData(watcher::EventType::DidApplicationSuspend,
        timestamp, p);
}

watcher::EventData MakeApplicationResumeEvent(int64_t timestamp,
    nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    p.resumedApplication.applicationId = applicationId;
    return MakeEventData(watcher::EventType::DidApplicationResume,
        timestamp, p);
}

watcher::EventData MakeApplicationPlayEvent(int64_t timestamp,
    nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    p.playedApplication.applicationId = applicationId;
    return MakeEventData(watcher::EventType::DidApplicationPlay,
        timestamp, p);
}

watcher::EventData MakeUnlockEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidUnlock,
        timestamp, p);
}

watcher::EventData MakeLockEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidLock,
        timestamp, p);
}

watcher::EventData MakeSleepEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidSleep,
        timestamp, p);
}

watcher::EventData MakeWakeupEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidWakeup,
        timestamp, p);
}

watcher::EventData MakeUserOpenEvent(int64_t timestamp,
    nn::account::Uid uid) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    p.openedUser.uid = uid;
    return MakeEventData(watcher::EventType::DidUserOpen,
        timestamp, p);
}

watcher::EventData MakeUserCloseEvent(int64_t timestamp,
    nn::account::Uid uid) NN_NOEXCEPT
{
    watcher::EventData::Payload p;
    p.closedUser.uid = uid;
    return MakeEventData(watcher::EventType::DidUserClose,
        timestamp, p);
}

watcher::EventData MakeShutdownEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidShutdown,
        timestamp, p);
}

watcher::EventData MakePlayTimerStartedEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidPlayTimerStart,
        timestamp, p);
}

watcher::EventData MakePlayTimerStoppedEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidPlayTimerStop,
        timestamp, p);
}

// (このイベント生成関数は地域名も受け取る)
watcher::EventData MakeLocationNameChangeEvent(int64_t timestamp, const nn::time::LocationName& locationName) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidLocationNameChange,
        timestamp, p, locationName);
}

watcher::EventData MakeUnexpectedShutdownOccurEvent(int64_t timestamp) NN_NOEXCEPT
{
    watcher::EventData::Payload p = {};
    return MakeEventData(watcher::EventType::DidUnexpectedShutdownOccur,
        timestamp, p);
}

////////////////////////////////////////////////////////////////////////////////

// アプリケーションを開始して終了した際の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartStop)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationStopped(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始してそのままの場合の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartRunning)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始・終了して再度開始しそのままの場合の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartStopStartRunning)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationStopped(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(150, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(200, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断した際の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartSuspend)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断、そのまま終了した際の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartSuspendStop)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStopped(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(250, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断、そのあと再開した際の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartSuspendResume)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(300, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断したときの「中断」が複数回ある場合のテスト
TEST_F(PlayStateTest, TestAppState_StartSuspendResume2)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(300, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認(重複して Suspend を呼び出しても問題ないことの確認)
    EXPECT_EQ(150, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断→再開したときの「再開」が複数回ある場合のテスト
TEST_F(PlayStateTest, TestAppState_StartSuspendResume3)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(400, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断、そのあと再開した際、プレイタイマーが途中で中断していた場合の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartSuspendResumeWithPlayTimerSuspend)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetPlayTimerEnabled(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(300);

    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(500);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(300, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    // プレイタイマー停止中は計上されない
    EXPECT_EQ(550, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断、そのあと再開した際、途中でスリープが挟まっているときの実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartSuspendResumeWithSleep)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetSleeping(true, timeAt);
    EXPECT_FALSE(g_PlayState.IsPlayTimerEnabled()); // プレイタイマーが止まるはず
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetSleeping(false, timeAt);
    g_PlayState.SetPlayTimerEnabled(true, timeAt); // プレイタイマーが止まるので再開させる
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(300, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(70, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始したあとスリープして復帰したときの実行時間テスト
// (※ 本来は中断・再開のイベントが起こるはず)
TEST_F(PlayStateTest, TestAppState_StartAndSleep)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetSleeping(true, timeAt);
    EXPECT_FALSE(g_PlayState.IsPlayTimerEnabled()); // プレイタイマーが止まるはず
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetSleeping(false, timeAt);
    g_PlayState.SetPlayTimerEnabled(true, timeAt); // プレイタイマーが止まるので再開させる
    timeAt += nn::TimeSpan::FromSeconds(20);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        // (明示的な中断・再開が無い場合はスリープ復帰時点でアプリケーションが稼働状態になるはず)
        EXPECT_EQ(120, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始したあとスリープして復帰したときに中断状態になった場合の実行時間テスト
TEST_F(PlayStateTest, TestAppState_StartAndSleep2)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetSleeping(true, timeAt);
    EXPECT_FALSE(g_PlayState.IsPlayTimerEnabled()); // プレイタイマーが止まるはず
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetSleeping(false, timeAt);
    g_PlayState.SetApplicationSuspended(appId, timeAt);
    g_PlayState.SetPlayTimerEnabled(true, timeAt); // プレイタイマーが止まるので再開させる
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        // スリープ復帰時に中断状態になるので 20 は含まれないはず
        EXPECT_EQ(300, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(20, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 途中で一時解除になるケース
TEST_F(PlayStateTest, TestAppState_StartRunningWithUnlock)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(150);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(200, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    EXPECT_EQ(100, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中の時間は計上されない)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 一時解除のままアプリが終了するケース
TEST_F(PlayStateTest, TestAppState_StartRunningWithUnlock2)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    EXPECT_EQ(100, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中の時間は計上されない)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 一時解除のままアプリが開始するケース
TEST_F(PlayStateTest, TestAppState_StartRunningWithUnlock3)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(150);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(150, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    EXPECT_EQ(150, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中の時間は計上されない)
    EXPECT_EQ(200, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// プレイタイマー開始前に一時解除が行われ、プレイタイマー開始してから一時解除のままアプリが開始するケース
TEST_F(PlayStateTest, TestAppState_StartRunningWithUnlock4)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(150);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(150, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 一時解除中の時間の確認(プレイタイマー停止中は計上されない)
    // 50 + 100
    EXPECT_EQ(150, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中とプレイタイマー停止中の時間は計上されない)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 何度も一時解除、間にスリープを挟む
TEST_F(PlayStateTest, TestAppState_StartRunningWithUnlockAndSleep)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetSleeping(true, timeAt);
    EXPECT_FALSE(g_PlayState.IsPlayTimerEnabled()); // プレイタイマーが止まるはず
    timeAt += nn::TimeSpan::FromSeconds(10);

    g_PlayState.SetSleeping(false, timeAt);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        // 200 + 20 + 800
        EXPECT_EQ(1020, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 50 + 100 + 400
    EXPECT_EQ(550, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
}

// 一時解除中にアプリが中断するケース
TEST_F(PlayStateTest, TestAppState_StartSuspendWithUnlock)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
        EXPECT_EQ(true, p->IsRunning());
    }
    EXPECT_EQ(300, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中の時間は計上されない)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 一時解除中にアプリが中断・再開するケース
TEST_F(PlayStateTest, TestAppState_StartSuspendWithUnlock2)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        // ずっと一時解除中なので計上されない
        EXPECT_EQ(50, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    EXPECT_EQ(700, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中の時間は計上されない)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 一時解除中にアプリが中断し、再開する前に一時解除が戻されたケース
TEST_F(PlayStateTest, TestAppState_StartSuspendWithUnlock3)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(850, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    EXPECT_EQ(300, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除を戻してから再開するまでの時間が計上される)
    EXPECT_EQ(400, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリが中断してから一時解除し、さらにプレイタイマーが一時停止するケース
TEST_F(PlayStateTest, TestAppState_StartSuspendUnlockAndTimerStop)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetPlayTimerEnabled(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);
    // プレイタイマー停止中の時間を含まない(停止中にチェックしたパターン)
    EXPECT_EQ(100, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());

    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);
    // プレイタイマー停止中の時間を含まない(再開後にチェックしたパターン)
    EXPECT_EQ(900, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(10);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(60, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
        EXPECT_EQ(true, p->IsRunning());
    }
    EXPECT_EQ(900, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中・プレイタイマー停止中の時間は計上されない)
    EXPECT_EQ(220, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリが中断してからプレイタイマーが一時停止し、その後に一時解除するケース
TEST_F(PlayStateTest, TestAppState_StartSuspendUnlockAndTimerStop2)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetPlayTimerEnabled(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);
    // プレイタイマー停止中の時間を含まない(停止中にチェックしたパターン)
    EXPECT_EQ(0, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());

    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);
    // プレイタイマー停止中の時間を含まない(再開後にチェックしたパターン)
    EXPECT_EQ(800, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(10);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(60, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
        EXPECT_EQ(true, p->IsRunning());
    }
    EXPECT_EQ(800, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中・プレイタイマー停止中の時間は計上されない)
    EXPECT_EQ(220, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始してそのまま過ぎ、日付を跨ぐ場合の実行時間テスト
TEST_F(PlayStateTest, TestAppState_DayChangedWithRunning)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    // 日付更新時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認(実行中のまま日付を跨いで停止していないので 0 となる)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して一定時間後に終了してから日付を跨ぐ場合の実行時間テスト
TEST_F(PlayStateTest, TestAppState_DayChangedWithStopped)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStopped(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // 日付更新時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    // 日付を跨いだときに停止している場合は無くなる
    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_EQ(nullptr, p);
    // 何もしていない時間の確認(停止中に日付を跨いでいるのでそこからの時間となる)
    EXPECT_EQ(200, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーションを開始して中断した後に日付を跨ぐ場合の実行時間テスト
TEST_F(PlayStateTest, TestAppState_DayChangedWithSuspendResume)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    // 日付更新時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(400, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
        EXPECT_EQ(400, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認(中断状態のまま日付を跨いでいるので 100 となる)
    EXPECT_EQ(100, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーション実行中にスリープして日付が変わり、復帰後に一時解除した場合のプレイ時間テスト
TEST_F(PlayStateTest, TestAppState_DayChangedAndUnlock)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    g_PlayState.SetSleeping(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // 日付変更時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetSleeping(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(10);

    EXPECT_FALSE(g_PlayState.IsPlayTimerEnabled()); // プレイタイマーが止まっているはず
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    auto pApp = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, pApp);
    if (pApp != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetApplicationRunningTime(*pApp, timeAt).GetSeconds());
        // 日付跨いでからのプレイ時間のみ
        EXPECT_EQ(50, g_PlayState.GetApplicationAnonymousPlayingTime(*pApp, timeAt).GetSeconds());
    }

    // 何もしていない時間の確認(スリープ復帰後即一時解除や Resume しているので 0 になる)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 複数のアプリケーションを開始して終了した際の実行時間テスト
TEST_F(PlayStateTest, TestAppState_MultiAppStartStop)
{
    const nn::ncm::ApplicationId appId1 = { 0x0100000000001000 };
    const nn::ncm::ApplicationId appId2 = { 0x0100000000001001 };
    const nn::ncm::ApplicationId appId3 = { 0x0100000000001002 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId1, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationStopped(appId1, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId2, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationSuspended(appId2, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(300);

    g_PlayState.SetApplicationStopped(appId2, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetApplicationStarted(appId3, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(500);

    g_PlayState.SetApplicationSuspended(appId3, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(600);

    g_PlayState.SetApplicationResumed(appId3, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(700);

    g_PlayState.SetApplicationStopped(appId3, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);

    auto e = g_PlayState.GetPlayedApplicationEnumerator();
    EXPECT_TRUE(e.HasCurrent());

    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        switch (p->applicationId.value)
        {
            case 0x0100000000001000:
                ++hitCount;
                EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
                break;
            case 0x0100000000001001:
                ++hitCount;
                EXPECT_EQ(200, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
                break;
            case 0x0100000000001002:
                ++hitCount;
                EXPECT_EQ(1200, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
                break;
            default:
                FAIL();
                break;
        }
        e.Next();
    }
    EXPECT_EQ(3, hitCount);

    // 何もしていない時間の確認
    // 50 + 300 + 400 + 600 + 800
    EXPECT_EQ(2150, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 一時解除中にアプリが中断し、シャットダウンするケース
TEST_F(PlayStateTest, TestAppState_StartSuspendWithUnlockAndShutdown)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.ProcessDeviceRunningStatus(false, timeAt, false);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, false);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
        // アプリ自体は停止しているはず
        EXPECT_EQ(false, p->IsRunning());
    }
    EXPECT_EQ(300, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
    // 何もしていない時間の確認(一時解除中の時間は計上されない)
    EXPECT_EQ(800, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリ実行後、強制電源断されていた(＝終了していないのに起動イベントがある)ケース1
TEST_F(PlayStateTest, TestAppState_StartAndForcedShutdown)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.lastTimestamp = timeAt;
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    g_PlayState.lastTimestamp = timeAt;
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, false);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.lastTimestamp = timeAt;
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
        // アプリ自体は停止しているはず
        EXPECT_EQ(false, p->IsRunning());
    }
    // 何もしていない時間の確認
    EXPECT_EQ(300, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリ実行後、強制電源断されていた(＝終了していないのに起動イベントがある)ケース2
// (一時解除やスリープも絡める)
TEST_F(PlayStateTest, TestAppState_StartAndForcedShutdown2)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // デバイス起動イベント＝(アプリ稼働中のどこかで)強制電源断あり
    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, false);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetSleeping(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);

    // デバイス起動イベント＝(スリープ中のどこかで)強制電源断あり
    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, false);
    EXPECT_FALSE(g_PlayState.IsPlayTimerEnabled()); // プレイタイマーが止まっているはず
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(10);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        // 50 + 400
        EXPECT_EQ(450, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
        // アプリ自体は停止しているはず
        EXPECT_EQ(false, p->IsRunning());
    }
    // 何もしていない時間の確認
    // 200 + 10
    EXPECT_EQ(210, g_PlayState.GetMiscTime(timeAt).GetSeconds());
    // 一時解除していた時間の確認
    EXPECT_EQ(100, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
}

// アプリ実行後、強制電源断されていた(＝終了していないのに起動イベントがある)ケース3
// 日付を跨いで電源がONになる
TEST_F(PlayStateTest, TestAppState_StartAndForcedShutdown3)
{
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    // ここで日付を跨ぐ

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    auto p = g_PlayState.FindApplicationStateData(appId);
    EXPECT_EQ(nullptr, p);
    // 何もしていない時間の確認
    EXPECT_EQ(200, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリ実行がないケース
TEST_F(PlayStateTest, TestAppState_NotRunning)
{
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.ProcessDeviceRunningStatus(false, timeAt, false);
    timeAt += nn::TimeSpan::FromSeconds(200);

    // 何もしていない時間の確認
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, false);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // まだプレイタイマーが動いていないので時間変動なし
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());

    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    EXPECT_EQ(450, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリ実行はないが強制電源断されていた(＝終了していないのに起動イベントがある)ケース
TEST_F(PlayStateTest, TestAppState_NotRunningAndForcedShutdown)
{
    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.lastTimestamp = timeAt;
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, false);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.lastTimestamp = timeAt;
    timeAt += nn::TimeSpan::FromSeconds(200);

    // 何もしていない時間の確認
    EXPECT_EQ(250, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

////////////////////////////////////////////////////////////////////////////////

// ユーザーを Open してそのままの場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_NoAppRunning)
{
    const nn::account::Uid uid = { {0x00000000, 0x00000001} };
    ASSERT_NE(uid, nn::account::InvalidUid);

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認(アプリケーションを起動していないので同じ時間になる)
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// ユーザーを Open して Close した場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_NoAppRunning2)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetUserClosed(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(50, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認(アプリケーションを起動していないので全体の合計時間になる)
    EXPECT_EQ(150, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// ユーザーを Open して、そのあとにアプリケーションを開始した場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_AppStartAfterUserOpen)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間は無い
            EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間
            EXPECT_EQ(150, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間
            EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動していない時間と同じになる)
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// ユーザーを Open して、そのあとにアプリケーションを開始し、さらに別ユーザーを Open した場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_AppStartAfterUserOpen2)
{
    const nn::account::Uid uid1 = { { 0x00000000, 0x00000001 } };
    const nn::account::Uid uid2 = { { 0x00000000, 0x00000002 } };
    ASSERT_NE(uid1, nn::account::InvalidUid);
    ASSERT_NE(uid2, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid1, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetUserOpened(uid2, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(300, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間は無い
            EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid1);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間
            EXPECT_EQ(350, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        p = g_PlayState.FindApplicationUserGlobalStateData(uid2);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間(開始前は Open していないのでアプリごとのプレイ時間と一致)
            EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            if (p->uid == uid1)
            {
                // アプリケーション単位のプレイ時間
                EXPECT_EQ(300, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
            }
            else if (p->uid == uid2)
            {
                // アプリケーション単位のプレイ時間
                EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
            }
            else
            {
                FAIL();
            }
        }
        e.Next();
    }
    EXPECT_EQ(2, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動していない時間と同じになる)
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// ユーザーを Open して、そのあとにアプリケーションを開始・終了した場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_AppStartStopAfterUserOpen)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationStopped(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間
            EXPECT_EQ(350, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間
            EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動していない時間と同じになる)
    EXPECT_EQ(250, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 先にアプリケーションを開始して、終了するまでに Open/Close をした場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_UserOpenCloseAfterAppStart)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetUserClosed(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationStopped(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(350, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間
            EXPECT_EQ(250, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間はアプリケーションをプレイした時間と同じ
            EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間
            EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動していない時間と同じになる)
    EXPECT_EQ(400, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 先にアプリケーションを開始して、終了するまでに ユーザーOpen・アプリ中断・再開・ユーザーClose をした場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_UserOpenCloseAfterAppStart2)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    //        +50 +100  +10 +20  +200 +400 +800
    //        0  50   150 160  180  380  780  1580
    // All    +---------------------------------+
    // App_1  +--------sssss---------sssss+
    // User       +-------------+
    // User$1     xxxxx     xxxxx
    // Anon$1 xxxx               xxxx
    // Misc            xxxxx         xxxxxxxxxxxx

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(10);

    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetUserClosed(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetApplicationStopped(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // (50 + 100 + 20 + 200)
            EXPECT_EQ(370, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間(50 + 200)
            EXPECT_EQ(250, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間はアプリケーションをプレイした時間とは異なり、中断中も含む
            // (100 + 10 + 20)
            EXPECT_EQ(130, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間(100 + 20)
            EXPECT_EQ(120, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動していない時間と同じになる)
    // (10 + 400 + 800)
    EXPECT_EQ(1210, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 先にアプリケーションを開始してSuspendし、終了するまでに Open/Close をした場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_UserOpenCloseAfterAppSuspend)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);

    g_PlayState.SetApplicationStarted(appId, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(10);
    g_PlayState.SetApplicationSuspended(appId, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(50);
    g_PlayState.SetUserOpened(uid, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(100);
    g_PlayState.SetUserClosed(uid, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(200);
    g_PlayState.SetApplicationStopped(appId, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(400);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(10, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間
            EXPECT_EQ(10, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間はOpen～Closeの時間
            EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間
            // SuspendしてからOpenになっており、その後そのままStopしたので 0 となる
            EXPECT_EQ(0, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認
    // 750 = 50 + 100 + 200 + 400
    EXPECT_EQ(750, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// 先にアプリケーションを開始してSuspendし、終了するまでに Open(User) → Resume(App) → Close(User) をした場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_UserOpenCloseAfterAppSuspend2)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);

    g_PlayState.SetApplicationStarted(appId, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(10);
    g_PlayState.SetApplicationSuspended(appId, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(50);
    g_PlayState.SetUserOpened(uid, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(100);
    g_PlayState.SetApplicationResumed(appId, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(200);
    g_PlayState.SetUserClosed(uid, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(400);
    g_PlayState.SetApplicationStopped(appId, timeAt);

    timeAt += nn::TimeSpan::FromSeconds(800);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(610, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間
            EXPECT_EQ(410, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間はOpen～Closeの時間
            EXPECT_EQ(300, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間
            // SuspendしてからOpenしているため、プレイ時間はResumeしてからの時間となる
            EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認
    // 950 = 50 + 100 + 800
    EXPECT_EQ(950, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// ユーザーを Open して、そのあとにアプリケーションを開始し一時解除状態にした場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_AppStartAfterUserOpenWithUnlock)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間は無い
            EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間
            EXPECT_EQ(150, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間
            EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動してない時間・一時解除を含まない時間と同じになる)
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());

    EXPECT_EQ(200, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
}

// ユーザーを Open して、そのあとに一時解除状態にしてからアプリケーションを開始した場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_AppStartAfterUserOpenWithUnlock2)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 一時解除中なので 0
            EXPECT_EQ(0, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間は無い
            EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間
            EXPECT_EQ(50, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間は一時解除中なので 0
            EXPECT_EQ(0, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動してない時間・一時解除を含まない時間と同じになる)
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());

    EXPECT_EQ(300, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
}

// ユーザーを Open して、そのあとに一時解除状態にしてからアプリケーションを開始し、さらに一時解除を戻した場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_AppStartAfterUserOpenWithUnlock3)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(400, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間は無い
            EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 全体のプレイ時間
            EXPECT_EQ(450, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間は一時解除の期間を除く
            EXPECT_EQ(400, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(アプリケーションを起動してない時間・一時解除を含まない時間と同じになる)
    EXPECT_EQ(50, g_PlayState.GetMiscTime(timeAt).GetSeconds());

    EXPECT_EQ(300, g_PlayState.GetTemporaryUnlockedTime(timeAt).GetSeconds());
}

// ユーザーを Open してそのまま日付が変わった場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_DayChangedWithNoAppRunning)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    // 日付変更時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認(アプリケーションを起動していないので同じ時間になる)
    EXPECT_EQ(100, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// ユーザーを Open して Close してから日付が変わった場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_DayChangedWithNoAppRunning2)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetUserClosed(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    // 日付変更時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // 日付が変わった場合 Close 済みのユーザーは出てこない
    auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
    EXPECT_EQ(nullptr, p);

    // 何もしていない時間の確認(アプリケーションを起動していないので同じ時間になる)
    EXPECT_EQ(100, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーション実行中にユーザーを Open してそのまま日付が変わった場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_DayChangedWithAppRunning)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(20);

    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    // 日付変更時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    auto pApp = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, pApp);
    if (pApp != nullptr)
    {
        EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*pApp, timeAt).GetSeconds());
        // 日付跨いでからはユーザーがずっと Open なので 0 になる
        EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*pApp, timeAt).GetSeconds());
    }

    auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
    EXPECT_NE(nullptr, p);
    if (p != nullptr)
    {
        EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
    }
    // 何もしていない時間の確認(アプリケーションを起動しているので 0 になる)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// ユーザーを Open して、そのあとにアプリケーションを開始し地域名変更を行った場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_AppStartAfterUserOpenWithLocationNameChanged)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // イベントを直接与えることで判定させる
    // (Asia/Tokyo における PosixTime: 10150 のローカル時計はUTC比較で日付が変わらない)
    g_PlayState.UpdatePlayStateFromEvent(MakeLocationNameChangeEvent(
        timeAt.value, nn::time::LocationName{ "Asia/Tokyo" }
    ));
    timeAt += nn::TimeSpan::FromSeconds(200);

    // アプリケーションの状態のチェック
    {
        auto p = g_PlayState.FindApplicationStateData(appId);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 地域名が変わってからの時間になる
            EXPECT_EQ(200, g_PlayState.GetApplicationRunningTime(*p, timeAt).GetSeconds());
            // ユーザー非選択の期間は無い
            EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    // ユーザー状態のチェック
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid);
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            // 地域名が変わってからの時間になる
            EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
    }

    auto e = g_PlayState.GetPlayedUserEnumerator(appId);
    EXPECT_TRUE(e.HasCurrent());
    int hitCount = 0;
    while (e.HasCurrent())
    {
        auto p = e.GetCurrent();
        EXPECT_NE(nullptr, p);
        if (p != nullptr)
        {
            EXPECT_EQ(uid, p->uid);
            EXPECT_EQ(appId, p->applicationId);
            ++hitCount;
            // アプリケーション単位のプレイ時間
            // (地域名が変わってからの時間になる)
            EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeAt).GetSeconds());
        }
        e.Next();
    }
    EXPECT_EQ(1, hitCount);

    // 何もしていない時間の確認(地域名が変わってからは 0)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

// アプリケーション実行中にユーザーを Open してそのまま日付が変わった場合のプレイ時間テスト
TEST_F(PlayStateTest, TestUserState_Other1)
{
    const nn::account::Uid uid = { { 0x00000000, 0x00000001 } };
    ASSERT_NE(uid, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId = { 0x0100000000001000 };

    nn::time::PosixTime timeAt = { 10000 };

    g_PlayState.ProcessDeviceRunningStatus(true, timeAt, true);
    g_PlayState.SetPlayTimerEnabled(true, timeAt);

    g_PlayState.SetApplicationStarted(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(20);

    //g_PlayState.SetUserOpened(uid, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(50);

    g_PlayState.SetApplicationSuspended(appId, timeAt);
    g_PlayState.SetSleeping(true, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(100);

    // 日付変更時の処理
    g_PlayState.ResetPlayedStatus(timeAt);
    timeAt += nn::TimeSpan::FromSeconds(200);

    g_PlayState.SetSleeping(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(10);

    EXPECT_FALSE(g_PlayState.IsPlayTimerEnabled()); // プレイタイマーが止まっているはず
    g_PlayState.SetPlayTimerEnabled(true, timeAt);
    g_PlayState.SetTemporaryUnlocked(true, timeAt);
    g_PlayState.SetApplicationResumed(appId, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(400);

    g_PlayState.SetTemporaryUnlocked(false, timeAt);
    timeAt += nn::TimeSpan::FromSeconds(800);

    auto pApp = g_PlayState.FindApplicationStateData(appId);
    EXPECT_NE(nullptr, pApp);
    if (pApp != nullptr)
    {
        EXPECT_EQ(800, g_PlayState.GetApplicationRunningTime(*pApp, timeAt).GetSeconds());
        // 日付跨いでからのプレイ時間のみ
        EXPECT_EQ(800, g_PlayState.GetApplicationAnonymousPlayingTime(*pApp, timeAt).GetSeconds());
    }

    // 何もしていない時間の確認(スリープ復帰後即一時解除や Resume しているので 0 になる)
    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeAt).GetSeconds());
}

////////////////////////////////////////////////////////////////////////////////

// イベントデータから状態を更新するテスト
TEST_F(PlayStateTest, TestFromEvent_Complex1)
{
    const nn::account::Uid uid1 = { { 0x00000000, 0x00000001 } };
    const nn::account::Uid uid2 = { { 0x00000000, 0x00000002 } };
    ASSERT_NE(uid1, nn::account::InvalidUid);
    ASSERT_NE(uid2, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId1 = { 0x0100000000001000 };
    const nn::ncm::ApplicationId appId2 = { 0x0100000000001001 };

    PlayStatePlayedApplicationEnumerator appEnumerator;
    PlayStatePlayedUserEnumerator userEnumerator;

    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceLaunchEvent        ( 100));
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent    ( 100));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent   ( 200, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationSuspendEvent  ( 300, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent( 400, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeIdleEvent                ( 450));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent   ( 500, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserOpenEvent            ( 500, uid1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationSuspendEvent  ( 600, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationResumeEvent   ( 700, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent( 800, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserCloseEvent           ( 800, uid1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent   ( 900, appId2));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserOpenEvent            ( 900, uid2));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserOpenEvent            (1200, uid1));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserCloseEvent           (1300, uid1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent(1400, appId2));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserCloseEvent           (1400, uid2));
    g_PlayState.UpdatePlayStateFromEvent(MakeSleepEvent               (1500));
    nn::time::PosixTime timeLast = { 1500 };

    // 100 + 100 + 50 + 50 + 100 + 100 + 100
    EXPECT_EQ(600, g_PlayState.GetMiscTime(timeLast).GetSeconds());
    //
    {
        appEnumerator = g_PlayState.GetPlayedApplicationEnumerator();
        EXPECT_TRUE(appEnumerator.HasCurrent());

        int hitCount = 0;
        while (appEnumerator.HasCurrent())
        {
            auto p = appEnumerator.GetCurrent();
            if (p->applicationId == appId1)
            {
                ++hitCount;
                // 100 + 100 + 100
                EXPECT_EQ(300, g_PlayState.GetApplicationRunningTime(*p, timeLast).GetSeconds());
                // 100 + 0
                EXPECT_EQ(100, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeLast).GetSeconds());
            }
            else if (p->applicationId == appId2)
            {
                ++hitCount;
                // 500
                EXPECT_EQ(500, g_PlayState.GetApplicationRunningTime(*p, timeLast).GetSeconds());
                // 0
                EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeLast).GetSeconds());
            }
            else
            {
                FAIL();
            }
            appEnumerator.Next();
        }
        EXPECT_EQ(2, hitCount);
    }

    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid1);
        EXPECT_NE(p, nullptr);
        if (p != nullptr)
        {
            // 300 + 100
            EXPECT_EQ(400, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
        }
    }
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid2);
        EXPECT_NE(p, nullptr);
        if (p != nullptr)
        {
            // 500
            EXPECT_EQ(500, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
        }
    }

    //
    {
        userEnumerator = g_PlayState.GetPlayedUserEnumerator(appId1);
        EXPECT_TRUE(userEnumerator.HasCurrent());

        int hitCount = 0;
        while (userEnumerator.HasCurrent())
        {
            auto p = userEnumerator.GetCurrent();
            if (p->uid == uid1)
            {
                ++hitCount;
                // 100 + 100
                EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
            }
            else if (p->uid == uid2)
            {
                EXPECT_TRUE(false && !!"uid2 must not be played");
            }
            else
            {
                FAIL();
            }
            userEnumerator.Next();
        }
        EXPECT_EQ(1, hitCount);
    }

    //
    {
        userEnumerator = g_PlayState.GetPlayedUserEnumerator(appId2);
        EXPECT_TRUE(userEnumerator.HasCurrent());

        int hitCount = 0;
        while (userEnumerator.HasCurrent())
        {
            auto p = userEnumerator.GetCurrent();
            if (p->uid == uid1)
            {
                ++hitCount;
                // 100
                EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
            }
            else if (p->uid == uid2)
            {
                ++hitCount;
                // 300 + 100 + 100
                EXPECT_EQ(500, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
            }
            else
            {
                FAIL();
            }
            userEnumerator.Next();
        }
        EXPECT_EQ(2, hitCount);
    }

    // 日付変更
    {
        nn::time::CalendarTime calTime = nn::time::ToCalendarTimeInUtc(timeLast);
        timeLast.value = 24 * 60 * 60;
        nn::time::CalendarTime calTime2 = nn::time::ToCalendarTimeInUtc(timeLast);
        EXPECT_NE(
            nn::time::DateToDays(calTime.year, calTime.month, calTime.day),
            nn::time::DateToDays(calTime2.year, calTime2.month, calTime2.day)
            );
    }
    g_PlayState.UpdatePlayStateFromEvent(MakeWakeupEvent(timeLast.value));

    EXPECT_EQ(0, g_PlayState.GetMiscTime(timeLast).GetSeconds());
    {
        auto e = g_PlayState.GetPlayedApplicationEnumerator();
        EXPECT_FALSE(e.HasCurrent());
    }
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid1);
        EXPECT_EQ(p, nullptr);
        p = g_PlayState.FindApplicationUserGlobalStateData(uid2);
        EXPECT_EQ(p, nullptr);
    }
    {
        auto e = g_PlayState.GetPlayedUserEnumerator(appId1);
        EXPECT_FALSE(e.HasCurrent());
    }
    {
        auto e = g_PlayState.GetPlayedUserEnumerator(appId2);
        EXPECT_FALSE(e.HasCurrent());
    }

} // NOLINT(impl/function_size)

// イベントデータから状態を更新するテスト2
TEST_F(PlayStateTest, TestFromEvent_Complex2)
{
    const nn::account::Uid uid1 = { { 0x00000000, 0x00000001 } };
    const nn::account::Uid uid2 = { { 0x00000000, 0x00000002 } };
    ASSERT_NE(uid1, nn::account::InvalidUid);
    ASSERT_NE(uid2, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId1 = { 0x0100000000001000 };
    const nn::ncm::ApplicationId appId2 = { 0x0100000000001001 };

    // 旧イベントが残っている仮定(中途半端に残っている)
    SetEventDataVersion(watcher::EventDataVersionInitial);
    g_PlayState.UpdatePlayStateFromEvent(MakeWakeupEvent              ( 100));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationResumeEvent   ( 200, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent( 300, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent   ( 400, appId2));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent( 500, appId2));
    // ここから新イベント込み
    SetEventDataVersion(watcher::EventDataVersionCurrent);
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceLaunchEvent        (1000));
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent    (1000));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent   (1100, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserOpenEvent            (1100, uid2));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationSuspendEvent  (1200, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent(1300, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserCloseEvent           (1300, uid2));
    g_PlayState.UpdatePlayStateFromEvent(MakeSleepEvent               (1400));
    nn::time::PosixTime timeLast = { 1400 };

    // ※ 旧イベント分はすべて計上されないことを考慮した確認

    // 100 + 100 + 100
    EXPECT_EQ(300, g_PlayState.GetMiscTime(timeLast).GetSeconds());
    //
    {
        auto e = g_PlayState.GetPlayedApplicationEnumerator();
        EXPECT_TRUE(e.HasCurrent());

        int hitCount = 0;
        while (e.HasCurrent())
        {
            auto p = e.GetCurrent();
            if (p->applicationId == appId1)
            {
                ++hitCount;
                // 100
                EXPECT_EQ(100, g_PlayState.GetApplicationRunningTime(*p, timeLast).GetSeconds());
                // 0
                EXPECT_EQ(0, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeLast).GetSeconds());
            }
            else if (p->applicationId == appId2)
            {
                // 出てこないはず
                EXPECT_TRUE(false && !!"appId2 must not be played");
            }
            else
            {
                FAIL();
            }
            e.Next();
        }
        EXPECT_EQ(1, hitCount);
    }

    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid1);
        EXPECT_EQ(p, nullptr);
    }
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid2);
        EXPECT_NE(p, nullptr);
        if (p != nullptr)
        {
            // 200
            EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
        }
    }

    //
    {
        auto e = g_PlayState.GetPlayedUserEnumerator(appId1);
        EXPECT_TRUE(e.HasCurrent());

        int hitCount = 0;
        while (e.HasCurrent())
        {
            auto p = e.GetCurrent();
            if (p->uid == uid1)
            {
                EXPECT_TRUE(false && !!"uid1 must not be played");
            }
            else if (p->uid == uid2)
            {
                ++hitCount;
                // 100
                EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
            }
            else
            {
                FAIL();
            }
            e.Next();
        }
        EXPECT_EQ(1, hitCount);
    }

    //
    {
        auto e = g_PlayState.GetPlayedUserEnumerator(appId2);
        EXPECT_FALSE(e.HasCurrent());
    }
} // NOLINT(impl/function_size)

// イベントデータから状態を更新するテスト3
TEST_F(PlayStateTest, TestFromEvent_Complex3)
{
    const nn::account::Uid uid1 = { { 0x00000000, 0x00000001 } };
    const nn::account::Uid uid2 = { { 0x00000000, 0x00000002 } };
    ASSERT_NE(uid1, nn::account::InvalidUid);
    ASSERT_NE(uid2, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId1 = { 0x0100000000001000 };
    const nn::ncm::ApplicationId appId2 = { 0x0100000000001001 };

    SetEventDataVersion(watcher::EventDataVersionCurrent);
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceActivateEvent      (1000));
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent    (1000));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent   (1000, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserOpenEvent            (1100, uid2));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationSuspendEvent  (1200, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent(1300, appId1));
    g_PlayState.UpdatePlayStateFromEvent(MakeUserCloseEvent           (1300, uid2));
    g_PlayState.UpdatePlayStateFromEvent(MakeSleepEvent               (1400));
    nn::time::PosixTime timeLast = { 1400 };

    // ※ 旧イベント分はすべて計上されないことを考慮した確認

    // 100 + 100
    EXPECT_EQ(200, g_PlayState.GetMiscTime(timeLast).GetSeconds());
    //
    {
        auto e = g_PlayState.GetPlayedApplicationEnumerator();
        EXPECT_TRUE(e.HasCurrent());

        int hitCount = 0;
        while (e.HasCurrent())
        {
            auto p = e.GetCurrent();
            if (p->applicationId == appId1)
            {
                ++hitCount;
                // 100 + 100
                EXPECT_EQ(200, g_PlayState.GetApplicationRunningTime(*p, timeLast).GetSeconds());
                // 100
                EXPECT_EQ(100, g_PlayState.GetApplicationAnonymousPlayingTime(*p, timeLast).GetSeconds());
            }
            else if (p->applicationId == appId2)
            {
                // 出てこないはず
                EXPECT_TRUE(false && !!"appId2 must not be played");
            }
            else
            {
                FAIL();
            }
            e.Next();
        }
        EXPECT_EQ(1, hitCount);
    }

    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid1);
        EXPECT_EQ(p, nullptr);
    }
    {
        auto p = g_PlayState.FindApplicationUserGlobalStateData(uid2);
        EXPECT_NE(p, nullptr);
        if (p != nullptr)
        {
            // 200
            EXPECT_EQ(200, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
        }
    }

    //
    {
        auto e = g_PlayState.GetPlayedUserEnumerator(appId1);
        EXPECT_TRUE(e.HasCurrent());

        int hitCount = 0;
        while (e.HasCurrent())
        {
            auto p = e.GetCurrent();
            if (p->uid == uid1)
            {
                EXPECT_TRUE(false && !!"uid1 must not be played");
            }
            else if (p->uid == uid2)
            {
                ++hitCount;
                // 100
                EXPECT_EQ(100, g_PlayState.GetUserPlayingTime(*p, timeLast).GetSeconds());
            }
            else
            {
                FAIL();
            }
            e.Next();
        }
        EXPECT_EQ(1, hitCount);
    }

    //
    {
        auto e = g_PlayState.GetPlayedUserEnumerator(appId2);
        EXPECT_FALSE(e.HasCurrent());
    }
} // NOLINT(impl/function_size)

// イベントデータからプレイ時間を計算するテスト1
TEST_F(PlayStateTest, TestFromEvent_PlayTime1)
{
    const nn::account::Uid uid1 = { { 0x00000000, 0x00000002 } };
    ASSERT_NE(uid1, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId1 = { 0x0100000000001000 };

    // UTCで日付が変わったタイミングの時刻を計算
    nn::time::PosixTime timeDayChange = { 86400 };

    // UpdatePlayStateFromEvent 実行前の lastElapsedTimeSecond
    int64_t preLastElapsedTimeSecond = 0;

    SetEventDataVersion(watcher::EventDataVersionCurrent);
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceActivateEvent(0));
    EXPECT_EQ(0, g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(1000));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // プレイタイマーが始まるまではカウントアップされない

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(1000, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1000, 1000), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeUserOpenEvent(1100, uid1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1000, 1100), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationSuspendEvent(1200, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1100, 1200), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent(1300, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1200, 1300), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeUserCloseEvent(1300, uid1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1300, 1300), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeSleepEvent(1400));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1300, 1400), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeWakeupEvent(1800));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // スリープして復帰したらタイマーは止まっている

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(2000));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // プレイタイマーが始まるまではカウントアップされない

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(2100, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 2000, 2100), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent(2200, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 2100, 2200), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeShutdownEvent(2300));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 2200, 2300), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceLaunchEvent(timeDayChange.value + 100));
    EXPECT_EQ(0, g_PlayState.lastElapsedTimeSeconds); // 日付が変わったので 0 にリセット

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(timeDayChange.value + 200));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // プレイタイマーが始まるまではカウントアップされない

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(timeDayChange.value + 300, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 200, 300), g_PlayState.lastElapsedTimeSeconds);
} // NOLINT(impl/function_size)

// イベントデータからプレイ時間を計算するテスト2(強制電源断パターン)
TEST_F(PlayStateTest, TestFromEvent_PlayTime2)
{
    const nn::account::Uid uid1 = { { 0x00000000, 0x00000002 } };
    ASSERT_NE(uid1, nn::account::InvalidUid);
    const nn::ncm::ApplicationId appId1 = { 0x0100000000001000 };

    // UTCで日付が変わったタイミングの時刻を計算
    nn::time::PosixTime timeDayChange = { 86400 };

    // UpdatePlayStateFromEvent 実行前の lastElapsedTimeSecond
    int64_t preLastElapsedTimeSecond = 0;

    SetEventDataVersion(watcher::EventDataVersionCurrent);
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceActivateEvent(0));
    EXPECT_EQ(0, g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(1000));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // プレイタイマーが始まるまではカウントアップされない

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(1000, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1000, 1000), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeUserOpenEvent(1100, uid1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1000, 1100), g_PlayState.lastElapsedTimeSeconds);

    // (強制電源断とその後の起動1)
    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeUnexpectedShutdownOccurEvent(1600));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1100, 1600), g_PlayState.lastElapsedTimeSeconds); // 10分未満の場合起動と強制電源断は同時刻

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceLaunchEvent(1600));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 1600, 1600), g_PlayState.lastElapsedTimeSeconds); // 前回イベントから10分未満はそのまま計上される

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(2000));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // 電源落ちして復帰したらタイマーは止まっている

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(2100, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 2000, 2100), g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationTerminateEvent(2200, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 2100, 2200), g_PlayState.lastElapsedTimeSeconds);

    // (強制電源断とその後の起動2)
    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeUnexpectedShutdownOccurEvent(2200 + 600));
    EXPECT_EQ(preLastElapsedTimeSecond + 600, g_PlayState.lastElapsedTimeSeconds); // 前回のイベントから10分後に強制電源断されたとみなされる

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceLaunchEvent(3200));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // 前回イベントから10分超えた分は10分だけ計上される

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(3500));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // 電源落ちして復帰したらタイマーは止まっている

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(3700, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 3500, 3700), g_PlayState.lastElapsedTimeSeconds);

    // (強制電源断とその後の起動3)
    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeUnexpectedShutdownOccurEvent(3800));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 3700, 3800), g_PlayState.lastElapsedTimeSeconds); // 日付変更前の電源断

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceLaunchEvent(timeDayChange.value + 800));
    EXPECT_EQ(0, g_PlayState.lastElapsedTimeSeconds); // 強制電源断から日付が変わった場合はリセット

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(timeDayChange.value + 900));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // プレイタイマーが始まるまではカウントアップされない

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(timeDayChange.value + 1000, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 900, 1000), g_PlayState.lastElapsedTimeSeconds);

    // (強制電源断とその後の起動4)
    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeUnexpectedShutdownOccurEvent(timeDayChange.value * 2 + 700));
    EXPECT_EQ(0, g_PlayState.lastElapsedTimeSeconds); // 日付変更後の電源断ではリセット

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeDeviceLaunchEvent(timeDayChange.value * 2 + 800));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds);

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakePlayTimerStartedEvent(timeDayChange.value * 2 + 900));
    EXPECT_EQ(preLastElapsedTimeSecond, g_PlayState.lastElapsedTimeSeconds); // プレイタイマーが始まるまではカウントアップされない

    preLastElapsedTimeSecond = g_PlayState.lastElapsedTimeSeconds;
    g_PlayState.UpdatePlayStateFromEvent(MakeApplicationLaunchEvent(timeDayChange.value * 2 + 1000, appId1));
    EXPECT_EQ(CalcElapsedTime(preLastElapsedTimeSecond, 900, 1000), g_PlayState.lastElapsedTimeSeconds);

} // NOLINT(impl/function_size)
