﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>

class SettingsTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_OVERRIDE
    {
        // テストケース開始時に毎回初期化を行う
        nn::pctl::DeleteSettings();
    }

    virtual void TearDown() NN_OVERRIDE
    {
        // テストケース終了時にも(ごみが残らないように)毎回初期化を行う
        nn::pctl::DeleteSettings();
    }
};

// SafetyLevelの設定テスト
TEST_F(SettingsTest, TestSetSafetyLevel)
{
    nn::pctl::SetPinCode("0123");
    EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());

    for (int i = 0; i < nn::pctl::SafetyLevel::SafetyLevel_Max; ++i)
    {
        nn::pctl::SafetyLevel level = static_cast<nn::pctl::SafetyLevel>(i);
        nn::pctl::SetSafetyLevel(level);
        EXPECT_EQ(level, nn::pctl::GetSafetyLevel());
    }
}

// SafetyLevelの設定テスト2(解除コード未設定でも正しく設定可能かどうか)
TEST_F(SettingsTest, TestSetSafetyLevel2)
{
    EXPECT_FALSE(nn::pctl::IsRestrictionEnabled());

    for (int i = 0; i < nn::pctl::SafetyLevel::SafetyLevel_Max; ++i)
    {
        nn::pctl::SafetyLevel level = static_cast<nn::pctl::SafetyLevel>(i);
        nn::pctl::SetSafetyLevel(level);
        EXPECT_EQ(level, nn::pctl::GetSafetyLevel());
    }
}

// SafetyLevelに基づいたプリセット値が現在の設定に反映されるかどうか
// Customの場合は SetCustomSafetyLevelSettings で設定した値がLevel変更で反映されるかどうか
TEST_F(SettingsTest, TestGetCurrentSettings)
{
    nn::pctl::SetPinCode("0123");

    nn::pctl::SafetyLevelSettings settingsA;
    nn::pctl::SafetyLevelSettings settingsB;
    nn::pctl::SafetyLevelSettings settingsCustom;
    settingsCustom.ratingAge = 15;
    settingsCustom.freeCommunicationRestriction = false;
    settingsCustom.snsPostRestriction = true;
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);
    for (int i = 0; i < nn::pctl::SafetyLevel::SafetyLevel_Max; ++i)
    {
        nn::pctl::SafetyLevel level = static_cast<nn::pctl::SafetyLevel>(i);
        nn::pctl::SetSafetyLevel(level);

        nn::pctl::GetSafetyLevelSettings(&settingsA, level);
        nn::pctl::GetCurrentSettings(&settingsB);
        NN_LOG("[pctl] check safety level = %d\n", i);
        EXPECT_EQ(settingsA.ratingAge, settingsB.ratingAge);
        EXPECT_EQ(settingsA.freeCommunicationRestriction, settingsB.freeCommunicationRestriction);
        EXPECT_EQ(settingsA.snsPostRestriction, settingsB.snsPostRestriction);

        if (level == nn::pctl::SafetyLevel::SafetyLevel_Custom)
        {
            EXPECT_EQ(settingsCustom.ratingAge, settingsB.ratingAge);
            EXPECT_EQ(settingsCustom.freeCommunicationRestriction, settingsB.freeCommunicationRestriction);
            EXPECT_EQ(settingsCustom.snsPostRestriction, settingsB.snsPostRestriction);
        }
    }
}

// SafetyLevelに基づいたプリセット値が現在の設定に反映されるかどうか2(解除コード未設定の場合)
// Customの場合は SetCustomSafetyLevelSettings で設定した値がLevel変更で反映されるかどうか
TEST_F(SettingsTest, TestGetCurrentSettings2)
{
    EXPECT_FALSE(nn::pctl::IsRestrictionEnabled());

    nn::pctl::SafetyLevelSettings settingsA;
    nn::pctl::SafetyLevelSettings settingsB;
    nn::pctl::SafetyLevelSettings settingsCustom;
    settingsCustom.ratingAge = 15;
    settingsCustom.freeCommunicationRestriction = false;
    settingsCustom.snsPostRestriction = true;
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);
    for (int i = 0; i < nn::pctl::SafetyLevel::SafetyLevel_Max; ++i)
    {
        nn::pctl::SafetyLevel level = static_cast<nn::pctl::SafetyLevel>(i);
        nn::pctl::SetSafetyLevel(level);

        nn::pctl::GetSafetyLevelSettings(&settingsA, level);
        nn::pctl::GetCurrentSettings(&settingsB);
        NN_LOG("[pctl] check safety level = %d\n", i);
        EXPECT_EQ(settingsA.ratingAge, settingsB.ratingAge);
        EXPECT_EQ(settingsA.freeCommunicationRestriction, settingsB.freeCommunicationRestriction);
        EXPECT_EQ(settingsA.snsPostRestriction, settingsB.snsPostRestriction);

        if (level == nn::pctl::SafetyLevel::SafetyLevel_Custom)
        {
            EXPECT_EQ(settingsCustom.ratingAge, settingsB.ratingAge);
            EXPECT_EQ(settingsCustom.freeCommunicationRestriction, settingsB.freeCommunicationRestriction);
            EXPECT_EQ(settingsCustom.snsPostRestriction, settingsB.snsPostRestriction);
        }
    }
}

// 先にSafetyLevelをセットしてからCustomSettingsをセットする
TEST_F(SettingsTest, TestSetCurrentCustomSettings_AfterLevelSet)
{
    nn::pctl::SetPinCode("0123");

    nn::pctl::SafetyLevelSettings settings;
    nn::pctl::SafetyLevelSettings settingsCustom;
    settingsCustom.ratingAge = 17;
    settingsCustom.freeCommunicationRestriction = false;
    settingsCustom.snsPostRestriction = true;
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);

    nn::pctl::GetCurrentSettings(&settings);

    EXPECT_EQ(settingsCustom.ratingAge, settings.ratingAge);
    EXPECT_EQ(settingsCustom.freeCommunicationRestriction, settings.freeCommunicationRestriction);
    EXPECT_EQ(settingsCustom.snsPostRestriction, settings.snsPostRestriction);
}

// RestrictedFeaturesのテスト(値の組み合わせは抜粋)
TEST_F(SettingsTest, TestGetRestrictedFeatures)
{
    nn::pctl::SetPinCode("0123");

    nn::pctl::SafetyLevelSettings settingsCustom;

    // カスタムを使ってテスト
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);

    settingsCustom.ratingAge = 15;
    settingsCustom.freeCommunicationRestriction = true;
    settingsCustom.snsPostRestriction = true;
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);
    EXPECT_EQ(
        nn::pctl::Feature::Feature_LaunchApplicationByRating | nn::pctl::Feature::Feature_SnsPost | nn::pctl::Feature::Feature_FreeCommunication,
        nn::pctl::GetRestrictedFeatures()
        );

    settingsCustom.ratingAge = 0;
    settingsCustom.freeCommunicationRestriction = true;
    settingsCustom.snsPostRestriction = true;
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);
    EXPECT_EQ(
        nn::pctl::Feature::Feature_SnsPost | nn::pctl::Feature::Feature_FreeCommunication,
        nn::pctl::GetRestrictedFeatures()
        );

    settingsCustom.ratingAge = 0;
    settingsCustom.freeCommunicationRestriction = false;
    settingsCustom.snsPostRestriction = true;
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);
    EXPECT_EQ(
        nn::pctl::Feature::Feature_SnsPost,
        nn::pctl::GetRestrictedFeatures()
        );

    settingsCustom.ratingAge = 0;
    settingsCustom.freeCommunicationRestriction = false;
    settingsCustom.snsPostRestriction = false;
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);
    EXPECT_EQ(
        nn::pctl::Feature::Feature_None,
        nn::pctl::GetRestrictedFeatures()
        );

    settingsCustom.ratingAge = 15;
    settingsCustom.freeCommunicationRestriction = true;
    settingsCustom.snsPostRestriction = false;
    nn::pctl::SetCustomSafetyLevelSettings(settingsCustom);
    EXPECT_EQ(
        nn::pctl::Feature::Feature_LaunchApplicationByRating | nn::pctl::Feature::Feature_FreeCommunication,
        nn::pctl::GetRestrictedFeatures()
        );

    // 解除コード未設定状態の場合は制限なしになる
    nn::pctl::SetPinCode(nullptr);
    EXPECT_EQ(
        nn::pctl::Feature::Feature_None,
        nn::pctl::GetRestrictedFeatures()
        );
}

// DefaultOrganizationの設定テスト
TEST_F(SettingsTest, TestSetDefaultOrganization)
{
    nn::pctl::SetPinCode("0123");

    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::ESRB);
    EXPECT_EQ(nn::ns::RatingOrganization::ESRB, nn::pctl::GetDefaultRatingOrganization());

    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    EXPECT_EQ(nn::ns::RatingOrganization::CERO, nn::pctl::GetDefaultRatingOrganization());
}

// DefaultOrganizationの設定テスト2(解除コード未設定の場合)
TEST_F(SettingsTest, TestSetDefaultOrganization2)
{
    EXPECT_FALSE(nn::pctl::IsRestrictionEnabled());

    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::ESRB);
    EXPECT_EQ(nn::ns::RatingOrganization::ESRB, nn::pctl::GetDefaultRatingOrganization());

    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    EXPECT_EQ(nn::ns::RatingOrganization::CERO, nn::pctl::GetDefaultRatingOrganization());
}

// SystemSettingsの出入りテスト
TEST_F(SettingsTest, TestEnterLeaveSystemSettings)
{
    nn::pctl::SetPinCode("0123");

    nn::pctl::RevertRestrictedSystemSettingsEntered();
    EXPECT_FALSE(nn::pctl::IsRestrictedSystemSettingsEntered());
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestricted, nn::pctl::ConfirmSystemSettingsPermission());

    nn::pctl::EnterRestrictedSystemSettings();
    EXPECT_TRUE(nn::pctl::IsRestrictedSystemSettingsEntered());

    // Enter状態でも影響を与えない
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestricted, nn::pctl::ConfirmSystemSettingsPermission());

    nn::pctl::EnterRestrictedSystemSettings();
    EXPECT_TRUE(nn::pctl::IsRestrictedSystemSettingsEntered());

    nn::pctl::LeaveRestrictedSystemSettings();
    EXPECT_TRUE(nn::pctl::IsRestrictedSystemSettingsEntered());

    nn::pctl::LeaveRestrictedSystemSettings();
    EXPECT_FALSE(nn::pctl::IsRestrictedSystemSettingsEntered());

    nn::pctl::LeaveRestrictedSystemSettings();
    EXPECT_FALSE(nn::pctl::IsRestrictedSystemSettingsEntered());

    // Revertでリセットされる
    nn::pctl::EnterRestrictedSystemSettings();
    nn::pctl::EnterRestrictedSystemSettings();
    nn::pctl::RevertRestrictedSystemSettingsEntered();
    EXPECT_FALSE(nn::pctl::IsRestrictedSystemSettingsEntered());
}

// DeleteSettingsの設定テスト
TEST_F(SettingsTest, TestDeleteSettings)
{
    nn::pctl::DeleteSettings();

    auto defaultRatingOrganization = nn::pctl::GetDefaultRatingOrganization();

    nn::pctl::SetPinCode("0123");
    EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());

    if (defaultRatingOrganization == nn::ns::RatingOrganization::ESRB)
    {
        nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
        EXPECT_EQ(nn::ns::RatingOrganization::CERO, nn::pctl::GetDefaultRatingOrganization());
    }
    else
    {
        nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::ESRB);
        EXPECT_EQ(nn::ns::RatingOrganization::ESRB, nn::pctl::GetDefaultRatingOrganization());
    }

    // 設定削除で状態もリセットされることを確認するための処理
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    nn::pctl::EnterRestrictedSystemSettings();

    nn::pctl::DeleteSettings();

    EXPECT_FALSE(nn::pctl::IsRestrictionEnabled());
    EXPECT_FALSE(nn::pctl::IsRestrictedSystemSettingsEntered());
    EXPECT_FALSE(nn::pctl::IsRestrictionTemporaryUnlocked());
    // 設定削除で既定値に戻る
    EXPECT_EQ(defaultRatingOrganization, nn::pctl::GetDefaultRatingOrganization());
    // 設定削除で初期値(= 制限なし)に戻る
    EXPECT_EQ(nn::pctl::SafetyLevel::SafetyLevel_None, nn::pctl::GetSafetyLevel());
    // カスタム設定もリセットされる
    nn::pctl::SafetyLevelSettings customSettings;
    std::memset(&customSettings, 0xCC, sizeof(customSettings));
    nn::pctl::GetSafetyLevelSettings(&customSettings, nn::pctl::SafetyLevel::SafetyLevel_Custom);
    EXPECT_EQ(customSettings.freeCommunicationRestriction, false);
    EXPECT_EQ(customSettings.ratingAge, 0);
    EXPECT_EQ(customSettings.snsPostRestriction, false);
}
