﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>

#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiPairing.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/pctl_ApiForDebug.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nsd/nsd_ApiForTest.h>
#else
#include <nn/nsd/nsd_ApiForMenu.h>
#endif

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/os.h>

#include "../Common/testPctl_TestUtil.h"


namespace
{
    nn::ncm::ApplicationId g_ApplicationId = { 0x0100008003486000 };
}

class RequestUpdateExemptionListTest : public ::testing::Test
{
public:
    static nn::pctl::test::PairingInfo PairingCodeInfo;
    static bool PairingEnabled;
    static bool UnlinkPairingEnabled;

protected:
    static void SetUpTestCase()
    {
        nn::nifm::Initialize();
        pNetworkConnection = new nn::nifm::NetworkConnection();
        pNetworkConnection->SubmitRequestAndWait();
    }
    static void TearDownTestCase()
    {
        delete pNetworkConnection;
        pNetworkConnection = nullptr;
    }
    static nn::nifm::NetworkConnection* pNetworkConnection;
};

nn::pctl::test::PairingInfo RequestUpdateExemptionListTest::PairingCodeInfo = {};
bool RequestUpdateExemptionListTest::PairingEnabled = true;
bool RequestUpdateExemptionListTest::UnlinkPairingEnabled = true;
nn::nifm::NetworkConnection* RequestUpdateExemptionListTest::pNetworkConnection = nullptr;

TEST_F(RequestUpdateExemptionListTest, BeforePairing)
{
    if (PairingEnabled)
    {
        // 未連携状態ではエラー
        NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPairingNotActive, nn::pctl::RequestUpdateExemptionList(g_ApplicationId));
    }
}

TEST_F(RequestUpdateExemptionListTest, Pairing)
{
    if (PairingEnabled)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::nsd::SetTd1EnvironmentForTest();
#else
        nn::nsd::EnvironmentIdentifier env;
        nn::nsd::GetEnvironmentIdentifier(&env);
        ASSERT_EQ(0, strncmp("td1", env.value, sizeof("td1")));
#endif

        if (PairingCodeInfo.pairingCode[0] == '\0')
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::pctl::test::GetPairingInfo(&PairingCodeInfo, "ido_junichi+pctl01@exmx.nintendo.co.jp", "Password"));
        }

        nn::pctl::SetPinCode(nullptr);
        nn::pctl::ClearFreeCommunicationApplicationListForDebug();
        nn::pctl::DeletePairing();

        // 未連携状態では制限対象外リストは存在しない
        ASSERT_EQ(0, nn::pctl::GetExemptApplicationListCountForDebug());

        // 未連携状態ではエラー
        NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPairingNotActive, nn::pctl::RequestUpdateExemptionList(g_ApplicationId));

        nn::pctl::PairingInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::RequestPairing(&info, PairingCodeInfo.pairingCode));

        nn::pctl::PairingAccountInfo accountInfo;
        char nickname[64];
        size_t size = 0;

        ASSERT_TRUE(info.IsInstanceValid());
        ASSERT_EQ(nn::pctl::PairingState::PairingState_Processing, info.GetState());
        ASSERT_TRUE(info.GetAccountInfo(&accountInfo));
        ASSERT_TRUE(accountInfo.IsInstanceValid());
        EXPECT_GT(accountInfo.GetNickname(nickname, 64), static_cast<size_t>(0));
        NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImageContentType(&size, nullptr, 0));
        NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImage(&size, nullptr, 0));

        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::AuthorizePairing(&info));

        ASSERT_TRUE(info.IsInstanceValid());
        ASSERT_EQ(nn::pctl::PairingState::PairingState_Active, info.GetState());
    }
}

TEST_F(RequestUpdateExemptionListTest, RequestUpdateExemptionList)
{
    nn::pctl::SafetyLevelSettings customSettings;
    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    nn::ns::ApplicationControlProperty prop;
    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);

    // 本体内のリストをクリア
    nn::pctl::ClearExemptApplicationListForDebug();

    // 年齢制限により起動・再開・動画再生不可
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmLaunchApplicationPermission(g_ApplicationId, prop));
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmResumeApplicationPermission(g_ApplicationId, prop));
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPlayingApplicationVideoRestricted, nn::pctl::ConfirmPlayableApplicationVideo(g_ApplicationId, prop.ratingAge));

    // リスト登録のキャンセルも可能
    NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];
    ASSERT_EQ(0, reinterpret_cast<uintptr_t>(&stackOfThread) % nn::os::ThreadStackAlignment);

    nn::os::ThreadType UpdateExemptionListThread;
    nn::Result result;
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&UpdateExemptionListThread,
        [](void* p)
        {
            *static_cast<nn::Result*>(p) = nn::pctl::RequestUpdateExemptionList(g_ApplicationId);
        },
        &result, &stackOfThread, 4 * 1024, 24));
    nn::os::StartThread(&UpdateExemptionListThread);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    nn::pctl::CancelNetworkRequest(); // キャンセル
    nn::os::WaitThread(&UpdateExemptionListThread);

    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultCanceled, result);

    // キャンセルで登録失敗なので起動・再開・動画再生不可
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmLaunchApplicationPermission(g_ApplicationId, prop));
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmResumeApplicationPermission(g_ApplicationId, prop));
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPlayingApplicationVideoRestricted, nn::pctl::ConfirmPlayableApplicationVideo(g_ApplicationId, prop.ratingAge));

    // 更新のキャンセル直後に再更新をリクエストすると，サーバ側の処理順序が入れ替わる可能性がある（MOON-2152）のでウエイト
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    // リスト登録成功後は起動・再開・動画再生可能
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::RequestUpdateExemptionList(g_ApplicationId));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(g_ApplicationId, prop));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(g_ApplicationId, prop));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmPlayableApplicationVideo(g_ApplicationId, prop.ratingAge));

    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::SynchronizeParentalControlSettings());
}

TEST_F(RequestUpdateExemptionListTest, UnlinkPairing)
{
    if (UnlinkPairingEnabled)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlinkPairing(true));

        // 未連携状態では制限対象外リストは存在しない
        EXPECT_EQ(0, nn::pctl::GetExemptApplicationListCountForDebug());
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    if (argc > 1 && (std::strcmp(argv[1], "--help") == 0 || std::strcmp(argv[1], "-h") == 0))
    {
        NN_LOG("Usage:\n" " testPctl_RequestUpdateExemptionList [pairing code] [--without_pairing] [--without_unlinkpairing]\n");
        return;
    }

    ::testing::InitGoogleTest(&argc, argv);

    for (int i = 1; i < argc; ++i)
    {
        if (std::strcmp(argv[i], "--without_pairing") == 0) // Pairing を行わない
        {
            RequestUpdateExemptionListTest::PairingEnabled = false;
        }
        else if (std::strcmp(argv[i], "--without_unlinkpairing") == 0) // UnlinkPairing を行わない
        {
            RequestUpdateExemptionListTest::UnlinkPairingEnabled = false;
        }
        else
        {
            nn::util::Strlcpy(RequestUpdateExemptionListTest::PairingCodeInfo.pairingCode, argv[i], sizeof(RequestUpdateExemptionListTest::PairingCodeInfo.pairingCode));
        }
    }

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
