﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/ovln/ovln_ReceiverForOverlay.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForTest.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/detail/service/overlay/pctl_OverlaySender.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherEventManager.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_PosixTime.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_StandardNetworkSystemClockPrivilegeApi.h>

// nn::os::SleepThread を伴うテストを有効化するマクロ
// (時間がかかるためマクロで制御、以下のコメントを除去で有効化可能)
//#define NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD 1

// 一定の誤差を許容するチェック
#define NNT_PCTL_EXPECT_ALMOST_EQUAL(expect, range, actual) \
    do { \
        auto _a = (actual); \
        EXPECT_LE(_a, (expect) + (range)); \
        EXPECT_GE(_a, (expect) - (range)); \
    } while (NN_STATIC_CONDITION(0))

// 許容する誤差(ミリ秒単位)
#if defined(NN_BUILD_CONFIG_OS_WIN)
// Windows環境では本来別プロセスになる処理がすべて同一プロセスに載り
// 重くなりやすいので誤差を大きく取る
static const int64_t IgnorableTimeError = 8000; // 7-8秒
#else
static const int64_t IgnorableTimeError = 4000; // 3-4秒
#endif
#if defined(NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD)
// WatcherEventManager::TimeSpanForCheck の値(秒単位)
static const int64_t TimeSpanForCheck = nn::pctl::detail::service::watcher::WatcherEventManager::TimeSpanForCheck;
// 通知待ちを行う最低の時間(秒単位、事前に進ませる時刻の差分にもこの値を反映させる)
static const int64_t WaitTimeSecondsBeforeNotification = 2;
#endif

namespace
{
    static nn::time::LocationName g_PreviousDeviceLocationName;
    static nn::time::SystemClockContext g_PreviousNetworkSystemClockContext;
    static bool g_IsClockAutoCorrectEnabled;
    static nn::time::LocationName g_UTCLocationName;
    static bool g_IsUTCLocationNameAvailable = false;
    static nn::time::LocationName g_AnotherLocationName; // 適当な地域名設定(何でもよい)
    static bool g_IsAnotherLocationNameAvailable = false;

    static bool g_IsPlayTimerRunning = false;

    void ClearPlayTimerSettings() NN_NOEXCEPT
    {
        nn::pctl::PlayTimerSettings set = {};
        set.isEnabled = false;
        set.isWeekSettingsUsed = false;
        set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
        set.dailySettings.isBedtimeEnabled = false;
        set.dailySettings.isLimitTimeEnabled = false;
        nn::pctl::SetPlayTimerSettingsForDebug(set);
    }

    void RefreshPlayTimerRemainingTime() NN_NOEXCEPT
    {
        nn::pctl::StartPlayTimer();
        nn::pctl::StopPlayTimer();
    }

    static void SetNetworkTimeFromCalendarForDebug(const nn::time::CalendarTime& calTime) NN_NOEXCEPT
    {
        nn::time::PosixTime posixTime;
        int c = 0;

        nn::time::ToPosixTime(&c, &posixTime, 1, calTime);
        // 1未満になるのは想定外なのでエラー扱いにする
        EXPECT_GE(c, 1);

        NNT_EXPECT_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(posixTime));
        //nn::time::PosixTime currentTime;
        //NNT_EXPECT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetCurrentTime(&currentTime));
        //EXPECT_EQ(posixTime, currentTime);
    }

    static nn::time::PosixTime g_DebugNetworkTime = { 0 };

    void SetInitialNetworkTimeForDebug(uint8_t hour, uint8_t minute) NN_NOEXCEPT
    {
        nn::time::CalendarTime calTime;
        calTime.year = 2016;
        calTime.month = 5;
        calTime.day = 2;
        calTime.hour = hour;
        calTime.minute = minute;
        calTime.second = 0;
        SetNetworkTimeFromCalendarForDebug(calTime);

        nn::time::PosixTime posixTime;
        int c = 0;

        nn::time::ToPosixTime(&c, &posixTime, 1, calTime);
        // 1未満になるのは想定外なのでエラー扱いにする
        EXPECT_GE(c, 1);
        g_DebugNetworkTime = posixTime;
        NNT_EXPECT_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(posixTime));
    }

    void SetInitialNetworkTimeForDebug() NN_NOEXCEPT
    {
        nn::time::CalendarTime calTime;
        calTime.year = 2014;
        calTime.month = 7;
        calTime.day = 7;
        calTime.hour = 7;
        calTime.minute = 7;
        calTime.second = 7;
        SetNetworkTimeFromCalendarForDebug(calTime);
    }

    void AddNetworkTimeForDebug(int64_t offsetSeconds) NN_NOEXCEPT
    {
        g_DebugNetworkTime.value += offsetSeconds;
        NNT_EXPECT_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(g_DebugNetworkTime));
    }

    void GetCalendarTimeForDebug(nn::time::CalendarTime& calTime) NN_NOEXCEPT
    {
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, g_DebugNetworkTime);
        NN_UNUSED(calInfo);
    }

    void ResetPlayTimerAndSetNetworkTime(uint8_t hour, uint8_t minute) NN_NOEXCEPT
    {
        SetInitialNetworkTimeForDebug();
        ClearPlayTimerSettings();
        RefreshPlayTimerRemainingTime();
        SetInitialNetworkTimeForDebug(hour, minute);
        NNT_PCTL_EXPECT_ALMOST_EQUAL(0, IgnorableTimeError, nn::pctl::GetPlayTimerRemainingTime().GetMilliSeconds());
    }

    void ResetPlayTimerAndSetNetworkTime() NN_NOEXCEPT
    {
        ResetPlayTimerAndSetNetworkTime(20, 25);
    }

    void TestTimerSettings(bool expectIsEnabledValue) NN_NOEXCEPT
    {
        nn::pctl::PlayTimerSettings test = {};
        test.isEnabled = !expectIsEnabledValue; // isEnabled が一致するか確認するため逆の値をセット
        nn::pctl::GetPlayTimerSettings(&test);
        EXPECT_EQ(expectIsEnabledValue, test.isEnabled);
    }

    void TestTimerSettings(bool expectIsEnabledValue, nn::pctl::PlayTimerMode expectPlayTimerModeValue) NN_NOEXCEPT
    {
        nn::pctl::PlayTimerSettings test = {};
        // 一致するか確認するため逆の値をセット
        test.isEnabled = !expectIsEnabledValue;
        test.playTimerMode = (expectPlayTimerModeValue == nn::pctl::PlayTimerMode::PlayTimerMode_Alarm) ?
            nn::pctl::PlayTimerMode::PlayTimerMode_Suspend :
            nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
        nn::pctl::GetPlayTimerSettings(&test);
        EXPECT_EQ(expectIsEnabledValue, test.isEnabled);
        EXPECT_EQ(expectPlayTimerModeValue, test.playTimerMode);
    }
}

#if defined(NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD)
// Win版でオーバーレイ通知のテストを行うために使用する関数などの宣言
// (Win版ではovlnを使用せずに独自のダミー実装を使っている)
#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace nn { namespace pctl { namespace detail { namespace service { namespace overlay {

// pctl_OverlaySender-os.win32.cpp と同じ定義を行う
struct NotifiedDataSettingChange
{
    nn::os::EventType event;
    nn::ovln::format::PctlSettingTypeFlagSet settingType;
    bool isNotified;
};
struct NotifiedDataRemainingTime
{
    nn::os::EventType event;
    int32_t time;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag;
    bool isNotified;
};
struct NotifiedDataTimeExceededOnStart
{
    nn::os::EventType event;
    int64_t time;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag;
    bool isNotified;
};

NotifiedDataSettingChange* GetNotifiedDataSettingChange() NN_NOEXCEPT;
NotifiedDataRemainingTime* GetNotifiedDataRemainingTime() NN_NOEXCEPT;
NotifiedDataTimeExceededOnStart* GetNotifiedDataTimeExceededOnStart() NN_NOEXCEPT;

}}}}}
#endif

// オーバーレイ通知テストの補助処理
namespace
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    nn::ovln::ReceiverForOverlayType g_OverlayReceiver;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForOverlay;
    nn::os::TimerEventType g_TempTimerEvent;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForTempTimer;
    nn::os::MultiWaitType g_MultiWaitForOverlay;
#endif

    void InitializeForOverlayTest() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        // 特に何もしない
#else
        // オーバーレイ通知関連の初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeReceiver(&g_OverlayReceiver));
        nn::os::InitializeMultiWait(&g_MultiWaitForOverlay);
        nn::ovln::InitializeMultiWaitHolder(&g_MultiWaitHolderForOverlay, &g_OverlayReceiver);
        nn::os::LinkMultiWaitHolder(&g_MultiWaitForOverlay, &g_MultiWaitHolderForOverlay);

        nn::os::InitializeTimerEvent(&g_TempTimerEvent, nn::os::EventClearMode::EventClearMode_AutoClear);
        nn::os::ClearTimerEvent(&g_TempTimerEvent);
        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForTempTimer, &g_TempTimerEvent);
        nn::os::LinkMultiWaitHolder(&g_MultiWaitForOverlay, &g_MultiWaitHolderForTempTimer);
#endif
    }

    void FinalizeForOverlayTest() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        // 特に何もしない
#else
        nn::os::UnlinkAllMultiWaitHolder(&g_MultiWaitForOverlay);
        nn::os::FinalizeMultiWait(&g_MultiWaitForOverlay);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForTempTimer);
        nn::os::FinalizeTimerEvent(&g_TempTimerEvent);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForOverlay);
        nn::ovln::FinalizeReceiver(&g_OverlayReceiver);
        nn::ovln::FinalizeReceicerLibraryForOverlay();
#endif
    }

    // 適切にイベント待ちできるようにリセットを行う
    void ResetForWaitingOverlayReceiveRemainingTime() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        {
            auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
            nn::os::ClearEvent(&pNotifiedData->event);
            pNotifiedData->isNotified = false;
        }
        {
            auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataTimeExceededOnStart();
            nn::os::ClearEvent(&pNotifiedData->event);
            pNotifiedData->isNotified = false;
        }
#else
        // 蓄えられたメッセージを消化する
        nn::ovln::Message msg;
        while (nn::ovln::TryReceive(&msg, &g_OverlayReceiver))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
#endif
    }

    // 指定時間内に通知が来た場合、その内容を返して true を返す
    // (通知が来なかった場合は false を返す)
    bool WaitForOverlayReceiveRemainingTime(int32_t* outTime, nn::ovln::format::PctlPlayTimerModeFlag* outMode,
        nn::TimeSpan timeToWait) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
        if (!nn::os::TimedWaitEvent(&pNotifiedData->event, timeToWait))
        {
            return false;
        }
        if (!pNotifiedData->isNotified)
        {
            return false;
        }
        if (outTime != nullptr)
        {
            *outTime = pNotifiedData->time;
        }
        if (outMode != nullptr)
        {
            *outMode = pNotifiedData->modeFlag;
        }
        return true;
#else
        // TimedWaitAny ではなく TimerEvent で時間経過を監視する
        // (別の通知イベントを受け取った場合は無視しつつ、それまで経過した時間を
        // リセットしないようにする)
        nn::os::StartOneShotTimerEvent(&g_TempTimerEvent, timeToWait);
        while (NN_STATIC_CONDITION(true))
        {
            auto pHolder = nn::os::WaitAny(&g_MultiWaitForOverlay);
            if (pHolder == &g_MultiWaitHolderForTempTimer)
            {
                nn::os::StopTimerEvent(&g_TempTimerEvent);
                nn::os::ClearTimerEvent(&g_TempTimerEvent);
                return false;
            }
            EXPECT_EQ(&g_MultiWaitHolderForOverlay, pHolder);

            nn::ovln::Message msg;
            nn::ovln::Receive(&msg, &g_OverlayReceiver);
            if (msg.tag == nn::ovln::format::PctlRemainingTimeMessage)
            {
                auto pData = reinterpret_cast<nn::ovln::format::PctlRemainingTimeMessageData*>(&msg.data);
                if (outTime != nullptr)
                {
                    *outTime = static_cast<int32_t>(pData->remainingTime);
                }
                if (outMode != nullptr)
                {
                    *outMode = pData->playTimerMode;
                }
                break;
            }
        }
        nn::os::StopTimerEvent(&g_TempTimerEvent);
        nn::os::ClearTimerEvent(&g_TempTimerEvent);
        return true;
#endif
    }

    // 指定時間内に通知が来た場合、その内容を返して true を返す
    // (通知が来なかった場合は false を返す)
    bool WaitForOverlayReceiveExceeded(int64_t* outTime, nn::ovln::format::PctlPlayTimerModeFlag* outMode,
        nn::TimeSpan timeToWait) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataTimeExceededOnStart();
        if (!nn::os::TimedWaitEvent(&pNotifiedData->event, timeToWait))
        {
            return false;
        }
        if (!pNotifiedData->isNotified)
        {
            return false;
        }
        if (outTime != nullptr)
        {
            *outTime = pNotifiedData->time;
        }
        if (outMode != nullptr)
        {
            *outMode = pNotifiedData->modeFlag;
        }
        return true;
#else
        // TimedWaitAny ではなく TimerEvent で時間経過を監視する
        // (別の通知イベントを受け取った場合は無視しつつ、それまで経過した時間を
        // リセットしないようにする)
        nn::os::StartOneShotTimerEvent(&g_TempTimerEvent, timeToWait);
        while (NN_STATIC_CONDITION(true))
        {
            auto pHolder = nn::os::WaitAny(&g_MultiWaitForOverlay);
            if (pHolder == &g_MultiWaitHolderForTempTimer)
            {
                nn::os::StopTimerEvent(&g_TempTimerEvent);
                nn::os::ClearTimerEvent(&g_TempTimerEvent);
                return false;
            }
            EXPECT_EQ(&g_MultiWaitHolderForOverlay, pHolder);

            nn::ovln::Message msg;
            nn::ovln::Receive(&msg, &g_OverlayReceiver);
            if (msg.tag == nn::ovln::format::PctlTimeExceededOnStartMessage)
            {
                auto pData = reinterpret_cast<nn::ovln::format::PctlTimeExceededOnStartMessageData*>(&msg.data);
                if (outTime != nullptr)
                {
                    *outTime = pData->remainingTime;
                }
                if (outMode != nullptr)
                {
                    *outMode = pData->playTimerMode;
                }
                break;
            }
        }
        nn::os::StopTimerEvent(&g_TempTimerEvent);
        nn::os::ClearTimerEvent(&g_TempTimerEvent);
        return true;
#endif
    }
}
#endif // defined(NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD)

class PlayTimerClientTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
#if defined(NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD)
        InitializeForOverlayTest();
#endif

        // タイムゾーン地域名と時刻の自動補正を扱うため ForMenu で初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());

        nn::time::GetDeviceLocationName(&g_PreviousDeviceLocationName);
        g_IsClockAutoCorrectEnabled = nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled();
        nn::time::StandardNetworkSystemClock::GetSystemClockContext(&g_PreviousNetworkSystemClockContext);

        // UTCタイムゾーンへの変更処理
        nn::time::LocationName* pLocationNames;
        int totalCount = nn::time::GetTotalLocationNameCount();
        if (totalCount == 0)
        {
            NN_LOG("Warning: No location name for time-zone is available. Test may not work.\n");
        }
        else
        {
            pLocationNames = static_cast<nn::time::LocationName*>(malloc(sizeof(nn::time::LocationName) * totalCount));
            NN_ABORT_UNLESS_NOT_NULL(pLocationNames);
            nn::time::LoadLocationNameList(&totalCount, pLocationNames, totalCount, 0);
            if (totalCount == 0)
            {
                NN_LOG("Warning: No location name for time-zone is available. Test may not work.\n");
            }
            else
            {
                bool found = false;
                for (int i = 0; i < totalCount; ++i)
                {
                    if (nn::util::Strnicmp(pLocationNames[i]._value, "UTC", nn::time::LocationName::Size) == 0)
                    {
                        // UTCに書き換える
                        found = true;
                        nn::time::SetDeviceLocationName(pLocationNames[i]);
                        std::memcpy(&g_UTCLocationName, &pLocationNames[i], sizeof(nn::time::LocationName));
                        g_IsUTCLocationNameAvailable = true;
                        break;
                    }
                    else if (!g_IsAnotherLocationNameAvailable)
                    {
                        // UTC以外なら何でもよいので地域名設定を得る
                        std::memcpy(&g_AnotherLocationName, &pLocationNames[i], sizeof(nn::time::LocationName));
                        g_IsAnotherLocationNameAvailable = true;
                    }
                }
                if (!found)
                {
                    // UTCが見つからないので変更しない
                    NN_LOG("Warning: UTC location name is not found. Test may not work properly.\n");
                }
            }
            free(pLocationNames);
        }

        nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(true);

        // 時刻変更を行うため ForSystem で再初期化(以降の既定とする)
        nn::time::Finalize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForSystem());
        SetInitialNetworkTimeForDebug();

        g_IsPlayTimerRunning = nn::pctl::IsPlayTimerEnabled();
        nn::pctl::StopPlayTimer();

        nn::pctl::SetPinCode("0000");
        ClearPlayTimerSettings();
    }

    static void TearDownTestCase()
    {
        ClearPlayTimerSettings();
        nn::pctl::SetPinCode(nullptr);
        if (g_IsPlayTimerRunning)
        {
            nn::pctl::StartPlayTimer();
        }

        // コンテキストを元に戻す
        nn::time::SetStandardNetworkSystemClockContext(g_PreviousNetworkSystemClockContext);
        nn::time::Finalize();
        nn::time::InitializeForMenu();
        // 地域名設定を元に戻す
        nn::time::SetDeviceLocationName(g_PreviousDeviceLocationName);
        // を元に戻す
        nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(g_IsClockAutoCorrectEnabled);

        nn::time::Finalize();

#if defined(NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD)
        FinalizeForOverlayTest();
#endif
    }
};


TEST_F(PlayTimerClientTest, SettingDisabled)
{
    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = false;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = false;
    nn::pctl::SetPlayTimerSettingsForDebug(set);

    TestTimerSettings(false);
}

TEST_F(PlayTimerClientTest, TimerIgnoreSetEnabled)
{
    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = false;
    nn::pctl::SetPlayTimerSettingsForDebug(set);

    TestTimerSettings(false); // 曜日別設定が何もなければ変わらない
}

TEST_F(PlayTimerClientTest, AlarmTimer_Everyday)
{
    ClearPlayTimerSettings();
    TestTimerSettings(false);

    nn::pctl::StartPlayTimer();
    nn::pctl::StopPlayTimer();
    auto prevRemainingTime = nn::pctl::GetPlayTimerRemainingTime();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = false; // true に書き換わるはず
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    nn::pctl::SetPlayTimerSettingsForDebug(set);

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);

    auto remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    int64_t milliSec = (remainingTime - prevRemainingTime).GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(60 * 60 * 1000, IgnorableTimeError, milliSec);
}

TEST_F(PlayTimerClientTest, AlarmTimer_Someday)
{
    ClearPlayTimerSettings();
    TestTimerSettings(false);

    auto prevRemainingTime = nn::pctl::GetPlayTimerRemainingTime();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = true;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    // 今の曜日を用いる
    nn::pctl::Week week;
    {
        nn::time::PosixTime time;
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&time).IsFailure())
        {
            time.value = 0;
        }
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        week = static_cast<nn::pctl::Week>(calInfo.dayOfWeek);
        NN_UNUSED(calTime);
    }
    set.weekSettings[week].isBedtimeEnabled = false;
    set.weekSettings[week].isLimitTimeEnabled = true;
    set.weekSettings[week].limitTime = 80;

    nn::pctl::SetPlayTimerSettingsForDebug(set);

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);

    auto remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    int64_t milliSec = (remainingTime - prevRemainingTime).GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(80 * 60 * 1000, IgnorableTimeError, milliSec);
}

TEST_F(PlayTimerClientTest, AlarmTimer_Bedtime)
{
    nn::pctl::PlayTimerSettings set = {};
    ClearPlayTimerSettings();
    TestTimerSettings(false);

    auto prevRemainingTime = nn::pctl::GetPlayTimerRemainingTime();

    static const uint8_t SetData_Hour = 22;
    static const uint8_t SetData_Minute = 30;
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    nn::time::CalendarTime calTime;
    {
        nn::time::PosixTime time;
        nn::time::CalendarAdditionalInfo calInfo;
        if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&time).IsFailure())
        {
            time.value = 0;
        }
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        NN_UNUSED(calInfo);
    }

    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = SetData_Hour;
    set.dailySettings.bedtimeMinute = SetData_Minute;

    nn::pctl::SetPlayTimerSettingsForDebug(set);

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);

    auto remainingTime = nn::pctl::GetPlayTimerRemainingTime();

    if (calTime.hour < 6)
    {
        int64_t exceedTime = (calTime.hour + 24 - SetData_Hour) * 60 * 60 +
            (calTime.minute - SetData_Minute) * 60 + calTime.second;
        // bedtime 範囲内なので負数になる
        int64_t milliSec = (remainingTime - prevRemainingTime).GetMilliSeconds();
        NNT_PCTL_EXPECT_ALMOST_EQUAL(-exceedTime * 1000, IgnorableTimeError, milliSec);
    }
    else if (calTime.hour > SetData_Hour || (calTime.hour == SetData_Hour && calTime.minute >= SetData_Minute))
    {
        int64_t exceedTime = (calTime.hour - SetData_Hour) * 60 * 60 +
            (calTime.minute - SetData_Minute) * 60 + calTime.second;
        // bedtime 範囲内なので負数になる
        int64_t milliSec = (remainingTime - prevRemainingTime).GetMilliSeconds();
        NNT_PCTL_EXPECT_ALMOST_EQUAL(-exceedTime * 1000, IgnorableTimeError, milliSec);
    }
    else
    {
        int64_t remainTime = (SetData_Hour - calTime.hour) * 60 * 60 +
            (SetData_Minute - calTime.minute) * 60 - calTime.second;
        // bedtime 範囲外なので正数になる
        int64_t milliSec = (remainingTime - prevRemainingTime).GetMilliSeconds();
        NNT_PCTL_EXPECT_ALMOST_EQUAL(remainTime * 1000, IgnorableTimeError, milliSec);
    }
}

// 日付を跨いだ場合06:00(午前6時)までは超過時間とみなされるかどうか
TEST_F(PlayTimerClientTest, AlarmTimer_ExceedingOnDayChanged)
{
    static const uint8_t StartTimeHour = 21;
    static const uint8_t StartTimeMinute = 0;
    static const uint8_t BedtimeHour = 23;
    static const uint8_t BedtimeMinute = 0;

    ResetPlayTimerAndSetNetworkTime(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = BedtimeHour;
    set.dailySettings.bedtimeMinute = BedtimeMinute;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 90分(1時間半)進める(この時点で22:30)
    AddNetworkTimeForDebug(90 * 60);

    auto remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 90分だけ減っているはず
    int64_t milliSec = remainingTime.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 90) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 60分進める(この時点で23:30)
    AddNetworkTimeForDebug(60 * 60);

    remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 累計150分だけ減って超過時間になるはず
    milliSec = remainingTime.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150 < 0);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 90分進める(この時点で01:00)
    AddNetworkTimeForDebug(90 * 60);

    remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 累計240分だけ減って超過時間になるはず
    milliSec = remainingTime.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240 < 0);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // さらに320分進める(この時点で06:20)
    AddNetworkTimeForDebug(320 * 60);
    {
        nn::time::CalendarTime calTime;
        GetCalendarTimeForDebug(calTime);
        EXPECT_EQ(6, calTime.hour);
        EXPECT_EQ(20, calTime.minute);
    }

    remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 境界線を越えて再び残り時間の扱いになるはず
    milliSec = remainingTime.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - 6) * 60) + (BedtimeMinute - 20)) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    nn::pctl::StopPlayTimer();
}

// 日付を跨いだ次の日にBedtime制限がない場合でも、場合06:00(午前6時)までは超過時間とみなされるかどうか
TEST_F(PlayTimerClientTest, AlarmTimer_ExceedingOnDayChangedToNoRestrictionDay)
{
    static const uint8_t StartTimeHour = 21;
    static const uint8_t StartTimeMinute = 0;
    static const uint8_t BedtimeHour = 23;
    static const uint8_t BedtimeMinute = 0;

    ResetPlayTimerAndSetNetworkTime(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = true;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    // 曜日を得て現在の曜日以外はクリアする
    nn::pctl::Week week;
    {
        nn::time::PosixTime time = g_DebugNetworkTime;
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        week = static_cast<nn::pctl::Week>(calInfo.dayOfWeek);
        NN_UNUSED(calTime);
    }
    memset(set.weekSettings, 0, sizeof(set.weekSettings));
    set.weekSettings[week].isBedtimeEnabled = true;
    set.weekSettings[week].isLimitTimeEnabled = false;
    set.weekSettings[week].bedtimeHour = BedtimeHour;
    set.weekSettings[week].bedtimeMinute = BedtimeMinute;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 90分(1時間半)進める(この時点で22:30)
    AddNetworkTimeForDebug(90 * 60);

    auto remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 90分だけ減っているはず
    int64_t milliSec = remainingTime.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 90) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 60分進める(この時点で23:30)
    AddNetworkTimeForDebug(60 * 60);

    remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 累計150分だけ減って超過時間になるはず
    milliSec = remainingTime.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150 < 0);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 90分進める(この時点で01:00)
    AddNetworkTimeForDebug(90 * 60);

    remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 累計240分だけ減って超過時間になるはず(翌日は制限がないが引き継がれるはず)
    milliSec = remainingTime.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240 < 0);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // さらに320分進める(この時点で06:20)
    AddNetworkTimeForDebug(320 * 60);
    {
        nn::time::CalendarTime calTime;
        GetCalendarTimeForDebug(calTime);
        EXPECT_EQ(6, calTime.hour);
        EXPECT_EQ(20, calTime.minute);
    }


    remainingTime = nn::pctl::GetPlayTimerRemainingTime();
    // 境界線を越えて制限のない状態になるはずなので 0 になるはず
    milliSec = remainingTime.GetMilliSeconds();
    EXPECT_EQ(0, milliSec);
    // さらに無効状態になるはず
    // (値を変えずに設定変更して即時に更新させる)
    nn::pctl::GetPlayTimerSettings(&set);
    set.isEnabled = true;
    nn::pctl::SetPlayTimerSettingsForDebug(set);
    TestTimerSettings(false);

    nn::pctl::StopPlayTimer();
}

// 日付を跨いだ次の日にBedtime制限がない場合でも、場合06:00(午前6時)までは超過時間とみなされるかどうか
TEST_F(PlayTimerClientTest, AlarmTimer_NotRestrictedOnDayChangedToRestrictedDay)
{
    static const uint8_t StartTimeHour = 21;
    static const uint8_t StartTimeMinute = 0;
    static const uint8_t BedtimeHour = 23;
    static const uint8_t BedtimeMinute = 0;

    ResetPlayTimerAndSetNetworkTime(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = true;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    // 曜日を得て現在の曜日の次以外はクリアする
    nn::pctl::Week week;
    {
        nn::time::PosixTime time = g_DebugNetworkTime;
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        week = static_cast<nn::pctl::Week>(static_cast<int>(calInfo.dayOfWeek) + 1);
        if (week == nn::pctl::Week::Week_TotalCount)
        {
            week = nn::pctl::Week::Week_Sunday;
        }
        NN_UNUSED(calTime);
    }
    memset(set.weekSettings, 0, sizeof(set.weekSettings));
    set.weekSettings[week].isBedtimeEnabled = true;
    set.weekSettings[week].isLimitTimeEnabled = false;
    set.weekSettings[week].bedtimeHour = BedtimeHour;
    set.weekSettings[week].bedtimeMinute = BedtimeMinute;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    // この時点ではまだ制限がない
    TestTimerSettings(false, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 90分(1時間半)進める(この時点で22:30)
    AddNetworkTimeForDebug(90 * 60);

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // まだ制限がないので 0 のはず
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(0, milliSec);
    TestTimerSettings(false);

    // 150分進める(この時点で01:00)
    AddNetworkTimeForDebug(150 * 60);

    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 日付を跨いで制限のある日になるが、制限なしが引き継がれるはず
    milliSec = tm.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240 < 0);
    EXPECT_EQ(0, milliSec);
    TestTimerSettings(false);

    // さらに320分進める(この時点で06:20)
    AddNetworkTimeForDebug(320 * 60);
    {
        nn::time::CalendarTime calTime;
        GetCalendarTimeForDebug(calTime);
        EXPECT_EQ(6, calTime.hour);
        EXPECT_EQ(20, calTime.minute);
    }

    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 境界線を越えて制限のある状態になるはず
    milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - 6) * 60) + (BedtimeMinute - 20)) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );
    // さらに有効状態になるはず
    // (値を変えずに設定変更して即時に更新させる)
    nn::pctl::GetPlayTimerSettings(&set);
    set.isEnabled = false;
    nn::pctl::SetPlayTimerSettingsForDebug(set);
    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);

    nn::pctl::StopPlayTimer();
}

TEST_F(PlayTimerClientTest, RemainingTime_Normal)
{
    ResetPlayTimerAndSetNetworkTime();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 12秒進める
    AddNetworkTimeForDebug(12);

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 12秒だけ減っているはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60 - 12) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();
}

// 無効の間は残り時間を消費しないテスト1
TEST_F(PlayTimerClientTest, RemainingTime_NoSpendingDuringDisabled1)
{
    ResetPlayTimerAndSetNetworkTime();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 無効化して12秒進める
    nn::pctl::StopPlayTimer();
    AddNetworkTimeForDebug(12);

    // 有効化して時間確認
    nn::pctl::StartPlayTimer();
    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 減っていないはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();
}

// 無効の間は残り時間を消費しないテスト2
TEST_F(PlayTimerClientTest, RemainingTime_NoSpendingDuringDisabled2)
{
    ResetPlayTimerAndSetNetworkTime();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 無効化して15秒待機
    nn::pctl::StopPlayTimer();
    AddNetworkTimeForDebug(15);

    // 有効化して15秒待機
    nn::pctl::StartPlayTimer();
    AddNetworkTimeForDebug(15);

    // 無効化して15秒待機
    nn::pctl::StopPlayTimer();
    AddNetworkTimeForDebug(15);

    // 有効化して時間確認
    nn::pctl::StartPlayTimer();
    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 15秒だけ減っているはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60 - 15) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();
}

// ロック状態(ペアコンの解除で発生)の間は残り時間を消費しないテスト1
TEST_F(PlayTimerClientTest, RemainingTime_NoSpendingDuringLocked1)
{
    ResetPlayTimerAndSetNetworkTime();

    // 一時解除させるために解除コードを設定
    nn::pctl::SetPinCode("0000");

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 一時解除して12秒進める
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0000"));
    AddNetworkTimeForDebug(12);

    // 一時解除を元に戻して時間確認
    nn::pctl::RevertRestrictionTemporaryUnlocked();
    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 減っていないはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();
    nn::pctl::SetPinCode(nullptr);
}

// ロック状態(ペアコンの解除で発生)の間は残り時間を消費しないテスト2
TEST_F(PlayTimerClientTest, RemainingTime_NoSpendingDuringLocked2)
{
    ResetPlayTimerAndSetNetworkTime();

    // 一時解除させるために解除コードを設定
    nn::pctl::SetPinCode("0000");

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 無効化して12秒待機
    nn::pctl::StopPlayTimer();
    AddNetworkTimeForDebug(12); // カウントされない((disabled, temporaryUnlocked) == (true, false))

    // 一時解除して12秒待機
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0000"));
    AddNetworkTimeForDebug(12); // カウントされない((disabled, temporaryUnlocked) == (true, true))

    // 有効化して12秒待機
    nn::pctl::StartPlayTimer();
    AddNetworkTimeForDebug(12); // カウントされない((disabled, temporaryUnlocked) == (false, true))

    // 一時解除を元に戻して12秒待機
    nn::pctl::RevertRestrictionTemporaryUnlocked();
    AddNetworkTimeForDebug(12); // カウントされる((disabled, temporaryUnlocked) == (false, false))

    // 有効化して時間確認
    nn::pctl::StartPlayTimer();
    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 12秒だけ減っているはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60 - 12) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();
    nn::pctl::SetPinCode(nullptr);
}

// 一時解除中にペアコン設定を削除して再設定したら適切に一時解除状態が外れてタイマーが動くことのテスト
TEST_F(PlayTimerClientTest, RemainingTime_ChangedLockStatusOnDeleteSettings)
{
    ResetPlayTimerAndSetNetworkTime();

    // 一時解除させるために解除コードを設定
    nn::pctl::SetPinCode("0000");

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 一時解除して12秒進める
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0000"));
    AddNetworkTimeForDebug(12);

    // 時間確認
    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 減っていないはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();

    // 一旦現在のタイマー設定を得る(バックアップ)
    nn::pctl::GetPlayTimerSettings(&set);
    EXPECT_EQ(true, set.isEnabled);
    // 設定削除を実施(ここで一時解除状態が外れるはず)
    nn::pctl::DeleteSettings();
    // 制限設定とタイマーの再設定(復元)を行う
    nn::pctl::SetPinCode("0000");
    nn::pctl::SetPlayTimerSettingsForDebug(set);
    // 再設定した情報を元にタイマーをスタートさせる
    nn::pctl::StartPlayTimer();

    // 12秒進める
    AddNetworkTimeForDebug(12);

    // 時間確認
    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 減っているはず
    milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60 - 12) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();
}

// 一時解除中にペアコン設定を削除した状態ではペアコン設定が無くなっていることのテスト
TEST_F(PlayTimerClientTest, RemainingTime_ChangedLockStatusOnDeleteSettings2)
{
    ResetPlayTimerAndSetNetworkTime();

    // 一時解除させるために解除コードを設定
    nn::pctl::SetPinCode("0000");

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 一時解除して12秒進める
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0000"));
    AddNetworkTimeForDebug(12);

    // 時間確認
    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 減っていないはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();

    // 一旦現在のタイマー設定を得る(バックアップ)
    nn::pctl::GetPlayTimerSettings(&set);
    EXPECT_EQ(true, set.isEnabled);
    // 設定削除を実施
    nn::pctl::DeleteSettings();

    // タイマーはスタートさせる
    nn::pctl::StartPlayTimer();

    // 12秒進める
    AddNetworkTimeForDebug(12);

    // 制限が無いことの確認
    TestTimerSettings(false);

    // 時間確認
    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 設定が無いので進んでいないはず
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(0, milliSec);

    nn::pctl::StopPlayTimer();
    // 元に戻しておく
    nn::pctl::SetPinCode("0000");
}

// 日付を跨いだら残り時間が変わるテスト
TEST_F(PlayTimerClientTest, RemainingTime_ResetOnDayChanged)
{
    ResetPlayTimerAndSetNetworkTime(21, 30);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 90分進める(この時点で23:00)
    AddNetworkTimeForDebug(90 * 60);

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 90分だけ減って残り30分のはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((30 * 60) * 1000, IgnorableTimeError, milliSec);

    // 40分進める(この時点で23:40)
    AddNetworkTimeForDebug(40 * 60);

    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 合計130分だけ減って10分超過のはず
    milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((-10 * 60) * 1000, IgnorableTimeError, milliSec);

    // 50分進める(この時点で00:30)
    AddNetworkTimeForDebug(50 * 60);

    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 0時でリセットされ、30分だけ減って残り90分のはず
    milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((90 * 60) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();
}

#if defined(NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD)
// 地域名設定が変わったら残り時間がリセットされるテスト
TEST_F(PlayTimerClientTest, RemainingTime_ResetOnLocationNameChanged)
{
    if (!g_IsAnotherLocationNameAvailable)
    {
        NN_LOG("Warning: This test cannot be run because any location name but UTC is not found.\n");
        return;
    }

    ResetPlayTimerAndSetNetworkTime(21, 30);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 90分進める(この時点で23:00)
    AddNetworkTimeForDebug(90 * 60);

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 90分だけ減って残り30分のはず
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((30 * 60) * 1000, IgnorableTimeError, milliSec);

    // 地域名設定を適当に変えて内部の地域名チェック間隔以上の時間を待つ
    // ※ time は ForSystem で初期化済みなので ForMenu に切り替える
    nn::time::Finalize();
    NNT_EXPECT_RESULT_SUCCESS(nn::time::InitializeForMenu());
    nn::time::SetDeviceLocationName(g_AnotherLocationName);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(TimeSpanForCheck + 2));

    // 待機した分時刻が進んでいるのでそれを上書きする
    // (time をまた ForSystem で再初期化する)
    nn::time::Finalize();
    NNT_EXPECT_RESULT_SUCCESS(nn::time::InitializeForSystem());
    AddNetworkTimeForDebug(0);
    tm = nn::pctl::GetPlayTimerRemainingTime();
    // リセットされるので残り時間は120分になるはず
    milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((120 * 60) * 1000, IgnorableTimeError, milliSec);

    nn::pctl::StopPlayTimer();

    // 元に戻しておく
    nn::time::Finalize();
    NNT_EXPECT_RESULT_SUCCESS(nn::time::InitializeForMenu());
    nn::time::SetDeviceLocationName(g_UTCLocationName);
    nn::time::Finalize();
    NNT_EXPECT_RESULT_SUCCESS(nn::time::InitializeForSystem());
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(TimeSpanForCheck + 2));
}
#endif

// 特定時刻が期限となる形(おやすみタイム)で作られる制限時間はタイマーが無効の間も減るテスト
TEST_F(PlayTimerClientTest, RemainingTime_SpendingWithBedtimeDuringDisabled1)
{
    static const uint8_t StartTimeHour = 19;
    static const uint8_t StartTimeMinute = 15;
    static const uint8_t BedtimeHour = 21;
    static const uint8_t BedtimeMinute = 0;

    ResetPlayTimerAndSetNetworkTime(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = BedtimeHour;
    set.dailySettings.bedtimeMinute = BedtimeMinute;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    // 無効にしておく
    nn::pctl::StopPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_FALSE(nn::pctl::IsPlayTimerEnabled());

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 念のため現在の残り時間チェック
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute)) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 45分進める(無効のまま)
    AddNetworkTimeForDebug(45 * 60);

    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 45分だけ減っているはず
    milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 45) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 開始させて時刻を15分進める
    nn::pctl::StartPlayTimer();
    AddNetworkTimeForDebug(15 * 60);

    tm = nn::pctl::GetPlayTimerRemainingTime();
    // 合計60分だけ減っているはず
    milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 60) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    nn::pctl::StopPlayTimer();
}

// タイマーが無効の間の bedtime と limitTime から得られる制限時間が変わるかのテスト
TEST_F(PlayTimerClientTest, RemainingTime_SpendingWithBedtimeDuringDisabled2)
{
    static const uint8_t StartTimeHour = 19;
    static const uint8_t StartTimeMinute = 15;
    static const uint8_t BedtimeHour = 21;
    static const uint8_t BedtimeMinute = 0;
    static const uint16_t LimitTimeMinute = 60;

    ResetPlayTimerAndSetNetworkTime(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.bedtimeHour = BedtimeHour;
    set.dailySettings.bedtimeMinute = BedtimeMinute;
    set.dailySettings.limitTime = LimitTimeMinute;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    // 無効にしておく
    nn::pctl::StopPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_FALSE(nn::pctl::IsPlayTimerEnabled());

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    // 念のため現在の残り時間チェック
    int64_t milliSec = tm.GetMilliSeconds();
    // 最初は limitTime の方が小さいのでそれが得られるはず
    // (60min. < (21:00 - 19:15))
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        (LimitTimeMinute * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 30分進める(無効のまま)
    AddNetworkTimeForDebug(30 * 60);

    tm = nn::pctl::GetPlayTimerRemainingTime();
    milliSec = tm.GetMilliSeconds();
    // まだ変化がないはず
    // (60min. < (21:00 - 19:45))
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        (LimitTimeMinute * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    // 30分進める(無効のまま)
    AddNetworkTimeForDebug(30 * 60);

    tm = nn::pctl::GetPlayTimerRemainingTime();
    milliSec = tm.GetMilliSeconds();
    // 変化があるはず
    // (60min. > (21:00 - 20:15))
    NN_STATIC_ASSERT((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 60) < LimitTimeMinute);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 60) * 60) * 1000,
        IgnorableTimeError,
        milliSec
        );

    nn::pctl::StopPlayTimer();
}

// オーバーレイ通知テストはひとまずWin版でのみ動かすこととする
#if defined(NNT_PCTL_CONFIG_ENABLE_TEST_WITH_SLEEP_THREAD)
// OverlayNotification が来るかどうかのテスト(残り30分)
TEST_F(PlayTimerClientTest, OverlayNotification1)
{
    ResetPlayTimerAndSetNetworkTime();
    ResetForWaitingOverlayReceiveRemainingTime();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 時刻を30分弱進める(残り30分 + αになる)
    AddNetworkTimeForDebug(30 * 60 - WaitTimeSecondsBeforeNotification);

    // 通知を正しく発行できるようにするためにタイマーを停止→開始する
    nn::pctl::StopPlayTimer();
    nn::pctl::StartPlayTimer();

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((30 * 60 + WaitTimeSecondsBeforeNotification) * 1000, IgnorableTimeError, milliSec);

    // 一定時間後に通知が来るはずなのでその時間 + 2秒待つ
    // (残り30分ちょうどになる)
    int32_t time = 0;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag = nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Suspend;
    EXPECT_TRUE(WaitForOverlayReceiveRemainingTime(&time, &modeFlag, nn::TimeSpan::FromSeconds(WaitTimeSecondsBeforeNotification + 2)));
    EXPECT_EQ(30 * 60, time);
    EXPECT_EQ(nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Alarm, modeFlag);

    nn::pctl::StopPlayTimer();
}

// OverlayNotification が来るかどうかのテスト(30分超過)
TEST_F(PlayTimerClientTest, OverlayNotification2)
{
    ResetPlayTimerAndSetNetworkTime();
    ResetForWaitingOverlayReceiveRemainingTime();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300)); // サービス側のイベントを処理させる
    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 時刻を約90分進める(30分 - α超過になる)
    AddNetworkTimeForDebug(90 * 60 - WaitTimeSecondsBeforeNotification * 2);

    // 通知を正しく発行できるようにするためにタイマーを停止→開始する
    nn::pctl::StopPlayTimer();
    // (この時点では通知は来ないはず)
    int32_t time = 0;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag = nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Suspend;
    EXPECT_FALSE(WaitForOverlayReceiveRemainingTime(&time, &modeFlag, nn::TimeSpan(0)));
    nn::pctl::StartPlayTimer();
    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200)); // サービス側のイベントを処理させる

    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL((-30 * 60 + WaitTimeSecondsBeforeNotification * 2) * 1000, IgnorableTimeError, milliSec);

    // まず時間切れの通知が来るはず
    int64_t timeExceeded = -30;
    modeFlag = nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Suspend;
    EXPECT_TRUE(WaitForOverlayReceiveExceeded(&timeExceeded, &modeFlag, nn::TimeSpan::FromSeconds(WaitTimeSecondsBeforeNotification)));
    NNT_PCTL_EXPECT_ALMOST_EQUAL((30 * 60 + WaitTimeSecondsBeforeNotification * 2), IgnorableTimeError, timeExceeded);
    EXPECT_EQ(nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Alarm, modeFlag);
    ResetForWaitingOverlayReceiveRemainingTime();

    // そして一定時間後に通知が来るはずなのでその時間 + 2秒待つ
    // (30分超過になる)
    time = 0;
    modeFlag = nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Suspend;
    EXPECT_TRUE(WaitForOverlayReceiveRemainingTime(&time, &modeFlag, nn::TimeSpan::FromSeconds(WaitTimeSecondsBeforeNotification * 2 + 2)));
    EXPECT_EQ(-30 * 60, time);
    EXPECT_EQ(nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Alarm, modeFlag);

    nn::pctl::StopPlayTimer();
}

// OverlayNotification が来るかどうかのテスト(時間切れ、強制中断モード)
TEST_F(PlayTimerClientTest, OverlayNotification3)
{
    ResetPlayTimerAndSetNetworkTime();
    ResetForWaitingOverlayReceiveRemainingTime();

    auto pSystemEvent = nn::pctl::GetPlayTimerEventToRequestSuspension();
    pSystemEvent->Clear();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Suspend);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 時刻を約60分進める
    AddNetworkTimeForDebug(60 * 60 - WaitTimeSecondsBeforeNotification);
    // 通知を正しく発行できるようにするために有効・無効を切り替える
    // (内部で強制中断用タイマーイベントも更新される)
    nn::pctl::StopPlayTimer();
    nn::pctl::StartPlayTimer();

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(WaitTimeSecondsBeforeNotification * 1000, IgnorableTimeError, milliSec);

    // 一定時間後に通知が来るはずなのでその時間 + 2秒待つ
    int32_t time = 2;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag = nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Alarm;
    EXPECT_TRUE(WaitForOverlayReceiveRemainingTime(&time, &modeFlag, nn::TimeSpan::FromSeconds(WaitTimeSecondsBeforeNotification + 2)));
    EXPECT_EQ(0, time);
    EXPECT_EQ(nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Suspend, modeFlag);

    // 13秒待ってイベントがトリガーされるか確認
    // (10秒でトリガーされるはず)
    EXPECT_TRUE(pSystemEvent->TimedWait(nn::TimeSpan::FromSeconds(13)));

    nn::pctl::StopPlayTimer();
}

// 強制中断モードでの時間切れ時、その通知から10秒経たずに微妙に設定が変わっても中断されるのであればちゃんと10秒待つテスト
TEST_F(PlayTimerClientTest, SuspendNotificationRemain)
{
    ResetPlayTimerAndSetNetworkTime();
    ResetForWaitingOverlayReceiveRemainingTime();

    auto pSystemEvent = nn::pctl::GetPlayTimerEventToRequestSuspension();
    pSystemEvent->Clear();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Suspend);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 時刻を約60分進める
    AddNetworkTimeForDebug(60 * 60 - WaitTimeSecondsBeforeNotification);
    // 通知を正しく発行できるようにするために有効・無効を切り替える
    // (内部で強制中断用タイマーイベントも更新される)
    nn::pctl::StopPlayTimer();
    nn::pctl::StartPlayTimer();

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(WaitTimeSecondsBeforeNotification * 1000, IgnorableTimeError, milliSec);

    // 一定時間後に通知が来るはずなのでその時間 + 2秒待つ
    int32_t time = 0;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag = nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Alarm;
    EXPECT_TRUE(WaitForOverlayReceiveRemainingTime(&time, &modeFlag, nn::TimeSpan::FromSeconds(WaitTimeSecondsBeforeNotification + 2)));
    EXPECT_EQ(0, time);
    EXPECT_EQ(nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Suspend, modeFlag);

    // 6秒待つ
    // (この時点ではトリガーされないはず)
    EXPECT_FALSE(pSystemEvent->TimedWait(nn::TimeSpan::FromSeconds(6)));
    EXPECT_TRUE(nn::pctl::IsRestrictedByPlayTimer()); // トリガー待ち中も制限されている扱いとするはず

    // 微妙に設定を変える
    set.dailySettings.limitTime = 50;
    nn::pctl::SetPlayTimerSettingsForDebug(set);
    EXPECT_TRUE(nn::pctl::IsRestrictedByPlayTimer()); // 残り時間が減る方向なので制限されている扱いは変わらないはず

    // 7秒待ってイベントがトリガーされるか確認
    // (合計10秒、すなわち4秒後にトリガーされるはず)
    EXPECT_TRUE(pSystemEvent->TimedWait(nn::TimeSpan::FromSeconds(7)));
    EXPECT_TRUE(nn::pctl::IsRestrictedByPlayTimer());

    nn::pctl::StopPlayTimer();
}

// 強制中断モードでの時間切れ時、その通知から10秒経たずに微妙に設定が変わって延長されたら通知されないテスト
TEST_F(PlayTimerClientTest, SuspendNotificationButExtended)
{
    ResetPlayTimerAndSetNetworkTime();
    ResetForWaitingOverlayReceiveRemainingTime();

    auto pSystemEvent = nn::pctl::GetPlayTimerEventToRequestSuspension();
    pSystemEvent->Clear();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Suspend);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    // 時刻を約60分進める
    AddNetworkTimeForDebug(60 * 60 - WaitTimeSecondsBeforeNotification);
    // 通知を正しく発行できるようにするために有効・無効を切り替える
    // (内部で強制中断用タイマーイベントも更新される)
    nn::pctl::StopPlayTimer();
    nn::pctl::StartPlayTimer();

    nn::TimeSpan tm = nn::pctl::GetPlayTimerRemainingTime();
    int64_t milliSec = tm.GetMilliSeconds();
    NNT_PCTL_EXPECT_ALMOST_EQUAL(WaitTimeSecondsBeforeNotification * 1000, IgnorableTimeError, milliSec);

    // 一定時間後に通知が来るはずなのでその時間 + 2秒待つ
    int32_t time = 0;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag = nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Alarm;
    EXPECT_TRUE(WaitForOverlayReceiveRemainingTime(&time, &modeFlag, nn::TimeSpan::FromSeconds(WaitTimeSecondsBeforeNotification + 2)));
    EXPECT_EQ(0, time);
    EXPECT_EQ(nn::ovln::format::PctlPlayTimerModeFlag::PctlPlayTimerModeFlag_Suspend, modeFlag);

    // 6秒待つ
    // (この時点ではトリガーされないはず)
    EXPECT_FALSE(pSystemEvent->TimedWait(nn::TimeSpan::FromSeconds(6)));
    EXPECT_TRUE(nn::pctl::IsRestrictedByPlayTimer()); // トリガー待ち中も制限されている扱いとするはず

    // 微妙に設定を変える
    set.dailySettings.limitTime = 70;
    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(60));
    EXPECT_FALSE(nn::pctl::IsRestrictedByPlayTimer()); // 残り時間が増える方向なので制限されている扱いが変わるはず

    // 7秒待ってイベントがトリガーされないか確認
    // (誤ってトリガーされるとしたら合計10秒、すなわち4秒後にトリガーされるはず)
    EXPECT_FALSE(pSystemEvent->TimedWait(nn::TimeSpan::FromSeconds(7)));
    EXPECT_FALSE(nn::pctl::IsRestrictedByPlayTimer());

    nn::pctl::StopPlayTimer();
}

// 即時強制中断の対象かどうかのテスト
TEST_F(PlayTimerClientTest, SuspendImmediate)
{
    ResetPlayTimerAndSetNetworkTime(23, 0);
    ResetForWaitingOverlayReceiveRemainingTime();

    auto pSystemEvent = nn::pctl::GetPlayTimerEventToRequestSuspension();
    pSystemEvent->Clear();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = 22;
    set.dailySettings.bedtimeMinute = 0;

    nn::pctl::SetPlayTimerSettingsForDebug(set);
    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Suspend);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    EXPECT_TRUE(nn::pctl::IsRestrictedByPlayTimer());
    // イベントは発行されるが、オーバーレイ通知は無い
    EXPECT_TRUE(pSystemEvent->TimedWait(nn::TimeSpan::FromSeconds(2)));
    EXPECT_FALSE(WaitForOverlayReceiveRemainingTime(nullptr, nullptr, nn::TimeSpan::FromSeconds(WaitTimeSecondsBeforeNotification + 1)));

    nn::pctl::StopPlayTimer();
}
#endif

// 消費時間(＝プレイタイマー有効時の連続稼働時間)テスト
TEST_F(PlayTimerClientTest, CalculateSpentTime)
{
    ResetPlayTimerAndSetNetworkTime(17, 0);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = 23;
    set.dailySettings.bedtimeMinute = 0;

    nn::pctl::SetPlayTimerSettingsForDebug(set);

    nn::pctl::StartPlayTimer();

    TestTimerSettings(true, nn::pctl::PlayTimerMode::PlayTimerMode_Alarm);
    EXPECT_TRUE(nn::pctl::IsPlayTimerEnabled());

    //// 内部の時刻状態を更新させるための仮呼び出し
    //nn::pctl::StopPlayTimer();
    //nn::pctl::StartPlayTimer();

    NNT_PCTL_EXPECT_ALMOST_EQUAL(0, IgnorableTimeError, nn::pctl::GetPlayTimerSpentTimeForTest().GetMilliSeconds());

    AddNetworkTimeForDebug(10 * 60);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(10 * 60 * 1000, IgnorableTimeError, nn::pctl::GetPlayTimerSpentTimeForTest().GetMilliSeconds());

    AddNetworkTimeForDebug(20 * 60);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(30 * 60 * 1000, IgnorableTimeError, nn::pctl::GetPlayTimerSpentTimeForTest().GetMilliSeconds());

    nn::pctl::StopPlayTimer();

    AddNetworkTimeForDebug(30 * 60);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(30 * 60 * 1000, IgnorableTimeError, nn::pctl::GetPlayTimerSpentTimeForTest().GetMilliSeconds());

    nn::pctl::StartPlayTimer();

    AddNetworkTimeForDebug(60 * 60);
    NNT_PCTL_EXPECT_ALMOST_EQUAL(90 * 60 * 1000, IgnorableTimeError, nn::pctl::GetPlayTimerSpentTimeForTest().GetMilliSeconds());

    nn::pctl::StopPlayTimer();
}
