﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nsd/nsd_ApiForTest.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_IpcServer.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/overlay/pctl_OverlaySender.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_PosixTime.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_ApiForMenu.h>

namespace
{
    static nn::time::LocationName g_PreviousDeviceLocationName;
    static nn::time::LocationName g_UTCLocationName;
    static bool g_IsUTCLocationNameAvailable = false;
    static nn::time::LocationName g_AnotherLocationName; // 適当な地域名設定(何でもよい)
    static bool g_IsAnotherLocationNameAvailable = false;

    static void SetNetworkTimeFromCalendarForDebug(const nn::time::CalendarTime& calTime) NN_NOEXCEPT
    {
        nn::time::PosixTime posixTime;
        int c = 0;

        nn::time::ToPosixTime(&c, &posixTime, 1, calTime);
        // 1未満になるのは想定外なのでエラー扱いにする
        EXPECT_GE(c, 1);
        nn::pctl::detail::service::common::SetNetworkTimeForDebug(true, posixTime);
    }

    static nn::time::PosixTime g_DebugNetworkTime = {};

    static void SetInitialNetworkTimeForDebug(uint8_t hour, uint8_t minute) NN_NOEXCEPT
    {
        nn::time::CalendarTime calTime;
        calTime.year = 2016;
        calTime.month = 5;
        calTime.day = 2;
        calTime.hour = hour;
        calTime.minute = minute;
        calTime.second = 0;
        SetNetworkTimeFromCalendarForDebug(calTime);

        nn::time::PosixTime posixTime;
        int c = 0;

        nn::time::ToPosixTime(&c, &posixTime, 1, calTime);
        // 1未満になるのは想定外なのでエラー扱いにする
        EXPECT_GE(c, 1);
        g_DebugNetworkTime = posixTime;
        nn::pctl::detail::service::common::SetNetworkTimeForDebug(true, posixTime);
    }

    static void SetInitialNetworkTimeForDebug() NN_NOEXCEPT
    {
        SetInitialNetworkTimeForDebug(20, 25);
    }

    static void AddNetworkTimeForDebug(int64_t offsetSeconds) NN_NOEXCEPT
    {
        g_DebugNetworkTime.value += offsetSeconds;
        nn::pctl::detail::service::common::SetNetworkTimeForDebug(true, g_DebugNetworkTime);
    }

    static void GetCalendarTimeForDebug(nn::time::CalendarTime& calTime) NN_NOEXCEPT
    {
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, g_DebugNetworkTime);
        NN_UNUSED(calInfo);
    }

    static void DisableUserNetworkTimeForDebug() NN_NOEXCEPT
    {
        nn::pctl::detail::service::common::SetNetworkTimeForDebug(false, nn::time::PosixTime());
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace nn { namespace pctl { namespace detail { namespace service { namespace overlay {

// pctl_OverlaySender-os.win32.cpp と同じ定義を行う
struct NotifiedDataSettingChange
{
    nn::os::EventType event;
    nn::ovln::format::PctlSettingTypeFlagSet settingType;
    bool isNotified;
};
struct NotifiedDataRemainingTime
{
    nn::os::EventType event;
    int32_t time;
    nn::ovln::format::PctlPlayTimerModeFlag modeFlag;
    bool isNotified;
};

NotifiedDataSettingChange* GetNotifiedDataSettingChange() NN_NOEXCEPT;
NotifiedDataRemainingTime* GetNotifiedDataRemainingTime() NN_NOEXCEPT;

}}}}}
#endif

using namespace nn::pctl::detail::service;

class PlayTimerTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::nsd::SetTd1EnvironmentForTest();
#endif

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());

        nn::pctl::detail::service::InitializeAllocatorForServer();
        nn::pctl::detail::service::InitializeMain();
        nn::pctl::detail::service::InitializeWatcher();

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pMain->GetSettingsManager().SetPinCode("0000", 5));

        nn::time::GetDeviceLocationName(&g_PreviousDeviceLocationName);

        // UTCタイムゾーンへの変更処理
        nn::time::LocationName* pLocationNames;
        int totalCount = nn::time::GetTotalLocationNameCount();
        if (totalCount == 0)
        {
            NN_LOG("Warning: No location name for time-zone is available. Test may not work.\n");
        }
        else
        {
            pLocationNames = static_cast<nn::time::LocationName*>(malloc(sizeof(nn::time::LocationName) * totalCount));
            NN_ABORT_UNLESS_NOT_NULL(pLocationNames);
            nn::time::LoadLocationNameList(&totalCount, pLocationNames, totalCount, 0);
            if (totalCount == 0)
            {
                NN_LOG("Warning: No location name for time-zone is available. Test may not work.\n");
            }
            else
            {
                bool found = false;
                for (int i = 0; i < totalCount; ++i)
                {
                    if (nn::util::Strnicmp(pLocationNames[i]._value, "UTC", nn::time::LocationName::Size) == 0)
                    {
                        // UTCに書き換える
                        found = true;
                        nn::time::SetDeviceLocationName(pLocationNames[i]);
                        std::memcpy(&g_UTCLocationName, &pLocationNames[i], sizeof(nn::time::LocationName));
                        g_IsUTCLocationNameAvailable = true;
                        break;
                    }
                    else if (!g_IsAnotherLocationNameAvailable)
                    {
                        // UTC以外なら何でもよいので地域名設定を得る
                        std::memcpy(&g_AnotherLocationName, &pLocationNames[i], sizeof(nn::time::LocationName));
                        g_IsAnotherLocationNameAvailable = true;
                    }
                }
                if (!found)
                {
                    // UTCが見つからないので変更しない
                    NN_LOG("Warning: UTC location name is not found. Test may not work properly.\n");
                }
            }
            free(pLocationNames);
        }
    }

    static void TearDownTestCase()
    {
        g_pMain->GetSettingsManager().DeleteSettings();

        nn::pctl::detail::service::FinalizeWatcher();
        nn::pctl::detail::service::FinalizeMain();

        // 地域名設定を元に戻す
        nn::time::SetDeviceLocationName(g_PreviousDeviceLocationName);

        nn::time::Finalize();
    }
};

TEST_F(PlayTimerTest, SettingDisabled)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = false;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = false;
    manager.UpdatePlayTimerSettings(set);

    watcher::PlayTimerStatus status;
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(!status.settings.isEnabled);
}

TEST_F(PlayTimerTest, TimerDisabledOnInitial)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = false;
    manager.UpdatePlayTimerSettings(set);

    watcher::PlayTimerStatus status;
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(!status.settings.isEnabled); // 曜日別設定が何もなければ変わらない
    EXPECT_FALSE(manager.IsPlayTimerEnabled());
}

TEST_F(PlayTimerTest, AlarmTimer_Everyday)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = false; // true に書き換わるはず
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    manager.UpdatePlayTimerSettings(set);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 前後1秒の誤差は許容する
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_LE((60 * 60 - 1) * 1000, milliSec);
    EXPECT_LE(milliSec, (60 * 60 + 1) * 1000);
}

TEST_F(PlayTimerTest, AlarmTimer_Someday)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = true;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    // 今の曜日を用いる
    nn::pctl::Week week;
    {
        nn::time::PosixTime time;
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        if (!common::GetNetworkTime(&time))
        {
            time.value = 0;
        }
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        week = static_cast<nn::pctl::Week>(calInfo.dayOfWeek);
        NN_UNUSED(calTime);
    }
    set.weekSettings[week].isBedtimeEnabled = false;
    set.weekSettings[week].isLimitTimeEnabled = true;
    set.weekSettings[week].limitTime = 80;

    manager.UpdatePlayTimerSettings(set);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 前後1秒の誤差は許容する
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_LE((80 * 60 - 1) * 1000, milliSec);
    EXPECT_LE(milliSec, (80 * 60 + 1) * 1000);
}

TEST_F(PlayTimerTest, AlarmTimer_Bedtime)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    static const uint8_t SetData_Hour = 22;
    static const uint8_t SetData_Minute = 30;
    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    nn::time::CalendarTime calTime;
    {
        nn::time::PosixTime time;
        nn::time::CalendarAdditionalInfo calInfo;
        if (!common::GetNetworkTime(&time))
        {
            time.value = 0;
        }
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        NN_UNUSED(calInfo);
    }

    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = SetData_Hour;
    set.dailySettings.bedtimeMinute = SetData_Minute;

    manager.UpdatePlayTimerSettings(set);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);

    if (calTime.hour < 6)
    {
        int64_t exceedTime = (calTime.hour + 24 - SetData_Hour) * 60 * 60 +
            (calTime.minute - SetData_Minute) * 60 + calTime.second;
        // bedtime 範囲内なので負数になる
        int64_t milliSec = tm.GetMilliSeconds();
        EXPECT_LE((-exceedTime - 1) * 1000, milliSec);
        EXPECT_LE(milliSec, (-exceedTime + 1) * 1000);
    }
    else if (calTime.hour > SetData_Hour || (calTime.hour == SetData_Hour && calTime.minute >= SetData_Minute))
    {
        int64_t exceedTime = (calTime.hour - SetData_Hour) * 60 * 60 +
            (calTime.minute - SetData_Minute) * 60 + calTime.second;
        // bedtime 範囲内なので負数になる
        int64_t milliSec = tm.GetMilliSeconds();
        EXPECT_LE((-exceedTime - 1) * 1000, milliSec);
        EXPECT_LE(milliSec, (-exceedTime + 1) * 1000);
    }
    else
    {
        int64_t remainTime = (SetData_Hour - calTime.hour) * 60 * 60 +
            (SetData_Minute - calTime.minute) * 60 - calTime.second;
        // bedtime 範囲外なので正数になる
        int64_t milliSec = tm.GetMilliSeconds();
        EXPECT_LE((remainTime - 1) * 1000, milliSec);
        EXPECT_LE(milliSec, (remainTime + 1) * 1000);
    }
}

// 日付を跨いだ場合06:00(午前6時)までは超過時間とみなされるかどうか
TEST_F(PlayTimerTest, AlarmTimer_ExceedingOnDayChanged)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    static const uint8_t StartTimeHour = 21;
    static const uint8_t StartTimeMinute = 0;
    static const uint8_t BedtimeHour = 23;
    static const uint8_t BedtimeMinute = 0;

    SetInitialNetworkTimeForDebug(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = BedtimeHour;
    set.dailySettings.bedtimeMinute = BedtimeMinute;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 90分(1時間半)進める(この時点で22:30)
    AddNetworkTimeForDebug(90 * 60);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 90分だけ減っているはず(仮想時計なので正確な時刻のはず)
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 90) * 60) * 1000,
        milliSec
        );

    // 60分進める(この時点で23:30)
    AddNetworkTimeForDebug(60 * 60);

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 累計150分だけ減って超過時間になるはず
    milliSec = tm.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150 < 0);
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150) * 60) * 1000,
        milliSec
        );

    // 90分進める(この時点で01:00)
    AddNetworkTimeForDebug(90 * 60);

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 累計240分だけ減って超過時間になるはず
    milliSec = tm.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240 < 0);
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240) * 60) * 1000,
        milliSec
        );

    // さらに320分進める(この時点で06:20)
    AddNetworkTimeForDebug(320 * 60);
    {
        nn::time::CalendarTime calTime;
        GetCalendarTimeForDebug(calTime);
        EXPECT_EQ(6, calTime.hour);
        EXPECT_EQ(20, calTime.minute);
    }

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 境界線を越えて再び残り時間の扱いになるはず
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(
        ((((BedtimeHour - 6) * 60) + (BedtimeMinute - 20)) * 60) * 1000,
        milliSec
        );

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 日付を跨いだ次の日にBedtime制限がない場合でも、場合06:00(午前6時)までは超過時間とみなされるかどうか
TEST_F(PlayTimerTest, AlarmTimer_ExceedingOnDayChangedToNoRestrictionDay)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    static const uint8_t StartTimeHour = 21;
    static const uint8_t StartTimeMinute = 0;
    static const uint8_t BedtimeHour = 23;
    static const uint8_t BedtimeMinute = 0;

    SetInitialNetworkTimeForDebug(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = true;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    // 曜日を得て現在の曜日以外はクリアする
    nn::pctl::Week week;
    {
        nn::time::PosixTime time = g_DebugNetworkTime;
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        week = static_cast<nn::pctl::Week>(calInfo.dayOfWeek);
        NN_UNUSED(calTime);
    }
    memset(set.weekSettings, 0, sizeof(set.weekSettings));
    set.weekSettings[week].isBedtimeEnabled = true;
    set.weekSettings[week].isLimitTimeEnabled = false;
    set.weekSettings[week].bedtimeHour = BedtimeHour;
    set.weekSettings[week].bedtimeMinute = BedtimeMinute;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 90分(1時間半)進める(この時点で22:30)
    AddNetworkTimeForDebug(90 * 60);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 90分だけ減っているはず(仮想時計なので正確な時刻のはず)
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 90) * 60) * 1000,
        milliSec
        );

    // 60分進める(この時点で23:30)
    AddNetworkTimeForDebug(60 * 60);

    // 日付跨ぎなどの処理をするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 累計150分だけ減って超過時間になるはず
    milliSec = tm.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150 < 0);
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 150) * 60) * 1000,
        milliSec
        );

    // 90分進める(この時点で01:00)
    AddNetworkTimeForDebug(90 * 60);

    // 日付跨ぎなどの処理をするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 累計240分だけ減って超過時間になるはず(翌日は制限がないが引き継がれるはず)
    milliSec = tm.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240 < 0);
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240) * 60) * 1000,
        milliSec
        );

    // さらに320分進める(この時点で06:20)
    AddNetworkTimeForDebug(320 * 60);
    {
        nn::time::CalendarTime calTime;
        GetCalendarTimeForDebug(calTime);
        EXPECT_EQ(6, calTime.hour);
        EXPECT_EQ(20, calTime.minute);
    }


    // 日付跨ぎなどの処理をするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 境界線を越えて制限のない状態になるはずなので 0 になるはず
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(0, milliSec);
    // さらに無効状態になるはず
    manager.GetPlayTimerSettings(&set);
    EXPECT_EQ(false, set.isEnabled);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 日付を跨いだ次の日にBedtime制限がない場合でも、場合06:00(午前6時)までは超過時間とみなされるかどうか
TEST_F(PlayTimerTest, AlarmTimer_NotRestrictedOnDayChangedToRestrictedDay)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    static const uint8_t StartTimeHour = 21;
    static const uint8_t StartTimeMinute = 0;
    static const uint8_t BedtimeHour = 23;
    static const uint8_t BedtimeMinute = 0;

    SetInitialNetworkTimeForDebug(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = true;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;

    // 曜日を得て現在の曜日の次以外はクリアする
    nn::pctl::Week week;
    {
        nn::time::PosixTime time = g_DebugNetworkTime;
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, time);
        week = static_cast<nn::pctl::Week>(static_cast<int>(calInfo.dayOfWeek) + 1);
        if (week == nn::pctl::Week::Week_TotalCount)
        {
            week = nn::pctl::Week::Week_Sunday;
        }
        NN_UNUSED(calTime);
    }
    memset(set.weekSettings, 0, sizeof(set.weekSettings));
    set.weekSettings[week].isBedtimeEnabled = true;
    set.weekSettings[week].isLimitTimeEnabled = false;
    set.weekSettings[week].bedtimeHour = BedtimeHour;
    set.weekSettings[week].bedtimeMinute = BedtimeMinute;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_FALSE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 90分(1時間半)進める(この時点で22:30)
    AddNetworkTimeForDebug(90 * 60);

    // 日付跨ぎなどの処理をするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // まだ制限がないので 0 のはず
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(0, milliSec);
    manager.GetPlayTimerSettings(&set);
    EXPECT_EQ(false, set.isEnabled);

    // 150分進める(この時点で01:00)
    AddNetworkTimeForDebug(150 * 60);

    // 日付跨ぎなどの処理をするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 日付を跨いで制限のある日になるが、制限なしが引き継がれるはず
    milliSec = tm.GetMilliSeconds();
    NN_STATIC_ASSERT(((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 240 < 0);
    EXPECT_EQ(0, milliSec);
    manager.GetPlayTimerSettings(&set);
    EXPECT_EQ(false, set.isEnabled);

    // さらに320分進める(この時点で06:20)
    AddNetworkTimeForDebug(320 * 60);
    {
        nn::time::CalendarTime calTime;
        GetCalendarTimeForDebug(calTime);
        EXPECT_EQ(6, calTime.hour);
        EXPECT_EQ(20, calTime.minute);
    }

    // 日付跨ぎなどの処理をするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 境界線を越えて制限のある状態になるはず
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(
        ((((BedtimeHour - 6) * 60) + (BedtimeMinute - 20)) * 60) * 1000,
        milliSec
        );
    // さらに有効状態になるはず
    manager.GetPlayTimerSettings(&set);
    EXPECT_EQ(true, set.isEnabled);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

TEST_F(PlayTimerTest, RemainingTime_Normal)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 2秒進める
    AddNetworkTimeForDebug(2);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 2秒だけ減っているはず(仮想時計なので正確な時刻のはず)
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((120 * 60 - 2) * 1000, milliSec);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 無効の間は残り時間を消費しないテスト1
TEST_F(PlayTimerTest, RemainingTime_NoSpendingDuringDisabled1)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 無効化して2秒進める
    manager.SetPlayTimerEnabled(false);
    AddNetworkTimeForDebug(2);

    // 有効化して時間確認
    manager.SetPlayTimerEnabled(true);
    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 減っていないはず(仮想時計なので正確な時刻のはず)
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_LE((120 * 60) * 1000, milliSec);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 無効の間は残り時間を消費しないテスト2
TEST_F(PlayTimerTest, RemainingTime_NoSpendingDuringDisabled2)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 無効化して1秒待機
    manager.SetPlayTimerEnabled(false);
    AddNetworkTimeForDebug(1);

    // 有効化して1秒待機
    manager.SetPlayTimerEnabled(true);
    AddNetworkTimeForDebug(1);

    // 無効化して1秒待機
    manager.SetPlayTimerEnabled(false);
    AddNetworkTimeForDebug(1);

    // 有効化して時間確認
    manager.SetPlayTimerEnabled(true);
    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 1秒だけ減っているはず
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((120 * 60 - 1) * 1000, milliSec);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// ロック状態(ペアコンの解除で発生)の間は残り時間を消費しないテスト1
TEST_F(PlayTimerTest, RemainingTime_NoSpendingDuringLocked1)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 一時解除して2秒進める
    g_pMain->GetStatesRef().temporaryUnlocked = true;
    manager.UpdateUnlockRestrictionStatus();
    AddNetworkTimeForDebug(2);

    // 一時解除を元に戻して時間確認
    g_pMain->GetStatesRef().temporaryUnlocked = false;
    manager.UpdateUnlockRestrictionStatus();
    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 減っていないはず(仮想時計なので正確な時刻のはず)
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_LE((120 * 60) * 1000, milliSec);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// ロック状態(ペアコンの解除で発生)の間は残り時間を消費しないテスト2
TEST_F(PlayTimerTest, RemainingTime_NoSpendingDuringLocked2)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug();

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 無効化して1秒待機
    manager.SetPlayTimerEnabled(false);
    AddNetworkTimeForDebug(1); // カウントされない((disabled, temporaryUnlocked) == (true, false))

    // 一時解除して1秒待機
    g_pMain->GetStatesRef().temporaryUnlocked = true;
    manager.UpdateUnlockRestrictionStatus();
    AddNetworkTimeForDebug(1); // カウントされない((disabled, temporaryUnlocked) == (true, true))

    // 有効化して1秒待機
    manager.SetPlayTimerEnabled(true);
    AddNetworkTimeForDebug(1); // カウントされない((disabled, temporaryUnlocked) == (false, true))

    // 一時解除を元に戻して1秒待機
    g_pMain->GetStatesRef().temporaryUnlocked = false;
    manager.UpdateUnlockRestrictionStatus();
    AddNetworkTimeForDebug(1); // カウントされる((disabled, temporaryUnlocked) == (false, false))

    // 有効化して時間確認
    manager.SetPlayTimerEnabled(true);
    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 1秒だけ減っているはず
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((120 * 60 - 1) * 1000, milliSec);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 日付を跨いだら残り時間が変わるテスト
TEST_F(PlayTimerTest, RemainingTime_ResetOnDayChanged)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug(21, 30);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 90分進める(この時点で23:00)
    AddNetworkTimeForDebug(90 * 60);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 90分だけ減って残り30分のはず
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((30 * 60) * 1000, milliSec);

    // 40分進める(この時点で23:40)
    AddNetworkTimeForDebug(40 * 60);

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 合計130分だけ減って10分超過のはず
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((-10 * 60) * 1000, milliSec);

    // 50分進める(この時点で00:30)
    AddNetworkTimeForDebug(50 * 60);

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 0時でリセットされ、30分だけ減って残り90分のはず
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((90 * 60) * 1000, milliSec);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 地域名設定が変わったら残り時間がリセットされるテスト
TEST_F(PlayTimerTest, RemainingTime_ResetOnLocationNameChanged)
{
    if (!g_IsAnotherLocationNameAvailable)
    {
        NN_LOG("Warning: This test cannot be run because any location name but UTC is not found.\n");
        return;
    }

    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug(21, 30);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 120;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 90分進める(この時点で23:00)
    AddNetworkTimeForDebug(90 * 60);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 90分だけ減って残り30分のはず
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((30 * 60) * 1000, milliSec);

    // 地域名設定を適当に変えて内部の地域名チェック間隔以上の時間を待つ
    nn::time::SetDeviceLocationName(g_AnotherLocationName);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(nn::pctl::detail::service::watcher::WatcherEventManager::TimeSpanForCheck + 2));
    manager.WaitForEventsForTest();
    manager.WaitForEventsForTest();

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // リセットされるので残り時間は120分になるはず
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((120 * 60) * 1000, milliSec);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();

    // 元に戻しておく
    nn::time::SetDeviceLocationName(g_UTCLocationName);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(nn::pctl::detail::service::watcher::WatcherEventManager::TimeSpanForCheck + 2));
    manager.WaitForEventsForTest();
    manager.WaitForEventsForTest();
}

// 特定時刻が期限となる形で作られる制限時間はタイマーが無効の間も減るテスト
TEST_F(PlayTimerTest, RemainingTime_SpendingWithBedtimeDuringDisabled1)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    static const uint8_t StartTimeHour = 19;
    static const uint8_t StartTimeMinute = 15;
    static const uint8_t BedtimeHour = 21;
    static const uint8_t BedtimeMinute = 0;

    SetInitialNetworkTimeForDebug(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = BedtimeHour;
    set.dailySettings.bedtimeMinute = BedtimeMinute;

    manager.UpdatePlayTimerSettings(set);
    // 無効にしておく
    manager.SetPlayTimerEnabled(false);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_FALSE(manager.IsPlayTimerEnabled());

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 念のため現在の残り時間チェック
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute)) * 60) * 1000,
        milliSec
        );

    // 45分進める(無効のまま)
    AddNetworkTimeForDebug(45 * 60);

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 45分だけ減っているはず(仮想時計なので正確な時刻のはず)
    milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 45) * 60) * 1000,
        milliSec
        );

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// タイマーが無効の間の bedtime と limitTime から得られる制限時間が変わるかのテスト
TEST_F(PlayTimerTest, RemainingTime_SpendingWithBedtimeDuringDisabled2)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    static const uint8_t StartTimeHour = 19;
    static const uint8_t StartTimeMinute = 15;
    static const uint8_t BedtimeHour = 21;
    static const uint8_t BedtimeMinute = 0;
    static const uint16_t LimitTimeMinute = 60;

    SetInitialNetworkTimeForDebug(StartTimeHour, StartTimeMinute);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.bedtimeHour = BedtimeHour;
    set.dailySettings.bedtimeMinute = BedtimeMinute;
    set.dailySettings.limitTime = LimitTimeMinute;

    manager.UpdatePlayTimerSettings(set);
    // 無効にしておく
    manager.SetPlayTimerEnabled(false);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_FALSE(manager.IsPlayTimerEnabled());

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    // 念のため現在の残り時間チェック
    int64_t milliSec = tm.GetMilliSeconds();
    // 最初は limitTime の方が小さいのでそれが得られるはず
    // (60min. < (21:00 - 19:15))
    EXPECT_EQ(
        (LimitTimeMinute * 60) * 1000,
        milliSec
        );

    // 30分進める(無効のまま)
    AddNetworkTimeForDebug(30 * 60);

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    milliSec = tm.GetMilliSeconds();
    // まだ変化がないはず
    // (60min. < (21:00 - 19:45))
    EXPECT_EQ(
        (LimitTimeMinute * 60) * 1000,
        milliSec
        );

    // 30分進める(無効のまま)
    AddNetworkTimeForDebug(30 * 60);

    tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    milliSec = tm.GetMilliSeconds();
    // 変化があるはず
    // (60min. > (21:00 - 20:15))
    NN_STATIC_ASSERT((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 60) < LimitTimeMinute);
    EXPECT_EQ(
        ((((BedtimeHour - StartTimeHour) * 60) + (BedtimeMinute - StartTimeMinute) - 60) * 60) * 1000,
        milliSec
        );

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// オーバーレイ通知テストはひとまずWin版でのみ動かすこととする
#if defined(NN_BUILD_CONFIG_OS_WIN)
// OverlayNotification が来るかどうかのテスト(残り30分)
TEST_F(PlayTimerTest, OverlayNotification1)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug();

    auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
    pNotifiedData->isNotified = false;
    nn::os::ClearEvent(&pNotifiedData->event);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 時刻を30分進める(残り30分になる)
    AddNetworkTimeForDebug(30 * 60);
    // 通知を正しく発行できるようにするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((30 * 60) * 1000, milliSec);

    EXPECT_TRUE(nn::os::TimedWaitEvent(&pNotifiedData->event, nn::TimeSpan::FromSeconds(3)));
    EXPECT_EQ(true, pNotifiedData->isNotified);
    EXPECT_EQ(30 * 60, pNotifiedData->time);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// OverlayNotification が来るかどうかのテスト(30分超過)
TEST_F(PlayTimerTest, OverlayNotification2)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    SetInitialNetworkTimeForDebug();

    auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
    pNotifiedData->isNotified = false;
    nn::os::ClearEvent(&pNotifiedData->event);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 時刻を90分進める(30分超過になる)
    AddNetworkTimeForDebug(90 * 60);
    // 通知を正しく発行できるようにするためにタイマーイベントを更新する
    manager.RefreshTimeStatus();
    manager.WaitForEventsForTest();

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Alarm, mode);
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ((-30 * 60) * 1000, milliSec);

    EXPECT_TRUE(nn::os::TimedWaitEvent(&pNotifiedData->event, nn::TimeSpan::FromSeconds(3)));
    EXPECT_EQ(true, pNotifiedData->isNotified);
    EXPECT_EQ(-30 * 60, pNotifiedData->time);

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// OverlayNotification が来るかどうかのテスト(時間切れ、強制中断モード)
TEST_F(PlayTimerTest, OverlayNotification3)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    manager.GetPlayTimerEventToRequestSuspension()->Clear();

    SetInitialNetworkTimeForDebug();

    auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
    pNotifiedData->isNotified = false;
    nn::os::ClearEvent(&pNotifiedData->event);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 時刻を59分59秒進める
    AddNetworkTimeForDebug(60 * 60 - 1);
    // 通知を正しく発行できるようにするために有効・無効を切り替える
    // (内部で強制中断用タイマーイベントも更新される)
    manager.SetPlayTimerEnabled(false);
    manager.SetPlayTimerEnabled(true);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Suspend, mode);
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(1000, milliSec);

    // (1秒後に通知があるはず)
    EXPECT_TRUE(nn::os::TimedWaitEvent(&pNotifiedData->event, nn::TimeSpan::FromSeconds(4)));
    EXPECT_EQ(true, pNotifiedData->isNotified);
    EXPECT_EQ(0, pNotifiedData->time);

    // 時刻を1秒進める
    AddNetworkTimeForDebug(1);
    manager.RefreshTimeStatus();

    // 13秒待ってイベントがトリガーされるか確認
    // (10秒でトリガーされるはず)
    EXPECT_TRUE(manager.GetPlayTimerEventToRequestSuspension()->TimedWait(nn::TimeSpan::FromSeconds(13)));

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 強制中断モードでの時間切れ時、その通知から10秒経たずに微妙に設定が変わっても中断されるのであればちゃんと10秒待つテスト
TEST_F(PlayTimerTest, SuspendNotificationRemain)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    manager.GetPlayTimerEventToRequestSuspension()->Clear();

    SetInitialNetworkTimeForDebug();

    auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
    pNotifiedData->isNotified = false;
    nn::os::ClearEvent(&pNotifiedData->event);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 時刻を59分59秒進める
    AddNetworkTimeForDebug(60 * 60 - 1);
    // 通知を正しく発行できるようにするために有効・無効を切り替える
    // (内部で強制中断用タイマーイベントも更新される)
    manager.SetPlayTimerEnabled(false);
    manager.SetPlayTimerEnabled(true);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Suspend, mode);
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(1000, milliSec);

    // (1秒後に通知があるはず)
    EXPECT_TRUE(nn::os::TimedWaitEvent(&pNotifiedData->event, nn::TimeSpan::FromSeconds(4)));
    EXPECT_EQ(true, pNotifiedData->isNotified);
    EXPECT_EQ(0, pNotifiedData->time);

    // 時刻を1秒進める
    AddNetworkTimeForDebug(1);
    manager.RefreshTimeStatus();

    // 6秒待つ
    // (この時点ではトリガーされないはず)
    EXPECT_FALSE(manager.GetPlayTimerEventToRequestSuspension()->TimedWait(nn::TimeSpan::FromSeconds(6)));
    EXPECT_TRUE(manager.IsRestrictedByPlayTimer()); // トリガー待ち中も制限されている扱いとするはず

    // 微妙に設定を変える
    set.dailySettings.limitTime = 50;
    manager.UpdatePlayTimerSettings(set);
    manager.WaitForEventsForTest();
    EXPECT_TRUE(manager.IsRestrictedByPlayTimer()); // 残り時間が減る方向なので制限されている扱いは変わらないはず

    // 7秒待ってイベントがトリガーされるか確認
    // (合計10秒、すなわち4秒後にトリガーされるはず)
    EXPECT_TRUE(manager.GetPlayTimerEventToRequestSuspension()->TimedWait(nn::TimeSpan::FromSeconds(7)));
    EXPECT_TRUE(manager.IsRestrictedByPlayTimer());

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 強制中断モードでの時間切れ時、その通知から10秒経たずに微妙に設定が変わって延長されたら通知されないテスト
TEST_F(PlayTimerTest, SuspendNotificationButExtended)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    manager.GetPlayTimerEventToRequestSuspension()->Clear();

    SetInitialNetworkTimeForDebug();

    auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
    pNotifiedData->isNotified = false;
    nn::os::ClearEvent(&pNotifiedData->event);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = false;
    set.dailySettings.isLimitTimeEnabled = true;
    set.dailySettings.limitTime = 60;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    // 時刻を59分59秒進める
    AddNetworkTimeForDebug(60 * 60 - 1);
    // 通知を正しく発行できるようにするために有効・無効を切り替える
    // (内部で強制中断用タイマーイベントも更新される)
    manager.SetPlayTimerEnabled(false);
    manager.SetPlayTimerEnabled(true);

    nn::pctl::PlayTimerMode mode;
    nn::TimeSpan tm = manager.GetRemainingTime(&mode);
    EXPECT_EQ(nn::pctl::PlayTimerMode_Suspend, mode);
    int64_t milliSec = tm.GetMilliSeconds();
    EXPECT_EQ(1000, milliSec);

    // (1秒後に通知があるはず)
    EXPECT_TRUE(nn::os::TimedWaitEvent(&pNotifiedData->event, nn::TimeSpan::FromSeconds(4)));
    EXPECT_EQ(true, pNotifiedData->isNotified);
    EXPECT_EQ(0, pNotifiedData->time);

    // 時刻を1秒進める
    AddNetworkTimeForDebug(1);
    manager.RefreshTimeStatus();

    // 6秒待つ
    // (この時点ではトリガーされないはず)
    EXPECT_FALSE(manager.GetPlayTimerEventToRequestSuspension()->TimedWait(nn::TimeSpan::FromSeconds(6)));
    EXPECT_TRUE(manager.IsRestrictedByPlayTimer()); // トリガー待ち中も制限されている扱いとするはず

    // 微妙に設定を変える
    set.dailySettings.limitTime = 70;
    manager.UpdatePlayTimerSettings(set);
    manager.WaitForEventsForTest();
    EXPECT_FALSE(manager.IsRestrictedByPlayTimer()); // 残り時間が増える方向なので制限されている扱いが変わるはず

    // 7秒待ってイベントがトリガーされないか確認
    // (誤ってトリガーされるとしたら合計10秒、すなわち4秒後にトリガーされるはず)
    EXPECT_FALSE(manager.GetPlayTimerEventToRequestSuspension()->TimedWait(nn::TimeSpan::FromSeconds(7)));
    EXPECT_FALSE(manager.IsRestrictedByPlayTimer());

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}

// 即時強制中断の対象かどうかのテスト
TEST_F(PlayTimerTest, SuspendImmediate)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    manager.GetPlayTimerEventToRequestSuspension()->Clear();

    SetInitialNetworkTimeForDebug(23, 0);

    auto pNotifiedData = nn::pctl::detail::service::overlay::GetNotifiedDataRemainingTime();
    pNotifiedData->isNotified = false;
    nn::os::ClearEvent(&pNotifiedData->event);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = 22;
    set.dailySettings.bedtimeMinute = 0;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    EXPECT_TRUE(manager.IsRestrictedByPlayTimer());
    // イベントは発行される
    EXPECT_TRUE(manager.GetPlayTimerEventToRequestSuspension()->TimedWait(nn::TimeSpan::FromSeconds(2)));

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}
#endif

// 消費時間(＝プレイタイマー有効時の連続稼働時間)テスト
TEST_F(PlayTimerTest, CalculateSpentTime)
{
    auto& manager = g_pWatcher->GetWatcherEventManager();

    manager.GetPlayTimerEventToRequestSuspension()->Clear();

    SetInitialNetworkTimeForDebug(17, 0);

    nn::pctl::PlayTimerSettings set = {};
    set.isEnabled = true;
    set.isWeekSettingsUsed = false;
    set.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
    set.dailySettings.isBedtimeEnabled = true;
    set.dailySettings.isLimitTimeEnabled = false;
    set.dailySettings.bedtimeHour = 23;
    set.dailySettings.bedtimeMinute = 0;

    manager.UpdatePlayTimerSettings(set);
    manager.SetPlayTimerEnabled(true);

    watcher::PlayTimerStatus status;
    manager.ResetTimeStatus();
    manager.GetPlayTimerStatus(&status);
    EXPECT_TRUE(status.settings.isEnabled);
    EXPECT_TRUE(manager.IsPlayTimerEnabled());

    //// 内部の時刻状態を更新させるための仮呼び出し
    //manager.SetPlayTimerEnabled(false);
    //manager.SetPlayTimerEnabled(true);

    EXPECT_EQ(0, manager.GetSpentTime().GetSeconds());

    AddNetworkTimeForDebug(10 * 60);
    EXPECT_EQ(10 * 60, manager.GetSpentTime().GetSeconds());

    AddNetworkTimeForDebug(20 * 60);
    EXPECT_EQ(30 * 60, manager.GetSpentTime().GetSeconds());

    manager.SetPlayTimerEnabled(false);

    AddNetworkTimeForDebug(30 * 60);
    EXPECT_EQ(30 * 60, manager.GetSpentTime().GetSeconds());

    manager.SetPlayTimerEnabled(true);

    AddNetworkTimeForDebug(60 * 60);
    EXPECT_EQ(90 * 60, manager.GetSpentTime().GetSeconds());

    manager.SetPlayTimerEnabled(false);
    DisableUserNetworkTimeForDebug();
}
